package net.minidev.json.parser;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import java.util.Map;

import static net.minidev.json.parser.ParseException.*;

/**
 * Parser for JSON text. Please note that JSONParser is NOT thread-safe.
 * 
 * @author Uriel Chemouni <uchemouni@gmail.com>
 */
public class JSONParser extends JSONParserBase {
	public final static byte EOI = 0x1A;
	private char c;
	private String in;
	private int len;

	/**
	 * @deprecated prefer usage of new JSONParser(JSONParser.MODE_*)
	 */
	public JSONParser() {
		super(MODE_PERMISSIVE);
	}

	public JSONParser(int permissiveMode) {
		super(permissiveMode);
	}

	/**
	 * use to return Primitive Type, or String, Or JsonObject or JsonArray
	 * generated by a ContainerFactory
	 */
	public Object parse(String in) throws ParseException, IOException {
		return parse(in, ContainerFactory.FACTORY);
	}

	/**
	 * use to return Primitive Type, or String, Or JsonObject or JsonArray
	 * generated by a ContainerFactory
	 */
	public Object parse(String in, ContainerFactory containerFactory) throws ParseException, IOException {
		this.len = in.length();
		this.in = in;
		this.containerFactory = containerFactory;
		this.pos = -1;
		read();
		handler.startJSON();
		Object result = readMain(stopX);
		handler.endJSON();
		xs = null;
		xo = null;
		return result;
	}

	final private void read() {
		if (++pos >= len)
			this.c = EOI;
		else
			this.c = in.charAt(pos);
	}

	private List<Object> readArray() throws ParseException, IOException {
		List<Object> obj = containerFactory.creatArrayContainer();
		if (c != '[')
			throw new RuntimeException("Internal Error");
		read();
		handler.startArray();
		for (;;) {
			switch (c) {
			case ' ':
			case '\r':
			case '\n':
			case '\t':
				read();
				continue;
			case ']':
				read(); /* unstack */
				handler.endArray();
				return obj;
			case ':':
			case '}':
				throw new ParseException(pos, ERROR_UNEXPECTED_CHAR, (char) c);
			case ',':
				read();
				continue;
			case EOI:
				throw new ParseException(pos - 1, ERROR_UNEXPECTED_EOF, "EOF");
			default:
				obj.add(readMain(stopArray));
				continue;
			}
		}
	}

	/**
	 * use to return Primitive Type, or String, Or JsonObject or JsonArray
	 * generated by a ContainerFactory
	 */
	private Object readMain(boolean stop[]) throws ParseException, IOException {
		for (;;) {
			switch (c) {
			// skip spaces
			case ' ':
			case '\r':
			case '\n':
			case '\t':
				read();
				continue;
				// invalid stats
			case ':':
			case '}':
			case ']':
				throw new ParseException(pos, ERROR_UNEXPECTED_CHAR, c);
				// start object
			case '{':
				return readObject();
				// start Array
			case '[':
				return readArray();
				// start string
			case '"':
			case '\'':
				xs = readString();
				handler.primitive(xs);
				return xs;
				// string or null
			case 'n':
				xs = readNQString(stop);
				if ("null".equals(xs)) {
					handler.primitive(null);
					return null;
				}
				if (!acceptNonQuote)
					throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, xs);
				handler.primitive(xs);
				return xs;
				// string or false
			case 'f':
				xs = readNQString(stop);
				if ("false".equals(xs)) {
					handler.primitive(Boolean.FALSE);
					return Boolean.FALSE;
				}
				if (!acceptNonQuote)
					throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, xs);
				handler.primitive(xs);
				return xs;
				// string or true
			case 't':
				xs = readNQString(stop);
				if ("true".equals(xs)) {
					handler.primitive(Boolean.TRUE);
					return Boolean.TRUE;
				}
				if (!acceptNonQuote)
					throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, xs);
				handler.primitive(xs);
				return xs;
				// string or NaN
			case 'N':
				xs = readNQString(stop);
				if (!acceptNaN)
					throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, xs);
				if ("NaN".equals(xs)) {
					handler.primitive(Float.NaN);
					return Float.valueOf(Float.NaN);
				}
				if (!acceptNonQuote)
					throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, xs);
				handler.primitive(xs);
				return xs;
				// digits
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
			case '-':
				xo = readNumber(stop);
				handler.primitive(xo);
				return xo;
			default:
				xs = readNQString(stop);
				if (!acceptNonQuote)
					throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, xs);
				handler.primitive(xs);
				return xs;
			}
		}
	}

	private String readNQString(boolean[] stop) {
		int start = pos;
		skipNQString(stop);
		return in.substring(start, pos).trim();
	}

	private Object readNumber(boolean[] stop) throws ParseException {
		int start = pos;
		// skip first char digit or -
		read();
		skipDigits();
		if (c != '.' && c != 'E' && c != 'e') {
			skipSpace();
			if (!stop[c]) {
				// convert string
				skipNQString(stop);
				xs = in.substring(start, pos).trim();
				if (!acceptNonQuote)
					throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, xs);
				return xs;
			}
			xs = in.substring(start, pos).trim();
			if (xs.length() > 20)
				return new BigInteger(xs);
			try {
				long v = Long.parseLong(xs);
				if (v >= (long) Integer.MIN_VALUE && v <= (long) Integer.MAX_VALUE)
					return Integer.valueOf((int) v);
				return Long.valueOf(v);
			} catch (NumberFormatException e) {
				return new BigInteger(xs);
			}
		}
		if (c == '.') {
			//
			read();
			skipDigits();
		}
		if (c != 'E' && c != 'e') {
			skipSpace();
			if (!stop[c]) {
				// convert string
				skipNQString(stop);
				xs = in.substring(start, pos).trim();
				if (!acceptNonQuote)
					throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, xs);
				return xs;
			}
			String num = in.substring(start, pos).trim();
			return Double.parseDouble(num);
		}
		sb.append('E');
		read();
		if (c == '+' || c == '-' || c >= '0' && c <= '9') {
			sb.append(c);
			read(); // skip first char
			skipDigits();
			skipSpace();
			if (!stop[c]) {
				// convert string
				skipNQString(stop);
				String text = in.substring(start, pos).trim();
				if (!acceptNonQuote)
					throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, text);
				return text;
			}
			return Double.parseDouble(in.substring(start, pos).trim());
		} else {
			skipNQString(stop);
			String text = in.substring(start, pos).trim();
			if (!acceptNonQuote)
				throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, text);
			return text;
		}
		// throw new ParseException(pos - 1, ERROR_UNEXPECTED_CHAR, null);
	}

	private Map<String, Object> readObject() throws ParseException, IOException {
		Map<String, Object> obj = this.containerFactory.createObjectContainer();
		if (c != '{')
			throw new RuntimeException("Internal Error");
		handler.startObject();
		for (;;) {
			read();
			switch (c) {
			case ' ':
			case '\r':
			case '\t':
			case '\n':
				continue;
			case ':':
			case ']':
			case '[':
			case '{':
				throw new ParseException(pos, ERROR_UNEXPECTED_CHAR, c);
			case '}':
				read(); /* unstack */
				handler.endObject();
				return obj;
			case ',':
				continue;
			case '"':
			case '\'':
			default:
				String key;
				if (c == '\"' || c == '\'')
					key = readString();
				else {
					key = readNQString(stopKey);
					if (!acceptNonQuote)
						throw new ParseException(pos, ERROR_UNEXPECTED_TOKEN, key);
				}
				handler.startObjectEntry(key);
				while (c != ':' && c != EOI) {
					read();
				}
				if (c == EOI)
					throw new ParseException(pos - 1, ERROR_UNEXPECTED_EOF, null);
				read(); /* skip : */
				obj.put(key, readMain(stopValue));
				handler.endObjectEntry();
				if (c == '}') {
					read(); /* unstack */
					handler.endObject();
					return obj;
				} /* if c==, confinue */
				continue;
			}
		}
	}

	private String readString() throws ParseException, IOException {
		if (!acceptSimpleQuote && c == '\'')
			throw new ParseException(pos, ERROR_UNEXPECTED_CHAR, c);
		int tmpP = in.indexOf(c, pos + 1);
		if (tmpP == -1)
			throw new ParseException(in.length(), ERROR_UNEXPECTED_EOF, null);
		xs = in.substring(pos + 1, tmpP);
		if (xs.indexOf('\\') == -1) {
			pos = tmpP;
			read();
			// handler.primitive(tmp);
			return xs;
		}
		sb.clear();// sb.delete(0, sb.length());
		/* assert (c == '\"' || c == '\'') */
		char sep = c;
		for (;;) {
			read();
			switch (c) {
			case EOI:
				throw new ParseException(pos - 1, ERROR_UNEXPECTED_EOF, null);
			case '"':
			case '\'':
				if (sep == c) {
					read();
					return sb.toString();
				}
				sb.append((char) c);
				break;
			case '\\':
				read();
				switch (c) {
				case 't':
					sb.append('\t');
					break;
				case 'n':
					sb.append('\n');
					break;
				case 'r':
					sb.append('\r');
					break;
				case 'f':
					sb.append('\f');
					break;
				case 'b':
					sb.append('\b');
					break;
				case '\\':
					sb.append('\\');
					break;
				case '/':
					sb.append('/');
					break;
				case '\'':
					sb.append('\'');
					break;
				case '"':
					sb.append('"');
					break;
				case 'u':
					sb.append(readUnicode());
					break;
				default:
					break;
				}
				break;
			case '\b':
			case '\t':
			case '\f':
			case '\r':
			case '\n':
				continue;
			default:
				sb.append((char) c);
			}
		}
	}

	private char readUnicode() throws ParseException {
		int value = 0;
		for (int i = 0; i < 4; i++) {
			value = value * 16;
			read();
			if (c >= '0' && c <= '9')
				value += c - '0';
			else if (c >= 'A' && c <= 'F')
				value += (c - 'A') + 10;
			else if (c >= 'a' && c <= 'f')
				value += (c - 'a') + 10;
			else if (c == EOI)
				throw new ParseException(pos, ERROR_UNEXPECTED_EOF, "EOF");
			else
				throw new ParseException(pos, ERROR_UNEXPECTED_UNICODE, c);
		}
		return (char) value;
	}

	private void skipDigits() {
		for (;;) {
			if (c == EOI)
				return;
			if (c < '0' || c > '9')
				return;
			//
			read();
		}
	}

	private void skipNQString(boolean[] stop) {
		for (;;) {
			if (c == EOI)
				return;
			if (c >= 0 && c <= 125 && stop[c])
				return;
			//
			read();
		}
	}

	private void skipSpace() throws ParseException {
		for (;;) {
			if (c == EOI)
				return;
			if (c != ' ' && c != '\r' && c != '\t' && c != '\n')
				return;
			//
			read();
		}
	}
}
