package net.ninjacat.smooth.collections;

import net.ninjacat.smooth.functions.Func;
import net.ninjacat.smooth.utils.Pair;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public final class Maps {

    private Maps() {
    }

    /**
     * <p>
     * Will create map, give list of key, value pairs.
     * </p>
     * <p>
     * Example: {@code Map<Integer, String> map = Maps.of(1, "One", 2, "Two", 3, "Three");}
     * </p>
     * <p>
     * <strong>Number of parameters must be even.</strong>
     * <strong>Will throw ClassCastException if wrong argument types are supplied</strong>
     * </p>
     *
     * @param pairs list of key-value pairs. There is no special separation of each pair.
     * @param <K>   Type of keys
     * @param <V>   Type of values
     * @return Map&lt;K,V&gt; of supplied key-value pairs
     */
    @SuppressWarnings("unchecked")
    public static <K, V> Map<K, V> of(final Object... pairs) {
        if (0 != pairs.length % 2) {
            throw new IllegalArgumentException("Initializer should have even number of elements");
        }
        final Map<K, V> map = new HashMap<K, V>();

        for (int i = 0; i < pairs.length; i += 2) {
            map.put((K) pairs[i], (V) pairs[i + 1]);
        }

        return map;
    }

    /**
     * <p>
     * Will create map, give list of key-value pairs.
     * </p>
     * <p>
     * Pairs are represented by {@link Pair} objects where left is used as key and right is used for value.
     * </p>
     *
     * @param pairs list of key-value pairs.
     * @param <K>   Type of keys
     * @param <V>   Type of values
     * @return Map&lt;K,V&gt; of supplied key-value pairs
     */
    public static <K, V> Map<K, V> of(final Pair<K, V>... pairs) {
        final Map<K, V> map = new HashMap<K, V>();

        for (final Pair<K, V> pair : pairs) {
            map.put(pair.getLeft(), pair.getRight());
        }

        return map;
    }

    /**
     * Converts iterable to map using provided Key generator function. Generator function is called on each
     * element of iterable and its result is used as a key for the map.
     *
     * @param list         - Iterable to convert into {@link Map}
     * @param keyGenerator Key generator function.
     * @param <K>          Type of map key.
     * @param <V>          Type of iterable values
     * @return Map with keys generated by function and values taken from iterable.
     */
    public static <K, V> Map<K, V> toMap(final Iterable<V> list, final Func<K, V> keyGenerator) {
        final Map<K, V> map = new HashMap<K, V>();
        for (final V item : list) {
            map.put(keyGenerator.apply(item), item);
        }
        return map;
    }

    /**
     * Converts iterable to map using provided Key generator function. Generator function is called on each
     * element of iterable and its result is used as a key for the map.
     * <p>
     * Generated Map will be unmodifiable.
     * </p>
     *
     * @param list         - Iterable to convert into {@link Map}
     * @param keyGenerator Key generator function.
     * @param <K>          Type of map key.
     * @param <V>          Type of iterable values
     * @return Map with keys generated by function and values taken from iterable.
     */
    public static <K, V> Map<K, V> toUnmodifiableMap(final Iterable<V> list, final Func<K, V> keyGenerator) {
        return Collections.unmodifiableMap(toMap(list, keyGenerator));
    }
}
