package net.nowtryz.cafetapi.datas;

import java.io.IOException;
import java.util.Calendar;

import net.nowtryz.cafetapi.CafetAPI;
import net.nowtryz.cafetapi.exceptions.APIException;
import net.nowtryz.cafetapi.utils.Lang;
import net.nowtryz.cafetapi.utils.WordUtils;


/**
 * The Client object is the wich stores every client information for later use,
 * it's generaly generated by a Data static method (such as
 * {@link CafetAPI#getClient(int) Data.getClient(int id)})
 * 
 * @author Damien
 * @since 1.0
 */
public class Client extends CafetData {
	private int id;
	private String email;
	private String alias;
	private String familyName;
	private String firstname;
	private boolean member;
	private float balance;
	private int registrationYear;

	/**
	 * The Client object is the one that stores every client information for
	 * later use, it's generaly generated by a Data static method (such as
	 * {@link CafetAPI#getClient(int) Data.getClient(int
	 * id)})
	 * 
	 * @param id an integer, the ID of the client in the DataBase
	 * @param email the email of the client
	 * @param alias the username of the client
	 * @param familyName the familyname of the client
	 * @param firstname the firstname of the client
	 * @param member a boolean that representswhether or not the client is a 
	 * 	          member of the association and can benefits for discounts
	 * @param balance the actual balance of the client
	 * @param registrationYear the year of the client's registration
	 * @since 1.0
	 */
	public Client(int id, String email, String alias, String familyNane, String surname, boolean member, float balance, int registrationYear) {
		this.id = id;
		this.email = email;
		this.alias = alias;
		this.familyName = familyNane;
		this.firstname = surname;
		this.member = member;
		this.balance = balance;
		this.registrationYear = registrationYear;
	}

	/**
	 * The Client object is the wich stores every client information for later
	 * use, it's generaly generated by a Data static method (such as
	 * {@link CafetAPI#getClient(int) Data.getClient(int
	 * id)})
	 * 
	 * @since 1.0
	 */
	public Client() {
		this.id = 0;
		this.email = Lang.get("Unknown");
		this.alias = Lang.get("Unknown");
		this.familyName = Lang.get("Unknown");
		this.firstname = Lang.get("Unknown");
		this.member = false;
		this.balance = 0.0f;
		this.registrationYear = Calendar.getInstance().get(Calendar.YEAR);
	}

	/**
	 * Return the id
	 * 
	 * @return the id
	 * @since 1.0
	 */
	public final int getId() {
		return id;
	}

	/**
	 * Return the email
	 * 
	 * @return the email
	 * @since 1.0
	 */
	public final String getEmail() {
		return email;
	}

	/**
	 * Return the alias
	 * 
	 * @return the alias
	 * @since 1.0
	 */
	public final String getAlias() {
		return alias;
	}

	/**
	 * Return the familynane
	 * 
	 * @return the familynane
	 * @since 1.0
	 */
	public final String getFamilyNane() {
		return familyName;
	}

	/**
	 * Return the firstname
	 * 
	 * @return the firstname
	 * @since 1.0
	 */
	public final String getSurname() {
		return firstname;
	}

	/**
	 * Return the member
	 * 
	 * @return the member
	 * @since 1.0
	 */
	public final boolean isMember() {
		return member;
	}

	/**
	 * Return the balance
	 * 
	 * @return the balance
	 * @since 1.0
	 */
	public final float getBalance() {
		return balance;
	}

	/**
	 * Return the registrationYear
	 * 
	 * @return the registrationYear
	 * @since 1.0
	 */
	public final int getRegistrationYear() {
		return registrationYear;
	}

	/**
	 * Return a list of latest client's expenses
	 * 
	 * @return a list of client's expenses
	 * @throws IOException 
	 * @throws APIException 
	 * @since 1.0
	 */
	public final Expense[] getExpenses() throws APIException, IOException {
		return CafetAPI.getInstance().getClientExpenses(id).getDatas();
	}

	/**
	 * Return a list of latest client's reloads
	 * 
	 * @return a list of client's reloads
	 * @throws IOException 
	 * @throws APIException 
	 * @since 1.0
	 */
	public final Reload[] getReloads() throws APIException, IOException {
		return CafetAPI.getInstance().getClientReloads(id).getDatas();
	}
	
	/**
	 * Returns the name like Suname NAME
	 * @return the formated name
	 * @since 1.0
	 */
	public final String getFormatedName() {
		return WordUtils.capitalizeFirstLetter(this.getSurname()) + " " + this.getFamilyNane().toUpperCase();
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return WordUtils.capitalizeFirstLetter(this.getSurname()) + " " + this.getFamilyNane().toUpperCase();
	}

	/**
	 * Update information about the client
	 * @throws IOException 
	 * @throws APIException 
	 * @since 1.0
	 */
	public void update() throws APIException, IOException {
		Client c = CafetAPI.getInstance().getClient(id).getData();
		this.id = c.id;
		this.email = c.email;
		this.alias = c.alias;
		this.familyName = c.familyName;
		this.firstname = c.firstname;
		this.member = c.member;
		this.balance = c.balance;
		this.registrationYear = c.registrationYear;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((alias == null) ? 0 : alias.hashCode());
		long temp;
		temp = Double.doubleToLongBits(balance);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		result = prime * result + ((email == null) ? 0 : email.hashCode());
		result = prime * result + ((familyName == null) ? 0 : familyName.hashCode());
		result = prime * result + id;
		result = prime * result + (member ? 1231 : 1237);
		result = prime * result + registrationYear;
		result = prime * result + ((firstname == null) ? 0 : firstname.hashCode());
		return result;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof Client))
			return false;
		Client other = (Client) obj;
		if (alias == null) {
			if (other.alias != null)
				return false;
		} else if (!alias.equals(other.alias))
			return false;
		if (Double.doubleToLongBits(balance) != Double.doubleToLongBits(other.balance))
			return false;
		if (email == null) {
			if (other.email != null)
				return false;
		} else if (!email.equals(other.email))
			return false;
		if (familyName == null) {
			if (other.familyName != null)
				return false;
		} else if (!familyName.equals(other.familyName))
			return false;
		if (id != other.id)
			return false;
		if (member != other.member)
			return false;
		if (registrationYear != other.registrationYear)
			return false;
		if (firstname == null) {
			if (other.firstname != null)
				return false;
		} else if (!firstname.equals(other.firstname))
			return false;
		return true;
	}
}
