package net.nowtryz.cafetapi.datas;

import java.io.IOException;
import java.util.Calendar;

import net.nowtryz.cafetapi.CafetAPI;
import net.nowtryz.cafetapi.exceptions.APIException;
import net.nowtryz.cafetapi.utils.Lang;

/**
 * {@linkplain ExpenseDetail} stores information about a piece of a bill, it
 * can be acquired by the {@linkplain Expense}'s methode
 * {@link Expense#getDetails() getDetails()}
 * 
 * @author Damien (aka the guys who uses Eclipse generators)
 * @since 1.0
 */
public abstract class ExpenseDetail extends CafetData {
	protected String name;
	protected int client;
	protected float price;
	protected int quantity;
	protected Calendar date;

	/**
	 * {@linkplain ExpenseDetail} stores information about a piece of a bill, it
	 * can be acquired by the {@linkplain Expense}'s methode
	 * {@link Expense#getDetails() getDetails()}
	 * 
	 * @param name the saved name of the formula/product bought
	 * @param client the clien wich performed this transaction
	 * @param price the price
	 * @param quantity the quantity bought
	 * @param date the date at wich the transaction has been completed (a {@link java.util.Calendar Calendar} object)
	 * @since 1.0
	 * @see ExpenseDetail
	 */
	public ExpenseDetail(String name, int client, float price, int quantity, Calendar date) {
		this.name = name;
		this.client = client;
		this.price = price;
		this.quantity = quantity;
		this.date = date;
	}

	/**
	 * The Expense object is the one that stores every expense's information for
	 * later use, it's generaly generated by the Client's method
	 * getExpenses or even directly whith the Data
	 * static method getClientExpenses().
	 * 
	 * @since 1.0
	 * @see ExpenseDetail
	 */
	public ExpenseDetail() {
		this.name = Lang.get("Unknown");
		this.client = 0;
		this.price = 0.0f;
		this.quantity = 0;
		this.date = Calendar.getInstance();
	}

	/**
	 * Return the name
	 * 
	 * @return the name
	 * @since 1.0
	 */
	public final String getName() {
		return name;
	}

	/**
	 * Return the client
	 * 
	 * @return the client
	 * @throws IOException 
	 * @throws APIException 
	 * @since 1.0
	 */
	public final Client getClient() throws APIException, IOException {
		return CafetAPI.getInstance().getClient(client).getData();
	}

	/**
	 * Return the price
	 * 
	 * @return the price
	 * @since 1.0
	 */
	public final float getPrice() {
		return price;
	}

	/**
	 * Return the quantity
	 * 
	 * @return the quantity
	 * @since 1.0
	 */
	public final int getQuantity() {
		return quantity;
	}

	/**
	 * Return the date of the transaction in a {@link java.util.Calendar Calendar} object
	 * 
	 * @return the date
	 * @since 1.0
	 */
	public final Calendar getDate() {
		return date;
	}
}