package net.minestom.server.world.biomes;

import net.minestom.server.utils.NamespaceID;
import org.jglrxavpok.hephaistos.nbt.NBT;
import org.jglrxavpok.hephaistos.nbt.NBTCompound;
import org.jglrxavpok.hephaistos.nbt.NBTType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;
import java.util.stream.IntStream;


/**
 * Allows servers to register custom dimensions. Also used during player joining to send the list of all existing dimensions.
 * <p>
 * Contains {@link Biomes#PLAINS} by default but can be removed.
 */
public final class BiomeManager {
    private final List<Biome> biomes = new ArrayList<>();
    private final static Logger LOGGER = LoggerFactory.getLogger(BiomeManager.class);

    public BiomeManager() {
    }

    /**
     * Adds a new biome. This does NOT send the new list to players.
     *
     * @param biome the biome to add
     */
    public synchronized void addBiome(Biome biome) {
        if (biomes.stream().anyMatch(listBiome -> listBiome.name().equals(biome.name()))) {
            LOGGER.error("You cannot register the biome because it is already registered. Please remove the biome to overwrite it");
            return;
        }
        this.biomes.add(biome);
    }

    /**
     * Removes a biome. This does NOT send the new list to players.
     *
     * @param biome the biome to remove
     */
    public synchronized void removeBiome(Biome biome) {
        this.biomes.remove(biome);
    }

    /**
     * Returns an immutable copy of the biomes already registered.
     *
     * @return an immutable copy of the biomes already registered
     */
    public synchronized Collection<Biome> unmodifiableCollection() {
        return Collections.unmodifiableCollection(biomes);
    }

    /**
     * Gets a biome by its id.
     *
     * @param id the id of the biome
     * @return the {@link Biome} linked to this id
     */
    public synchronized Biome getById(int id) {
        return biomes.get(id);
    }

    public synchronized Biome getByName(NamespaceID namespaceID) {
        Biome biome = null;
        for (final Biome biomeT : biomes) {
            if (biomeT.name().equals(namespaceID)) {
                biome = biomeT;
                break;
            }
        }
        return biome;
    }

    public synchronized NBTCompound toNBT() {
        var biomes = new ArrayList<>(unmodifiableCollection());
        List<NBTCompound> nbtCompounds = IntStream
                .range(0, biomes.size())
                .mapToObj(i -> {
                    var biome = biomes.get(i);
                    biome.setId(i);
                    return biome;
                }).map(Biome::toNbt).toList();
        return NBT.Compound(Map.of(
                "type", NBT.String("minecraft:worldgen/biome"),
                "value", NBT.List(NBTType.TAG_Compound, nbtCompounds)));
    }
}
