/*
 * Copyright (c) 2016-2022 chronicle.software
 *
 *     https://chronicle.software
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.openhft.chronicle.bytes;

import net.openhft.chronicle.bytes.internal.ReferenceCountedUtil;
import net.openhft.chronicle.core.OS;
import net.openhft.chronicle.core.annotation.NonNegative;
import net.openhft.chronicle.core.io.IOTools;
import net.openhft.chronicle.core.io.ReferenceChangeListener;
import net.openhft.chronicle.core.io.ReferenceOwner;
import org.jetbrains.annotations.NotNull;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;

import static net.openhft.chronicle.core.util.Ints.requireNonNegative;
import static net.openhft.chronicle.core.util.Longs.requireNonNegative;
import static net.openhft.chronicle.core.util.ObjectUtils.requireNonNull;

/**
 * This is a ByteStore which uses no space but could be resized to be larger (by replacing it with a ByteStore with space)
 *
 * @deprecated from the public API for removal in x.25. Use {@link BytesStore#empty()} } instead.
 */
@Deprecated(/* to be removed in x.25 */)
@SuppressWarnings({"rawtypes", "unchecked"})
public enum NoBytesStore implements BytesStore {
    NO_BYTES_STORE;

    public static final long NO_PAGE;

    @NotNull
    public static final Bytes<?> NO_BYTES;

    static {
        NO_PAGE = OS.memory().allocate(OS.pageSize());
        NO_BYTES = new NativeBytes<>(noBytesStore());
        IOTools.unmonitor(NO_BYTES);
    }

    @NotNull
    public static <T, B extends BytesStore<B, T>> BytesStore<B, T> noBytesStore() {
        return (BytesStore<B, T>) NO_BYTES_STORE;
    }

    @Override
    public void reserve(ReferenceOwner owner)
            throws IllegalStateException {
        // Do nothing
    }

    @Override
    public void release(ReferenceOwner owner)
            throws IllegalStateException {
        // Do nothing
    }

    @Override
    public void releaseLast(ReferenceOwner id)
            throws IllegalStateException {
        // Do nothing
    }

    @Override
    public int refCount() {
        return 1;
    }

    @Override
    public void addReferenceChangeListener(ReferenceChangeListener referenceChangeListener) {
        // Do nothing
    }

    @Override
    public void removeReferenceChangeListener(ReferenceChangeListener referenceChangeListener) {
        // Do nothing
    }

    @Override
    public boolean tryReserve(ReferenceOwner owner) {
        return false;
    }

    @Override
    public boolean reservedBy(ReferenceOwner owner) {
        return true;
    }

    @NotNull
    @Override
    public RandomDataOutput writeByte(@NonNegative long offset, byte i8) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeShort(@NonNegative long offset, short i) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeInt(@NonNegative long offset, int i) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeOrderedInt(@NonNegative long offset, int i) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeLong(@NonNegative long offset, long i) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeOrderedLong(@NonNegative long offset, long i) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeFloat(@NonNegative long offset, float d) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeDouble(@NonNegative long offset, double d) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeVolatileByte(@NonNegative long offset, byte i8) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeVolatileShort(@NonNegative long offset, short i16) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeVolatileInt(@NonNegative long offset, int i32) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput writeVolatileLong(@NonNegative long offset, long i64) {
        throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput write(@NonNegative final long offsetInRDO,
                                  final byte[] byteArray,
                                  @NonNegative final int offset,
                                  @NonNegative final int length) {
        requireNonNegative(offsetInRDO);
        requireNonNull(byteArray);
        requireNonNegative(offset);
        requireNonNegative(length);
        if (length != 0)
            throw new UnsupportedOperationException();
        return this;
    }

    @Override
    public void write(@NonNegative long offsetInRDO, @NotNull ByteBuffer bytes, @NonNegative int offset, @NonNegative int length) {
        requireNonNull(bytes);
        if (length != 0)
            throw new UnsupportedOperationException();
    }

    @NotNull
    @Override
    public RandomDataOutput write(@NonNegative long writeOffset, @NotNull RandomDataInput bytes, @NonNegative long readOffset, @NonNegative long length) {
        requireNonNegative(writeOffset);
        ReferenceCountedUtil.throwExceptionIfReleased(bytes);
        requireNonNegative(readOffset);
        requireNonNegative(length);
        if (length != 0)
            throw new UnsupportedOperationException();
        return this;
    }


    @Override
    public byte readByte(@NonNegative long offset) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int peekUnsignedByte(@NonNegative long offset) {
        return -1;
    }

    @Override
    public short readShort(@NonNegative long offset) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int readInt(@NonNegative long offset) {
        throw new UnsupportedOperationException();
    }

    @Override
    public long readLong(@NonNegative long offset) {
        throw new UnsupportedOperationException();
    }

    @Override
    public float readFloat(@NonNegative long offset) {
        throw new UnsupportedOperationException();
    }

    @Override
    public double readDouble(@NonNegative long offset) {
        throw new UnsupportedOperationException();
    }

    @Override
    public byte readVolatileByte(@NonNegative long offset)
            throws BufferUnderflowException {
        throw new UnsupportedOperationException();
    }

    @Override
    public short readVolatileShort(@NonNegative long offset)
            throws BufferUnderflowException {
        throw new UnsupportedOperationException();
    }

    @Override
    public int readVolatileInt(@NonNegative long offset)
            throws BufferUnderflowException {
        throw new UnsupportedOperationException();
    }

    @Override
    public long readVolatileLong(@NonNegative long offset)
            throws BufferUnderflowException {
        throw new BufferUnderflowException();
    }

    @Override
    public boolean isDirectMemory() {
        return false;
    }

    @NotNull
    @Override
    public BytesStore copy() {
        return this;
    }

    @Override
    public @NotNull Bytes<?> bytesForRead() throws IllegalStateException {
        return new VanillaBytes(this, 0, 0);
    }

    @Override
    public @NonNegative long capacity() {
        return 0;
    }

    @Override
    public Void underlyingObject() {
        return null;
    }

    @Override
    public boolean inside(@NonNegative long offset) {
        return false;
    }

    @Override
    public boolean inside(@NonNegative long offset, @NonNegative long bufferSize) {
        return offset == 0 && bufferSize == 0;
    }

    @Override
    public long copyTo(@NotNull BytesStore store) {
        requireNonNull(store);
        // nothing to copy.
        return 0L;
    }

    @Override
    public void nativeWrite(long address, @NonNegative long position, @NonNegative long size) {
        throw new UnsupportedOperationException();
    }

    @Override
    public long write8bit(@NonNegative long position, @NotNull BytesStore bs) {
        requireNonNull(bs);
        throw new BufferOverflowException();
    }

    @Override
    public long write8bit(@NonNegative long position, @NotNull String s, @NonNegative int start, @NonNegative int length) {
        requireNonNull(s);
        throw new BufferOverflowException();
    }

    @Override
    public void nativeRead(@NonNegative long position, long address, @NonNegative long size) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean compareAndSwapInt(@NonNegative long offset, int expected, int value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void testAndSetInt(@NonNegative long offset, int expected, int value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean compareAndSwapLong(@NonNegative long offset, long expected, long value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public long appendAndReturnLength(long writePosition, boolean negative, long mantissa, int exponent, boolean append0) {
        throw new UnsupportedOperationException();
    }

    @Deprecated(/* to be removed in x.25 */)
    @Override
    public byte[] internalNumberBuffer() {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean equalBytes(@NotNull BytesStore bytesStore, long length) {
        requireNonNull(bytesStore);
        return length == 0;
    }

    @Override
    public void move(@NonNegative long from, @NonNegative long to, @NonNegative long length) {
        throw new UnsupportedOperationException();
    }

    @Override
    public long addressForRead(@NonNegative long offset)
            throws BufferUnderflowException {
        if (offset != 0)
            throw new BufferUnderflowException();
        return NO_PAGE;
    }

    @Override
    public long addressForWrite(@NonNegative long offset)
            throws BufferOverflowException {
        if (offset != 0)
            throw new BufferOverflowException();
        return NO_PAGE;
    }

    @Override
    public long addressForWritePosition()
            throws UnsupportedOperationException, BufferOverflowException {
        return NO_PAGE;
    }

    @NotNull
    @Override
    public Bytes<?> bytesForWrite() {
        throw new UnsupportedOperationException("todo");
    }

    @Override
    public boolean sharedMemory() {
        return false;
    }

    @Override
    public boolean isImmutableEmptyByteStore() {
        return true;
    }
}
