/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.collect.impl.hash;

import net.openhft.collect.*;
import net.openhft.collect.hash.*;
import net.openhft.collect.impl.*;
import net.openhft.collect.map.hash.HashLongCharMapFactory;
import net.openhft.function.LongCharConsumer;
import java.util.function.Predicate;
import net.openhft.collect.map.hash.HashLongCharMap;

import javax.annotation.Nonnull;
import java.util.*;

import static net.openhft.collect.impl.Containers.sizeAsInt;
import static net.openhft.collect.impl.hash.LHashCapacities.configIsSuitableForMutableLHash;


public abstract class LHashSeparateKVLongCharMapFactoryGO
        extends LHashSeparateKVLongCharMapFactorySO {

    LHashSeparateKVLongCharMapFactoryGO(HashConfig hashConf, int defaultExpectedSize
            , long lower, long upper) {
        super(hashConf, defaultExpectedSize, lower, upper);
    }

    

    abstract HashLongCharMapFactory thisWith(HashConfig hashConf, int defaultExpectedSize, long lower, long upper);

    abstract HashLongCharMapFactory lHashLikeThisWith(HashConfig hashConf, int defaultExpectedSize, long lower, long upper);

    abstract HashLongCharMapFactory qHashLikeThisWith(HashConfig hashConf, int defaultExpectedSize, long lower, long upper);

    @Override
    public final HashLongCharMapFactory withHashConfig(@Nonnull HashConfig hashConf) {
        if (configIsSuitableForMutableLHash(hashConf))
            return lHashLikeThisWith(hashConf, getDefaultExpectedSize()
            
                    , getLowerKeyDomainBound(), getUpperKeyDomainBound());
        return qHashLikeThisWith(hashConf, getDefaultExpectedSize()
            
                , getLowerKeyDomainBound(), getUpperKeyDomainBound());
    }

    @Override
    public final HashLongCharMapFactory withDefaultExpectedSize(int defaultExpectedSize) {
        if (defaultExpectedSize == getDefaultExpectedSize())
            return this;
        return thisWith(getHashConfig(), defaultExpectedSize
                
                , getLowerKeyDomainBound(), getUpperKeyDomainBound());
    }

    final HashLongCharMapFactory withDomain(long lower, long upper) {
        if (lower == getLowerKeyDomainBound() && upper == getUpperKeyDomainBound())
            return this;
        return thisWith(getHashConfig(), getDefaultExpectedSize(), lower, upper);
    }

    @Override
    public final HashLongCharMapFactory withKeysDomain(long lower, long upper) {
        if (lower > upper)
            throw new IllegalArgumentException("minPossibleKey shouldn't be greater " +
                    "than maxPossibleKey");
        return withDomain(lower, upper);
    }

    @Override
    public final HashLongCharMapFactory withKeysDomainComplement(long lower, long upper) {
        if (lower > upper)
            throw new IllegalArgumentException("minImpossibleKey shouldn't be greater " +
                    "than maxImpossibleKey");
        return withDomain((long) (upper + 1), (long) (lower - 1));
    }

    @Override
    public String toString() {
        return "HashLongCharMapFactory[" + commonString() + keySpecialString() +
                ",defaultValue=" + getDefaultValue() +
        "]";
    }

    @Override
    public int hashCode() {
        int hashCode = keySpecialHashCode(commonHashCode());
        hashCode = hashCode * 31 + Primitives.hashCode(getDefaultValue());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj instanceof HashLongCharMapFactory) {
            HashLongCharMapFactory factory = (HashLongCharMapFactory) obj;
            return commonEquals(factory) && keySpecialEquals(factory) &&
                    // boxing to treat NaNs correctly
                   ((Character) getDefaultValue()).equals(factory.getDefaultValue())
                    ;
        } else {
            return false;
        }
    }

    @Override
    public char getDefaultValue() {
        return (char) 0;
    }

    

    

    

    

    

    

    
    

    
    

    private  UpdatableLHashSeparateKVLongCharMapGO shrunk(
            UpdatableLHashSeparateKVLongCharMapGO map) {
        Predicate<HashContainer> shrinkCondition;
        if ((shrinkCondition = hashConf.getShrinkCondition()) != null) {
            if (shrinkCondition.test(map))
                map.shrink();
        }
        return map;
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap() {
        return newUpdatableMap(getDefaultExpectedSize());
    }
    @Override
    public  MutableLHashSeparateKVLongCharMapGO newMutableMap() {
        return newMutableMap(getDefaultExpectedSize());
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Map<Long, Character> map) {
        return shrunk(super.newUpdatableMap(map));
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Map<Long, Character> map1, Map<Long, Character> map2) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        return newUpdatableMap(map1, map2, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Map<Long, Character> map1, Map<Long, Character> map2,
            Map<Long, Character> map3) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        return newUpdatableMap(map1, map2, map3, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Map<Long, Character> map1, Map<Long, Character> map2,
            Map<Long, Character> map3, Map<Long, Character> map4) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        expectedSize += (long) map4.size();
        return newUpdatableMap(map1, map2, map3, map4, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Map<Long, Character> map1, Map<Long, Character> map2,
            Map<Long, Character> map3, Map<Long, Character> map4,
            Map<Long, Character> map5) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        expectedSize += (long) map4.size();
        expectedSize += (long) map5.size();
        return newUpdatableMap(map1, map2, map3, map4, map5, sizeAsInt(expectedSize));
    }


    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Map<Long, Character> map1, Map<Long, Character> map2,
            int expectedSize) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Map<Long, Character> map1, Map<Long, Character> map2,
            Map<Long, Character> map3, int expectedSize) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Map<Long, Character> map1, Map<Long, Character> map2,
            Map<Long, Character> map3, Map<Long, Character> map4,
            int expectedSize) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        map.putAll(map4);
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Map<Long, Character> map1, Map<Long, Character> map2,
            Map<Long, Character> map3, Map<Long, Character> map4,
            Map<Long, Character> map5, int expectedSize) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        map.putAll(map4);
        map.putAll(map5);
        return shrunk(map);
    }


    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            net.openhft.function.Consumer<LongCharConsumer> entriesSupplier) {
        return newUpdatableMap(entriesSupplier, getDefaultExpectedSize());
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            net.openhft.function.Consumer<LongCharConsumer> entriesSupplier,
            int expectedSize) {
        final UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(expectedSize);
        entriesSupplier.accept(new LongCharConsumer() {
             @Override
             public void accept(long k, char v) {
                 map.put(k, v);
             }
         });
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(long[] keys,
            char[] values) {
        return newUpdatableMap(keys, values, keys.length);
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(long[] keys,
            char[] values, int expectedSize) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(expectedSize);
        int keysLen = keys.length;
        if (keysLen != values.length)
            throw new IllegalArgumentException("keys and values arrays must have the same size");
        for (int i = 0; i < keys.length; i++) {
            map.put(keys[i], values[i]);
        }
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Long[] keys, Character[] values) {
        return newUpdatableMap(keys, values, keys.length);
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(
            Long[] keys, Character[] values, int expectedSize) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(expectedSize);
        int keysLen = keys.length;
        if (keysLen != values.length)
            throw new IllegalArgumentException("keys and values arrays must have the same size");
        for (int i = 0; i < keys.length; i++) {
            map.put(keys[i], values[i]);
        }
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(Iterable<Long> keys,
            Iterable<Character> values) {
        int expectedSize = keys instanceof Collection ? ((Collection) keys).size() :
                getDefaultExpectedSize();
        return newUpdatableMap(keys, values, expectedSize);
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMap(Iterable<Long> keys,
            Iterable<Character> values, int expectedSize) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(expectedSize);
        Iterator<Long> keysIt = keys.iterator();
        Iterator<Character> valuesIt = values.iterator();
        try {
            while (keysIt.hasNext()) {
                map.put(keysIt.next(), valuesIt.next());
            }
            return shrunk(map);
        } catch (NoSuchElementException e) {
            throw new IllegalArgumentException(
                    "keys and values iterables must have the same size", e);
        }
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMapOf(
            long k1, char v1) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(1);
        map.put(k1, v1);
        return map;
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMapOf(
            long k1, char v1, long k2, char v2) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(2);
        map.put(k1, v1);
        map.put(k2, v2);
        return map;
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMapOf(
            long k1, char v1, long k2, char v2,
            long k3, char v3) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(3);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        return map;
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMapOf(
            long k1, char v1, long k2, char v2,
            long k3, char v3, long k4, char v4) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(4);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        map.put(k4, v4);
        return map;
    }

    @Override
    public  UpdatableLHashSeparateKVLongCharMapGO newUpdatableMapOf(
            long k1, char v1, long k2, char v2,
            long k3, char v3, long k4, char v4,
            long k5, char v5) {
        UpdatableLHashSeparateKVLongCharMapGO map = newUpdatableMap(5);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        map.put(k4, v4);
        map.put(k5, v5);
        return map;
    }

    
    


    @Override
    public  HashLongCharMap newMutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, int expectedSize) {
        MutableLHashSeparateKVLongCharMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, expectedSize));
        return res;
    }

    @Override
    public  HashLongCharMap newMutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3, int expectedSize) {
        MutableLHashSeparateKVLongCharMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, expectedSize));
        return res;
    }

    @Override
    public  HashLongCharMap newMutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3,
            Map<Long, Character> map4, int expectedSize) {
        MutableLHashSeparateKVLongCharMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, expectedSize));
        return res;
    }

    @Override
    public  HashLongCharMap newMutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3,
            Map<Long, Character> map4, Map<Long, Character> map5, int expectedSize) {
        MutableLHashSeparateKVLongCharMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5, expectedSize));
        return res;
    }

    
    

    @Override
    public  HashLongCharMap newMutableMap(
            Map<Long, Character> map) {
        MutableLHashSeparateKVLongCharMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map));
        return res;
    }

    @Override
    public  HashLongCharMap newMutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2) {
        MutableLHashSeparateKVLongCharMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2));
        return res;
    }

    @Override
    public  HashLongCharMap newMutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3) {
        MutableLHashSeparateKVLongCharMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3));
        return res;
    }

    @Override
    public  HashLongCharMap newMutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3,
            Map<Long, Character> map4) {
        MutableLHashSeparateKVLongCharMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4));
        return res;
    }

    @Override
    public  HashLongCharMap newMutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3,
            Map<Long, Character> map4, Map<Long, Character> map5) {
        MutableLHashSeparateKVLongCharMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5));
        return res;
    }


    @Override
    public  HashLongCharMap newMutableMap(
            net.openhft.function.Consumer<LongCharConsumer> entriesSupplier) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(entriesSupplier));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMap(
            net.openhft.function.Consumer<LongCharConsumer> entriesSupplier,
            int expectedSize) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(entriesSupplier, expectedSize));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMap(long[] keys,
            char[] values) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMap(long[] keys,
            char[] values, int expectedSize) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMap(
            Long[] keys, Character[] values) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMap(
            Long[] keys, Character[] values, int expectedSize) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMap(Iterable<Long> keys,
            Iterable<Character> values) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMap(Iterable<Long> keys,
            Iterable<Character> values, int expectedSize) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMapOf(long k1, char v1) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMapOf(long k1, char v1,
             long k2, char v2) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMapOf(long k1, char v1,
             long k2, char v2, long k3, char v3) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMapOf(long k1, char v1,
             long k2, char v2, long k3, char v3,
             long k4, char v4) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4));
        return map;
    }

    @Override
    public  HashLongCharMap newMutableMapOf(long k1, char v1,
             long k2, char v2, long k3, char v3,
             long k4, char v4, long k5, char v5) {
        MutableLHashSeparateKVLongCharMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5));
        return map;
    }
    
    


    @Override
    public  HashLongCharMap newImmutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, int expectedSize) {
        ImmutableLHashSeparateKVLongCharMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, expectedSize));
        return res;
    }

    @Override
    public  HashLongCharMap newImmutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3, int expectedSize) {
        ImmutableLHashSeparateKVLongCharMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, expectedSize));
        return res;
    }

    @Override
    public  HashLongCharMap newImmutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3,
            Map<Long, Character> map4, int expectedSize) {
        ImmutableLHashSeparateKVLongCharMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, expectedSize));
        return res;
    }

    @Override
    public  HashLongCharMap newImmutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3,
            Map<Long, Character> map4, Map<Long, Character> map5, int expectedSize) {
        ImmutableLHashSeparateKVLongCharMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5, expectedSize));
        return res;
    }

    
    

    @Override
    public  HashLongCharMap newImmutableMap(
            Map<Long, Character> map) {
        ImmutableLHashSeparateKVLongCharMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map));
        return res;
    }

    @Override
    public  HashLongCharMap newImmutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2) {
        ImmutableLHashSeparateKVLongCharMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2));
        return res;
    }

    @Override
    public  HashLongCharMap newImmutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3) {
        ImmutableLHashSeparateKVLongCharMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3));
        return res;
    }

    @Override
    public  HashLongCharMap newImmutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3,
            Map<Long, Character> map4) {
        ImmutableLHashSeparateKVLongCharMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4));
        return res;
    }

    @Override
    public  HashLongCharMap newImmutableMap(Map<Long, Character> map1,
            Map<Long, Character> map2, Map<Long, Character> map3,
            Map<Long, Character> map4, Map<Long, Character> map5) {
        ImmutableLHashSeparateKVLongCharMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5));
        return res;
    }


    @Override
    public  HashLongCharMap newImmutableMap(
            net.openhft.function.Consumer<LongCharConsumer> entriesSupplier) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(entriesSupplier));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMap(
            net.openhft.function.Consumer<LongCharConsumer> entriesSupplier,
            int expectedSize) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(entriesSupplier, expectedSize));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMap(long[] keys,
            char[] values) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMap(long[] keys,
            char[] values, int expectedSize) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMap(
            Long[] keys, Character[] values) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMap(
            Long[] keys, Character[] values, int expectedSize) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMap(Iterable<Long> keys,
            Iterable<Character> values) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMap(Iterable<Long> keys,
            Iterable<Character> values, int expectedSize) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMapOf(long k1, char v1) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMapOf(long k1, char v1,
             long k2, char v2) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMapOf(long k1, char v1,
             long k2, char v2, long k3, char v3) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMapOf(long k1, char v1,
             long k2, char v2, long k3, char v3,
             long k4, char v4) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4));
        return map;
    }

    @Override
    public  HashLongCharMap newImmutableMapOf(long k1, char v1,
             long k2, char v2, long k3, char v3,
             long k4, char v4, long k5, char v5) {
        ImmutableLHashSeparateKVLongCharMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5));
        return map;
    }
}

