/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.collect.impl.hash;

import net.openhft.collect.*;
import net.openhft.collect.hash.*;
import net.openhft.collect.impl.*;
import net.openhft.collect.map.hash.HashIntByteMapFactory;
import net.openhft.function.IntByteConsumer;
import java.util.function.Predicate;
import net.openhft.collect.map.hash.HashIntByteMap;

import javax.annotation.Nonnull;
import java.util.*;

import static net.openhft.collect.impl.Containers.sizeAsInt;
import static net.openhft.collect.impl.hash.LHashCapacities.configIsSuitableForMutableLHash;


public abstract class QHashSeparateKVIntByteMapFactoryGO
        extends QHashSeparateKVIntByteMapFactorySO {

    QHashSeparateKVIntByteMapFactoryGO(HashConfig hashConf, int defaultExpectedSize
            , int lower, int upper) {
        super(hashConf, defaultExpectedSize, lower, upper);
    }

    

    abstract HashIntByteMapFactory thisWith(HashConfig hashConf, int defaultExpectedSize, int lower, int upper);

    abstract HashIntByteMapFactory lHashLikeThisWith(HashConfig hashConf, int defaultExpectedSize, int lower, int upper);

    abstract HashIntByteMapFactory qHashLikeThisWith(HashConfig hashConf, int defaultExpectedSize, int lower, int upper);

    @Override
    public final HashIntByteMapFactory withHashConfig(@Nonnull HashConfig hashConf) {
        if (configIsSuitableForMutableLHash(hashConf))
            return lHashLikeThisWith(hashConf, getDefaultExpectedSize()
            
                    , getLowerKeyDomainBound(), getUpperKeyDomainBound());
        return qHashLikeThisWith(hashConf, getDefaultExpectedSize()
            
                , getLowerKeyDomainBound(), getUpperKeyDomainBound());
    }

    @Override
    public final HashIntByteMapFactory withDefaultExpectedSize(int defaultExpectedSize) {
        if (defaultExpectedSize == getDefaultExpectedSize())
            return this;
        return thisWith(getHashConfig(), defaultExpectedSize
                
                , getLowerKeyDomainBound(), getUpperKeyDomainBound());
    }

    final HashIntByteMapFactory withDomain(int lower, int upper) {
        if (lower == getLowerKeyDomainBound() && upper == getUpperKeyDomainBound())
            return this;
        return thisWith(getHashConfig(), getDefaultExpectedSize(), lower, upper);
    }

    @Override
    public final HashIntByteMapFactory withKeysDomain(int lower, int upper) {
        if (lower > upper)
            throw new IllegalArgumentException("minPossibleKey shouldn't be greater " +
                    "than maxPossibleKey");
        return withDomain(lower, upper);
    }

    @Override
    public final HashIntByteMapFactory withKeysDomainComplement(int lower, int upper) {
        if (lower > upper)
            throw new IllegalArgumentException("minImpossibleKey shouldn't be greater " +
                    "than maxImpossibleKey");
        return withDomain((int) (upper + 1), (int) (lower - 1));
    }

    @Override
    public String toString() {
        return "HashIntByteMapFactory[" + commonString() + keySpecialString() +
                ",defaultValue=" + getDefaultValue() +
        "]";
    }

    @Override
    public int hashCode() {
        int hashCode = keySpecialHashCode(commonHashCode());
        hashCode = hashCode * 31 + Primitives.hashCode(getDefaultValue());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj instanceof HashIntByteMapFactory) {
            HashIntByteMapFactory factory = (HashIntByteMapFactory) obj;
            return commonEquals(factory) && keySpecialEquals(factory) &&
                    // boxing to treat NaNs correctly
                   ((Byte) getDefaultValue()).equals(factory.getDefaultValue())
                    ;
        } else {
            return false;
        }
    }

    @Override
    public byte getDefaultValue() {
        return (byte) 0;
    }

    

    

    

    

    

    

    
    

    
    

    private  UpdatableQHashSeparateKVIntByteMapGO shrunk(
            UpdatableQHashSeparateKVIntByteMapGO map) {
        Predicate<HashContainer> shrinkCondition;
        if ((shrinkCondition = hashConf.getShrinkCondition()) != null) {
            if (shrinkCondition.test(map))
                map.shrink();
        }
        return map;
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap() {
        return newUpdatableMap(getDefaultExpectedSize());
    }
    @Override
    public  MutableQHashSeparateKVIntByteMapGO newMutableMap() {
        return newMutableMap(getDefaultExpectedSize());
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Map<Integer, Byte> map) {
        return shrunk(super.newUpdatableMap(map));
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Map<Integer, Byte> map1, Map<Integer, Byte> map2) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        return newUpdatableMap(map1, map2, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Map<Integer, Byte> map1, Map<Integer, Byte> map2,
            Map<Integer, Byte> map3) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        return newUpdatableMap(map1, map2, map3, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Map<Integer, Byte> map1, Map<Integer, Byte> map2,
            Map<Integer, Byte> map3, Map<Integer, Byte> map4) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        expectedSize += (long) map4.size();
        return newUpdatableMap(map1, map2, map3, map4, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Map<Integer, Byte> map1, Map<Integer, Byte> map2,
            Map<Integer, Byte> map3, Map<Integer, Byte> map4,
            Map<Integer, Byte> map5) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        expectedSize += (long) map4.size();
        expectedSize += (long) map5.size();
        return newUpdatableMap(map1, map2, map3, map4, map5, sizeAsInt(expectedSize));
    }


    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Map<Integer, Byte> map1, Map<Integer, Byte> map2,
            int expectedSize) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Map<Integer, Byte> map1, Map<Integer, Byte> map2,
            Map<Integer, Byte> map3, int expectedSize) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Map<Integer, Byte> map1, Map<Integer, Byte> map2,
            Map<Integer, Byte> map3, Map<Integer, Byte> map4,
            int expectedSize) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        map.putAll(map4);
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Map<Integer, Byte> map1, Map<Integer, Byte> map2,
            Map<Integer, Byte> map3, Map<Integer, Byte> map4,
            Map<Integer, Byte> map5, int expectedSize) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        map.putAll(map4);
        map.putAll(map5);
        return shrunk(map);
    }


    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            net.openhft.function.Consumer<IntByteConsumer> entriesSupplier) {
        return newUpdatableMap(entriesSupplier, getDefaultExpectedSize());
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            net.openhft.function.Consumer<IntByteConsumer> entriesSupplier,
            int expectedSize) {
        final UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(expectedSize);
        entriesSupplier.accept(new IntByteConsumer() {
             @Override
             public void accept(int k, byte v) {
                 map.put(k, v);
             }
         });
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(int[] keys,
            byte[] values) {
        return newUpdatableMap(keys, values, keys.length);
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(int[] keys,
            byte[] values, int expectedSize) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(expectedSize);
        int keysLen = keys.length;
        if (keysLen != values.length)
            throw new IllegalArgumentException("keys and values arrays must have the same size");
        for (int i = 0; i < keys.length; i++) {
            map.put(keys[i], values[i]);
        }
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Integer[] keys, Byte[] values) {
        return newUpdatableMap(keys, values, keys.length);
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(
            Integer[] keys, Byte[] values, int expectedSize) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(expectedSize);
        int keysLen = keys.length;
        if (keysLen != values.length)
            throw new IllegalArgumentException("keys and values arrays must have the same size");
        for (int i = 0; i < keys.length; i++) {
            map.put(keys[i], values[i]);
        }
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(Iterable<Integer> keys,
            Iterable<Byte> values) {
        int expectedSize = keys instanceof Collection ? ((Collection) keys).size() :
                getDefaultExpectedSize();
        return newUpdatableMap(keys, values, expectedSize);
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMap(Iterable<Integer> keys,
            Iterable<Byte> values, int expectedSize) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(expectedSize);
        Iterator<Integer> keysIt = keys.iterator();
        Iterator<Byte> valuesIt = values.iterator();
        try {
            while (keysIt.hasNext()) {
                map.put(keysIt.next(), valuesIt.next());
            }
            return shrunk(map);
        } catch (NoSuchElementException e) {
            throw new IllegalArgumentException(
                    "keys and values iterables must have the same size", e);
        }
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMapOf(
            int k1, byte v1) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(1);
        map.put(k1, v1);
        return map;
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMapOf(
            int k1, byte v1, int k2, byte v2) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(2);
        map.put(k1, v1);
        map.put(k2, v2);
        return map;
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(3);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        return map;
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3, int k4, byte v4) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(4);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        map.put(k4, v4);
        return map;
    }

    @Override
    public  UpdatableQHashSeparateKVIntByteMapGO newUpdatableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3, int k4, byte v4,
            int k5, byte v5) {
        UpdatableQHashSeparateKVIntByteMapGO map = newUpdatableMap(5);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        map.put(k4, v4);
        map.put(k5, v5);
        return map;
    }

    
    


    @Override
    public  HashIntByteMap newMutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, int expectedSize) {
        MutableQHashSeparateKVIntByteMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, expectedSize));
        return res;
    }

    @Override
    public  HashIntByteMap newMutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3, int expectedSize) {
        MutableQHashSeparateKVIntByteMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, expectedSize));
        return res;
    }

    @Override
    public  HashIntByteMap newMutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3,
            Map<Integer, Byte> map4, int expectedSize) {
        MutableQHashSeparateKVIntByteMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, expectedSize));
        return res;
    }

    @Override
    public  HashIntByteMap newMutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3,
            Map<Integer, Byte> map4, Map<Integer, Byte> map5, int expectedSize) {
        MutableQHashSeparateKVIntByteMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5, expectedSize));
        return res;
    }

    
    

    @Override
    public  HashIntByteMap newMutableMap(
            Map<Integer, Byte> map) {
        MutableQHashSeparateKVIntByteMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map));
        return res;
    }

    @Override
    public  HashIntByteMap newMutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2) {
        MutableQHashSeparateKVIntByteMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2));
        return res;
    }

    @Override
    public  HashIntByteMap newMutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3) {
        MutableQHashSeparateKVIntByteMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3));
        return res;
    }

    @Override
    public  HashIntByteMap newMutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3,
            Map<Integer, Byte> map4) {
        MutableQHashSeparateKVIntByteMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4));
        return res;
    }

    @Override
    public  HashIntByteMap newMutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3,
            Map<Integer, Byte> map4, Map<Integer, Byte> map5) {
        MutableQHashSeparateKVIntByteMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5));
        return res;
    }


    @Override
    public  HashIntByteMap newMutableMap(
            net.openhft.function.Consumer<IntByteConsumer> entriesSupplier) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(entriesSupplier));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMap(
            net.openhft.function.Consumer<IntByteConsumer> entriesSupplier,
            int expectedSize) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(entriesSupplier, expectedSize));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMap(int[] keys,
            byte[] values) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMap(int[] keys,
            byte[] values, int expectedSize) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMap(
            Integer[] keys, Byte[] values) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMap(
            Integer[] keys, Byte[] values, int expectedSize) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMap(Iterable<Integer> keys,
            Iterable<Byte> values) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMap(Iterable<Integer> keys,
            Iterable<Byte> values, int expectedSize) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMapOf(int k1, byte v1) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMapOf(int k1, byte v1,
             int k2, byte v2) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMapOf(int k1, byte v1,
             int k2, byte v2, int k3, byte v3) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMapOf(int k1, byte v1,
             int k2, byte v2, int k3, byte v3,
             int k4, byte v4) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4));
        return map;
    }

    @Override
    public  HashIntByteMap newMutableMapOf(int k1, byte v1,
             int k2, byte v2, int k3, byte v3,
             int k4, byte v4, int k5, byte v5) {
        MutableQHashSeparateKVIntByteMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5));
        return map;
    }
    
    


    @Override
    public  HashIntByteMap newImmutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, int expectedSize) {
        ImmutableQHashSeparateKVIntByteMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, expectedSize));
        return res;
    }

    @Override
    public  HashIntByteMap newImmutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3, int expectedSize) {
        ImmutableQHashSeparateKVIntByteMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, expectedSize));
        return res;
    }

    @Override
    public  HashIntByteMap newImmutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3,
            Map<Integer, Byte> map4, int expectedSize) {
        ImmutableQHashSeparateKVIntByteMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, expectedSize));
        return res;
    }

    @Override
    public  HashIntByteMap newImmutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3,
            Map<Integer, Byte> map4, Map<Integer, Byte> map5, int expectedSize) {
        ImmutableQHashSeparateKVIntByteMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5, expectedSize));
        return res;
    }

    
    

    @Override
    public  HashIntByteMap newImmutableMap(
            Map<Integer, Byte> map) {
        ImmutableQHashSeparateKVIntByteMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map));
        return res;
    }

    @Override
    public  HashIntByteMap newImmutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2) {
        ImmutableQHashSeparateKVIntByteMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2));
        return res;
    }

    @Override
    public  HashIntByteMap newImmutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3) {
        ImmutableQHashSeparateKVIntByteMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3));
        return res;
    }

    @Override
    public  HashIntByteMap newImmutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3,
            Map<Integer, Byte> map4) {
        ImmutableQHashSeparateKVIntByteMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4));
        return res;
    }

    @Override
    public  HashIntByteMap newImmutableMap(Map<Integer, Byte> map1,
            Map<Integer, Byte> map2, Map<Integer, Byte> map3,
            Map<Integer, Byte> map4, Map<Integer, Byte> map5) {
        ImmutableQHashSeparateKVIntByteMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5));
        return res;
    }


    @Override
    public  HashIntByteMap newImmutableMap(
            net.openhft.function.Consumer<IntByteConsumer> entriesSupplier) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(entriesSupplier));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMap(
            net.openhft.function.Consumer<IntByteConsumer> entriesSupplier,
            int expectedSize) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(entriesSupplier, expectedSize));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMap(int[] keys,
            byte[] values) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMap(int[] keys,
            byte[] values, int expectedSize) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMap(
            Integer[] keys, Byte[] values) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMap(
            Integer[] keys, Byte[] values, int expectedSize) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMap(Iterable<Integer> keys,
            Iterable<Byte> values) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMap(Iterable<Integer> keys,
            Iterable<Byte> values, int expectedSize) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMapOf(int k1, byte v1) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMapOf(int k1, byte v1,
             int k2, byte v2) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMapOf(int k1, byte v1,
             int k2, byte v2, int k3, byte v3) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMapOf(int k1, byte v1,
             int k2, byte v2, int k3, byte v3,
             int k4, byte v4) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4));
        return map;
    }

    @Override
    public  HashIntByteMap newImmutableMapOf(int k1, byte v1,
             int k2, byte v2, int k3, byte v3,
             int k4, byte v4, int k5, byte v5) {
        ImmutableQHashSeparateKVIntByteMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5));
        return map;
    }
}

