/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.map.hash;

import net.openhft.koloboke.collect.map.ObjLongMap;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.ServiceLoader;


/**
 * This class consists only of static factory methods to construct {@code HashObjLongMap}s, and
 * the default {@link HashObjLongMapFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashObjLongMap
 */
public final class HashObjLongMaps {
    
    private static class DefaultFactoryHolder {
        private static final HashObjLongMapFactory defaultFactory =
                ServiceLoader.load(HashObjLongMapFactory.class).iterator().next();
    }

    /**
     * Returns the default {@link HashObjLongMapFactory} implementation, to which
     * all static methods in this class delegate.
     *
      * @param <K> the most general key type of the maps that could
                               be constructed by the returned factory 
     
     * @return the default {@link HashObjLongMapFactory} implementation
     * @throws RuntimeException if no implementations
     *         of {@link HashObjLongMapFactory} are provided
     */
    @Nonnull
    public static <K> HashObjLongMapFactory<K> getDefaultFactory() {
        return (HashObjLongMapFactory<K>) DefaultFactoryHolder.defaultFactory;
    }

    

    

    

    


    /**
     * Constructs a new empty mutable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap() newMutableMap()}.
     *
      * @param <K> the key type of the returned map 
     
     * @return a new empty mutable map
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap() {
        return getDefaultFactory().newMutableMap();
    }

    /**
     * Constructs a new empty mutable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(int) newMutableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new empty mutable map of the given expected size
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(int expectedSize) {
        return getDefaultFactory().newMutableMap(expectedSize);
    }

    
    
    
    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     * <p>If the specified map is an instance of {@code ObjLongMap} and has the same
     * {@linkplain ObjLongMap#keyEquivalence() key equivalence} with this factory (and thus
     * the constructed map), the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(
     * Map, int) newMutableMap(map, expectedSize)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Map<? extends K, Long> map, int expectedSize) {
        return getDefaultFactory().newMutableMap(map, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Map,
     * Map, int) newMutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Map, Map,
     * Map, int) newMutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Map, Map,
     * Map, Map, int) newMutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map, int) newMutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4,
            @Nonnull Map<? extends K, Long> map5, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(
     * Consumer, int) newMutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.ObjLongConsumer<K>> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newMutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Object[], long[], int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull K[] keys, @Nonnull long[] values, int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Object[],
     * Long[], int) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull K[] keys, @Nonnull Long[] values, int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Iterable, Iterable, int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Long> values
            , int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(
     * Map) newMutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Map<? extends K, Long> map) {
        return getDefaultFactory().newMutableMap(map);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Map,
     * Map) newMutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2) {
        return getDefaultFactory().newMutableMap(map1, map2);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Map, Map,
     * Map) newMutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3) {
        return getDefaultFactory().newMutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Map, Map,
     * Map, Map) newMutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map) newMutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4,
            @Nonnull Map<? extends K, Long> map5) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(
     * Consumer) newMutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.ObjLongConsumer<K>> entriesSupplier
            ) {
        return getDefaultFactory().newMutableMap(entriesSupplier);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Object[], long[]
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull K[] keys, @Nonnull long[] values) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Object[],
     * Long[]) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull K[] keys, @Nonnull Long[] values) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMap(Iterable, Iterable
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Long> values
            ) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMapOf(Object, long
     * ) newMutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map of the single specified mapping
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMapOf(
            K k1, long v1) {
        return getDefaultFactory().newMutableMapOf(k1, v1);
    }

    /**
     * Constructs a new mutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMapOf(Object, long,
     * Object, long) newMutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map of the two specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMapOf(
            K k1, long v1, K k2, long v2) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new mutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMapOf(Object, long,
     * Object, long, Object, long) newMutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map of the three specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMapOf(
            K k1, long v1, K k2, long v2,
            K k3, long v3) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new mutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMapOf(Object, long,
     * Object, long, Object, long, Object, long
     * ) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map of the four specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMapOf(
            K k1, long v1, K k2, long v2,
            K k3, long v3, K k4, long v4) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new mutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newMutableMapOf(Object, long,
     * Object, long, Object, long, Object, long,
     * Object, long) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map of the five specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newMutableMapOf(
            K k1, long v1, K k2, long v2,
            K k3, long v3, K k4, long v4,
            K k5, long v5) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    /**
     * Constructs a new empty updatable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap() newUpdatableMap()}.
     *
      * @param <K> the key type of the returned map 
     
     * @return a new empty updatable map
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap() {
        return getDefaultFactory().newUpdatableMap();
    }

    /**
     * Constructs a new empty updatable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(int) newUpdatableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new empty updatable map of the given expected size
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(int expectedSize) {
        return getDefaultFactory().newUpdatableMap(expectedSize);
    }

    
    
    
    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     * <p>If the specified map is an instance of {@code ObjLongMap} and has the same
     * {@linkplain ObjLongMap#keyEquivalence() key equivalence} with this factory (and thus
     * the constructed map), the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(
     * Map, int) newUpdatableMap(map, expectedSize)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Long> map, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Map,
     * Map, int) newUpdatableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Map, Map,
     * Map, int) newUpdatableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, int) newUpdatableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map, int) newUpdatableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4,
            @Nonnull Map<? extends K, Long> map5, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(
     * Consumer, int) newUpdatableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.ObjLongConsumer<K>> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newUpdatableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Object[], long[], int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull K[] keys, @Nonnull long[] values, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Object[],
     * Long[], int) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull K[] keys, @Nonnull Long[] values, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Iterable, Iterable, int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Long> values
            , int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(
     * Map) newUpdatableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Long> map) {
        return getDefaultFactory().newUpdatableMap(map);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Map,
     * Map) newUpdatableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2) {
        return getDefaultFactory().newUpdatableMap(map1, map2);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Map, Map,
     * Map) newUpdatableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Map, Map,
     * Map, Map) newUpdatableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map) newUpdatableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4,
            @Nonnull Map<? extends K, Long> map5) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(
     * Consumer) newUpdatableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.ObjLongConsumer<K>> entriesSupplier
            ) {
        return getDefaultFactory().newUpdatableMap(entriesSupplier);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Object[], long[]
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull K[] keys, @Nonnull long[] values) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Object[],
     * Long[]) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull K[] keys, @Nonnull Long[] values) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMap(Iterable, Iterable
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Long> values
            ) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMapOf(Object, long
     * ) newUpdatableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map of the single specified mapping
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMapOf(
            K k1, long v1) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1);
    }

    /**
     * Constructs a new updatable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMapOf(Object, long,
     * Object, long) newUpdatableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map of the two specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMapOf(
            K k1, long v1, K k2, long v2) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new updatable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMapOf(Object, long,
     * Object, long, Object, long) newUpdatableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map of the three specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMapOf(
            K k1, long v1, K k2, long v2,
            K k3, long v3) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new updatable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMapOf(Object, long,
     * Object, long, Object, long, Object, long
     * ) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map of the four specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMapOf(
            K k1, long v1, K k2, long v2,
            K k3, long v3, K k4, long v4) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new updatable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newUpdatableMapOf(Object, long,
     * Object, long, Object, long, Object, long,
     * Object, long) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map of the five specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newUpdatableMapOf(
            K k1, long v1, K k2, long v2,
            K k3, long v3, K k4, long v4,
            K k5, long v5) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }


    
    
    
    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     * <p>If the specified map is an instance of {@code ObjLongMap} and has the same
     * {@linkplain ObjLongMap#keyEquivalence() key equivalence} with this factory (and thus
     * the constructed map), the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(
     * Map, int) newImmutableMap(map, expectedSize)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Long> map, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Map,
     * Map, int) newImmutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Map, Map,
     * Map, int) newImmutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Map, Map,
     * Map, Map, int) newImmutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map, int) newImmutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4,
            @Nonnull Map<? extends K, Long> map5, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(
     * Consumer, int) newImmutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.ObjLongConsumer<K>> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newImmutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Object[], long[], int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull K[] keys, @Nonnull long[] values, int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Object[],
     * Long[], int) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull K[] keys, @Nonnull Long[] values, int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Iterable, Iterable, int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Long> values
            , int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(
     * Map) newImmutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Long> map) {
        return getDefaultFactory().newImmutableMap(map);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Map,
     * Map) newImmutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2) {
        return getDefaultFactory().newImmutableMap(map1, map2);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Map, Map,
     * Map) newImmutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Map, Map,
     * Map, Map) newImmutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map) newImmutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Long> map1,
            @Nonnull Map<? extends K, Long> map2,
            @Nonnull Map<? extends K, Long> map3,
            @Nonnull Map<? extends K, Long> map4,
            @Nonnull Map<? extends K, Long> map5) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(
     * Consumer) newImmutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.ObjLongConsumer<K>> entriesSupplier
            ) {
        return getDefaultFactory().newImmutableMap(entriesSupplier);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Object[], long[]
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull K[] keys, @Nonnull long[] values) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Object[],
     * Long[]) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull K[] keys, @Nonnull Long[] values) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMap(Iterable, Iterable
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Long> values
            ) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMapOf(Object, long
     * ) newImmutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map of the single specified mapping
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMapOf(
            K k1, long v1) {
        return getDefaultFactory().newImmutableMapOf(k1, v1);
    }

    /**
     * Constructs a new immutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMapOf(Object, long,
     * Object, long) newImmutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map of the two specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMapOf(
            K k1, long v1, K k2, long v2) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new immutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMapOf(Object, long,
     * Object, long, Object, long) newImmutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map of the three specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMapOf(
            K k1, long v1, K k2, long v2,
            K k3, long v3) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new immutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMapOf(Object, long,
     * Object, long, Object, long, Object, long
     * ) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map of the four specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMapOf(
            K k1, long v1, K k2, long v2,
            K k3, long v3, K k4, long v4) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new immutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjLongMapFactory#newImmutableMapOf(Object, long,
     * Object, long, Object, long, Object, long,
     * Object, long) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map of the five specified mappings
     */
    @Nonnull
    public static <K> HashObjLongMap<K> newImmutableMapOf(
            K k1, long v1, K k2, long v2,
            K k3, long v3, K k4, long v4,
            K k5, long v5) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    private HashObjLongMaps() {}
}

