/**
 * Copyright (C) 2011-2015 Incapture Technologies LLC
 *
 * This is an autogenerated license statement. When copyright notices appear below
 * this one that copyright supercedes this statement.
 *
 * Unless required by applicable law or agreed to in writing, software is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * Unless explicit permission obtained in writing this software cannot be distributed.
 */
package rapture.postgres;

import java.sql.SQLException;
import java.util.Map;
import java.util.concurrent.ExecutionException;

import javax.sql.DataSource;

import org.apache.log4j.Logger;

import rapture.common.exception.ExceptionToString;
import rapture.common.exception.RaptureException;
import rapture.common.exception.RaptureExceptionFactory;
import rapture.postgres.connection.DataSourceMonitor;
import rapture.postgres.connection.cache.ConnectionCacheLoader;
import rapture.postgres.connection.cache.ConnectionInfo;
import rapture.repo.jdbc.TransactionAwareDataSource;
import rapture.repo.postgres.PostgresSanitizer;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.LoadingCache;
import com.mchange.v2.c3p0.PooledDataSource;

/**
 * Methods to get things from postgres in a standard way
 * 
 * @author alanmoore
 */
public enum PostgresFactory {
    INSTANCE;

    private static final Logger log = Logger.getLogger(PostgresFactory.class);

    private LoadingCache<String, ConnectionInfo> connectionCache;

    public static TransactionAwareDataSource getDataSource(String instanceName) {
        return INSTANCE.getConnectionInfo(instanceName).getDataSource();
    }

    public static PostgresSanitizer getSanitizer(String instanceName) {
        return INSTANCE.getConnectionInfo(instanceName).getSanitizer();
    }

    private PostgresFactory() {
        connectionCache = CacheBuilder.newBuilder().maximumSize(10).build(new ConnectionCacheLoader(new DataSourceMonitor()));

        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {
            @Override
            public void run() {
                for (Map.Entry<String, ConnectionInfo> entry : connectionCache.asMap().entrySet()) {
                    DataSource ds = entry.getValue().getDataSource();
                    if (ds != null && ds instanceof PooledDataSource) {
                        PooledDataSource pds = (PooledDataSource) ds;
                        try {
                            pds.close();
                        } catch (SQLException e) {
                            log.error(ExceptionToString.format(e));
                        }
                    }
                }
            }
        }));
    }

    private ConnectionInfo getConnectionInfo(String instanceName) {
        if (instanceName == null) {
            instanceName = "default";
        }
        try {
            return connectionCache.get(instanceName);
        } catch (Exception e) {
            if (e.getCause() != null && e.getCause() instanceof RaptureException) {
                throw (RaptureException) e.getCause();
            } else {
                throw RaptureExceptionFactory.create(String.format("Error while creating postgres connection for instance [%s]: %s", instanceName,
                        ExceptionToString.format(e)));
            }
        }
    }

}
