/**
 * Copyright (C) 2011-2015 Incapture Technologies LLC
 *
 * This is an autogenerated license statement. When copyright notices appear below
 * this one that copyright supercedes this statement.
 *
 * Unless required by applicable law or agreed to in writing, software is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * Unless explicit permission obtained in writing this software cannot be distributed.
 */
package rapture.repo.postgres;

import rapture.common.RaptureFolderInfo;
import rapture.common.exception.RaptNotSupportedException;
import rapture.common.impl.jackson.JacksonUtil;
import rapture.index.IndexProducer;
import rapture.index.IndexHandler;
import rapture.postgres.PostgresFactory;
import rapture.repo.AbstractKeyStore;
import rapture.repo.KeyStore;
import rapture.repo.StoreKeyVisitor;
import rapture.table.postgres.PostgresIndexHandler;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

import com.google.common.collect.Lists;

public class PostgresDataStore extends AbstractKeyStore implements KeyStore {
    public static final String PREFIX = "prefix";

    @Override
    public void setConfig(Map<String, String> config) {
        tableName = config.get(PREFIX).replace(".", "_");
        if (tableName == null || tableName.isEmpty()) {
            tableName = "__data__" + instanceName;
        }

        docHandler = new PostgresDocHandler(instanceName, tableName);
        docHandler.setDataSource(PostgresFactory.getDataSource(instanceName));
        docHandler.initialize();

        if (needsFolderHandling) {
            folderHandler = new PostgresFolderHandler(instanceName, tableName);
            folderHandler.setDataSource(PostgresFactory.getDataSource(instanceName));
            folderHandler.initialize();
        }
    }

    @Override
    public KeyStore createRelatedKeyStore(String relation) {
        Map<String, String> config = new HashMap<String, String>();
        config.put(PREFIX, tableName + "_" + relation);
        PostgresDataStore related = new PostgresDataStore();
        related.resetNeedsFolderHandling();
        related.setInstanceName(instanceName);
        related.setConfig(config);
        return related;
    }

    @Override
    public void setInstanceName(String instanceName) {
        this.instanceName = instanceName;
    }

    @Override
    public void visitKeys(final String prefix, final StoreKeyVisitor iStoreKeyVisitor) {
        // Visit all keys and values
        docHandler.visitKeys(prefix, iStoreKeyVisitor);
    }

    @Override
    public IndexHandler createIndexHandler(IndexProducer indexProducer) {
        Map<String, String> config = new HashMap<String, String>();
        config.put(PREFIX, tableName);

        PostgresIndexHandler indexHandler = new PostgresIndexHandler();
        indexHandler.setInstanceName(instanceName);
        indexHandler.setIndexProducer(indexProducer);
        indexHandler.setConfig(config);
        indexHandler.setDataSource(PostgresFactory.getDataSource(instanceName));

        indexHandler.initialize();
        return indexHandler;
    }

    @Override
    public Boolean validate() {
        return true;
    }

    @Override
    public long getSize() {
        return docHandler.getSize();
    }

    @Override
    public boolean matches(String key, String value) {
        if (value != null) {
            String existing = get(key);
            if (existing != null) {
                existing = StringUtils.deleteWhitespace(JacksonUtil.jsonFromObject(JacksonUtil.getMapFromJson(existing)));
                //noinspection ConstantConditions
                return existing.equals(StringUtils.deleteWhitespace(value));
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    @Override
    public void resetFolderHandling() {

    }

    @Override
    public boolean containsKey(String ref) {
        return docHandler.exists(ref);
    }

    @Override
    public void put(String key, String value) {
        docHandler.put(key, value);
        if (needsFolderHandling) {
            folderHandler.addKey(key);
        }
    }

    @Override
    public String get(String k) {
        return docHandler.get(k);
    }

    @Override
    public long countKeys() throws RaptNotSupportedException {
        return docHandler.getCount();
    }

    @Override
    public boolean delete(String key) {
        boolean ret = docHandler.delete(key);

        if (ret && needsFolderHandling) {
            // Do something with folder handling
            folderHandler.removeKey(key);
        }
        return ret;
    }

    @Override
    public boolean dropKeyStore() {
        docHandler.drop();
        if (needsFolderHandling) {
            folderHandler.drop();
        }
        return true;
    }

    private boolean needsFolderHandling = true;

    public void resetNeedsFolderHandling() {
        needsFolderHandling = false;
    }

    @Override
    public List<String> getBatch(final List<String> keys) {
        List<String> ret = new ArrayList<String>();
        // Batch is slow at present
        for (String k : keys) {
            ret.add(get(k));
        }
        return ret;
    }

    @Override
    public String getStoreId() {
        return tableName;
    }

    @Override
    public List<RaptureFolderInfo> getSubKeys(String prefix) {
        // return dir.getChildren(prefix);
        return needsFolderHandling ? folderHandler.getChildren(prefix) : new ArrayList<RaptureFolderInfo>();
    }

    @Override
    public List<RaptureFolderInfo> removeSubKeys(String folder, Boolean force) {
        List<RaptureFolderInfo> ret = new ArrayList<RaptureFolderInfo>();
        removeEntries(ret, folder);
        return ret;
    }

    private void removeEntries(List<RaptureFolderInfo> ret, String prefix) {
        List<RaptureFolderInfo> entries = getSubKeys(prefix);
        for (RaptureFolderInfo i : entries) {
            String nextLevel = prefix + "/" + i.getName();
            if (i.isFolder()) {
                removeEntries(ret, nextLevel);
            } else {
                delete(nextLevel);
                RaptureFolderInfo nextRfi = new RaptureFolderInfo();
                nextRfi.setName(nextLevel);
                nextRfi.setFolder(false);
                ret.add(nextRfi);
            }
        }
        RaptureFolderInfo topRfi = new RaptureFolderInfo();
        topRfi.setFolder(true);
        topRfi.setName(prefix);
        ret.add(topRfi);
        folderHandler.removeKey(prefix);
    }

    @Override
    public List<String> getAllSubKeys(String displayNamePart) {
        List<String> result = Lists.newArrayList();
        // expandTree(base, result, null);
        return result;
    }

    private String tableName;
    private String instanceName = "default";
    private PostgresDocHandler docHandler;
    private PostgresFolderHandler folderHandler;
}
