/**
 * Copyright (C) 2011-2015 Incapture Technologies LLC
 *
 * This is an autogenerated license statement. When copyright notices appear below
 * this one that copyright supercedes this statement.
 *
 * Unless required by applicable law or agreed to in writing, software is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * Unless explicit permission obtained in writing this software cannot be distributed.
 */
package rapture.table.postgres;

import rapture.common.exception.RaptureExceptionFactory;
import rapture.dsl.idef.IndexFieldType;
import rapture.dsl.iqry.WhereClause;
import rapture.dsl.iqry.WhereExtension;
import rapture.dsl.iqry.WhereStatement;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpStatus;
import org.apache.log4j.Logger;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

/**
 * @author bardhi
 * @since 4/9/15.
 */
public class QueryParser {
    private static final Logger log = Logger.getLogger(QueryParser.class);

    static PostgresWhereClause parseWhereClause(Map<String, IndexFieldType> fieldToType, WhereClause where) {
        List<String> sqls = new LinkedList<>();

        Map<String, Object> paramsMap = new HashMap<>();
        sqls.add(parseClause(fieldToType, where.getPrimary(), paramsMap));
        for (WhereExtension whereExtension : where.getExtensions()) {
            WhereStatement clause = whereExtension.getClause();
            sqls.add(parseClause(fieldToType, clause, paramsMap));
        }
        String sql = StringUtils.join(sqls, " AND ");
        PostgresWhereClause whereClause = new PostgresWhereClause();
        whereClause.setSql(sql);
        MapSqlParameterSource parameterSource = new MapSqlParameterSource(paramsMap);
        whereClause.setParameterSource(parameterSource);
        return whereClause;
    }

    private static String parseClause(Map<String, IndexFieldType> fieldToType, WhereStatement whereStatement, Map<String, Object> paramsMap) {
        String format = null;
        switch (whereStatement.getOper()) {
            case EQUAL:
                format = "%s = %s";
                break;
            case GT:
                format = "%s > %s";
                break;
            case LT:
                format = "%s < %s";
                break;
            case NOTEQUAL:
                format = "%s != %s";
                break;
        }

        String fieldName = whereStatement.getField();
        IndexFieldType type = fieldToType.get(fieldName);
        String pgType = FieldTypeMapper.getPgType(type);
        String formattedField = String.format("(content->>'%s')::%s", fieldName, pgType);
        String valueString = whereStatement.getValue().getValue().toString();

        try {
            paramsMap.put(fieldName, convertValue(valueString, type));
        } catch (Exception e) {
            throw RaptureExceptionFactory
                    .create(HttpStatus.SC_BAD_REQUEST, String.format("Bad value in query. [%s] is bad for field %s of type %s", valueString, fieldName, type));
        }

        return String.format(format, formattedField, ":" + fieldName);

    }

    private static Object convertValue(String valueString, IndexFieldType type) {
        Object retVal;
        switch (type) {
            case LONG:
                retVal = Long.parseLong(valueString);
                break;
            case DOUBLE:
                retVal = Double.parseDouble(valueString);
                break;
            case INTEGER:
                retVal = Integer.parseInt(valueString);
                break;
            case BOOLEAN:
                retVal = Boolean.valueOf(valueString);
                break;
            case STRING:
                retVal = valueString;
                break;
            default:
                retVal = valueString;
                log.error(String.format("Unexpected index field type %s", type));
                break;
        }
        return retVal;
    }
}
