/*
 * Copyright 2009-2010 Nanjing RedOrange ltd (http://www.red-orange.cn)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package redora.generator;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;

import java.io.File;

import static java.io.File.separator;

/**
 * Generates persistence sources and resources. If any XML schema is located in
 * the xsd directory, also the JAXB sources will be generated.
 * 
 * @author Nanjing RedOrange (http://www.red-orange.cn)
 * @goal generate
 * @phase generate-sources
 */
public class GenerateMojo extends AbstractMojo {

    /**
     * The basePackage is always suffixed by this value. This will make all the
     * projects using this library more easy to use because the persistence
     * classes are all to be found under basePackage.PACKAGE_SUFFIX.
     */
    private static final String PACKAGE_SUFFIX = "rdo";
    public static final String GENERATION_TARGET = "redora-target";
    /**
     * Location of the file.
     * 
     * @parameter expression="${project.build.directory}"
     * @required
     */
    private File buildDir;
    /**
     * Package where the sources should be generated into.
     * 
     * @parameter expression=${persistulate.baseParameter}
     *            default="${project.groupId}.${project.artifactId}"
     * @required
     */
    private String basePackage;
    /**
     * @parameter expression="${project.artifactId}"
     * @required
     */
    private String artifactId;
    /**
     * Default language of the project.
     * 
     * @parameter expression=${persistulate.baseParameter} default="en"
     * @required
     */
    private String defaultLanguage;

    public void execute() throws MojoExecutionException {
        // Setting the directories
        File generateDirectory = new File(buildDir + separator + "generated-sources"
                + separator + GENERATION_TARGET);
        FileLocations where;
        try {
            where = new FileLocations(buildDir.getAbsolutePath(), artifactId);
        } catch (ModelGenerationException e) {
            throw new MojoExecutionException("File error", e);
        }

        getLog().info("Generating source for you. I will use the following directories and settings");
        getLog().info("- Java source: " + where.sourceDir);
        getLog().info("- Generated java source " + generateDirectory.getPath());
        getLog().info("- Base package " + basePackage + "(." + PACKAGE_SUFFIX + ')');

        ModelProcessor p = null;
        try {
            p = new ModelProcessor(where, basePackage + '.' + PACKAGE_SUFFIX, artifactId,
                    defaultLanguage);
            p.generate();
            getLog().info("All model sources are generated");
        } catch (ModelGenerationException e) {
            if (p == null) {
                throw new MojoExecutionException("Could not initialize the ModelProcessor", e);
            }
            throw new MojoExecutionException("Something went wrong generating your source, sorry.\r\n"
                    + p.dump(), e);
        }
    }
}
