/*
 * Copyright 2009-2010 Nanjing RedOrange ltd (http://www.red-orange.cn)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package redora.generator;

import org.apache.commons.io.filefilter.SuffixFileFilter;
import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.util.*;

import static java.io.File.separatorChar;

/**
 * Finds model files.
 * 
 * @author Nanjing RedOrange (www.red-orange.cn)
 * @see redora.generator.FileLocations
 */
public class ModelFileFinder {
    
    private final FileLocations where;
    private final String artifactId;

    public ModelFileFinder(@NotNull final FileLocations where, @NotNull final String artifactId) {
        this.where = where;
        this.artifactId = artifactId;
    }

    /**
     * Get all model documents file list
     * 
     * @return All "???.xml" files in the model directory (no sub directories)
     */
    @NotNull
    public Set<String> findModelFiles() {
        Set<String> retVal = new HashSet<String>();
        String[] model = new File(where.modelDir).list(new SuffixFileFilter(".xml"));
        if (model != null && model.length > 0)
            Collections.addAll(retVal, model);

        return retVal;
    }

    /**
     * Get all include documents file list
     * 
     * @return All "???.xml" files in the include directory (no sub directories)
     */
    @NotNull
    public Set<String> findIncludeFiles() {
        Set<String> retVal = new HashSet<String>();
        String[] model = new File(where.includeDir).list(new SuffixFileFilter(".xml"));

        if (model != null && model.length > 0)
            Collections.addAll(retVal, model);
        return retVal;
    }

    /**
     * Finds upgrade files in the upgrade directory.
     * 
     * @return Empty or alphabetically sorted list
     */
    @NotNull
    public SortedSet<String> upgradeFiles() {
        SortedSet<String> retVal = new TreeSet<String>();
        String[] upgrades = new File(where.upgradeDir).list(new SuffixFileFilter(".sql"));
        if (upgrades != null && upgrades.length > 0) {
            Collections.addAll(retVal, upgrades);
        }
        return retVal;
    }

    /**
     * Finds dumped model xml file in the ~/.redora directory.
     * 
     * @param modelName Name of the model including .xml
     * @return Path + file name of the model
     */
    @NotNull
    public File normalizedModelFile(@NotNull final String modelName) {
        File retVal = new File(where.redoraDir + separatorChar + artifactId, modelName + ".xml");
        if (!retVal.exists()) {
            retVal.mkdirs();
        }
        return retVal;
    }

    /**
     * Finds dumped allModel xml file in the local directory.
     *
     * @return Empty or directory
     */
    @NotNull
    public File allModelFiles() {
        File retVal = new File(where.redoraDir + separatorChar + artifactId, "allModels.xml");
        if (!retVal.exists()) {
            retVal.mkdirs();
        }
        return retVal;
    }
}
