/*
 * Copyright 2009-2010 Nanjing RedOrange ltd (http://www.red-orange.cn)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package redora.generator;

import org.jetbrains.annotations.NotNull;

import java.io.File;

/**
 * Archetype Template without content. Gives hints to the ModelProcessor what to
 * do with this template.
 * 
 * See also templates.xsd.
 * 
 * @author Nanjing RedOrange (http://www.red-orange.cn)
 */
public class Template {
    public enum Type {
        freemarker, xslt, copy
    }

    public enum Destination {
        source, target, redora
    }

    /** File type of the destination. */
    public enum DestinationType {
        css, html, java, properties, sql, xml
    }

    /** Data input for the template. Normally it is the Model files, but there are different options. */
    public enum Input {

        /** The standard model file. */
        Model,
        /** Bundle of all model files.*/
        AllModels,
        /** List of all the languaes used in the model files. */
        Language,
        /** All global scoped enums. */
        Enum
    }
    
    public final String name;
    public final Type type;
    public final Destination destination;

    public String packageSuffix;
    public String path;
    public DestinationType destinationType;
    public Input input;
    public String ignoreProjects;

    String destinationFileName;
    String templateFileName;

    final String templatesDir;

    public Template(@NotNull String name, @NotNull Type type, @NotNull Destination destination
            , @NotNull String templatesDir) {
        super();
        this.name = name;
        this.type = type;
        this.destination = destination;
        this.templatesDir = templatesDir;
    }

    public void setPackageSuffix(String packageSuffix) {
        this.packageSuffix = packageSuffix;
    }

    public void setPath(String path) {
        this.path = path;
    }

    public void setIgnoreProjects(String ignoreProjects) {
        this.ignoreProjects = ignoreProjects;
    }

    public void setDestinationType(DestinationType destinationType) {
        this.destinationType = destinationType;
    }

    public void setInput(Input input) {
        this.input = input;
    }

    public void setTemplateFileName(String templateFileName) {
        this.templateFileName = templateFileName;
    }

    public String getDestinationFileName(String objectName, String language, String classAttribute) {
        if (destinationFileName == null) {
            destinationFileName = name;
            if (objectName != null) {
                destinationFileName = destinationFileName.replace(
                        "${objectName}", objectName);
            }
            if (language != null) {
                destinationFileName = destinationFileName.replace(
                        "${language}", language);
            }
            if (classAttribute != null) {
                destinationFileName = destinationFileName.replace(
                        "${classAttribute}", classAttribute);
            }

            if (destinationType != null) {
                destinationFileName = destinationFileName + '.'
                        + destinationType;
            }
        }
        return destinationFileName;
    }

    public String getAbsolutePath() {
        return templatesDir + File.separatorChar + getTemplateFileName();
    }

    @NotNull
    public String getTemplateFileName() {
        if (templateFileName == null) {
            templateFileName = name.replace("${objectName}", "Pojo");

            if (type == Type.freemarker) {
                templateFileName = templateFileName + ".ftl";
            } else if (type == Type.xslt) {
                templateFileName = templateFileName + ".xsl";
            }
        }
        return templateFileName;
    }

    @Override
    public String toString() {
        return getTemplateFileName();
    }
}
