package net.saliman.gradle.plugin.cobertura

import org.gradle.api.DefaultTask
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.TaskAction

/**
 * This task copies the input datafile, generated by the instrument task, to
 * the output datafile, used to generate the coverage report.
 * <p>
 * This task is necessary because tests change the datafile as they run, so we
 * can't use a single file.  We need an input file that only changes when we
 * instrument so that we can detect when the instrument task is up to date, and
 * an output file that is used during testing and reporting.  We need that file
 * to only contain data from a single test run, so we need to make a new copy
 * of the output file every time.
 * <p>
 * Because we need a new file every time, there are no inputs or outputs for
 * this task, but the plugin is smart enough to only run this task if the user
 * wants a coverage report in the build.
 */
class CopyDatafileTask extends DefaultTask {
	static final String NAME = 'copyCoberturaDatafile'

	@Internal
	CoberturaExtension configuration

	/**
	 * Copy the input datafile to the output datafile.  The datafile is a binary
	 * file, so we use streams instead of a text based approach.
	 */
	@TaskAction
	def copyFile() {
		project.logger.info("${path} - Copying the datafile...")

		def input = configuration.coverageInputDatafile.newInputStream()
		def output = configuration.coverageOutputDatafile.newOutputStream()
		output << input
		input.close()
		output.close()
	}
}
