package net.serenitybdd.core.webdriver.driverproviders;

import net.serenitybdd.model.buildinfo.DriverCapabilityRecord;
import net.serenitybdd.core.di.SerenityInfrastructure;
import net.thucydides.core.fixtureservices.FixtureProviderService;
import net.thucydides.core.steps.StepEventBus;
import net.thucydides.core.steps.TestContext;
import net.thucydides.model.util.EnvironmentVariables;
import net.thucydides.core.webdriver.capabilities.W3CCapabilities;
import net.thucydides.core.webdriver.stubs.WebDriverStub;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.chrome.ChromeDriverService;
import org.openqa.selenium.chrome.ChromeOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Create a new instance of a ChromeDriver driver, using the configuration options defined in serenity.conf
 * and/or in the custom fixture classes.
 */
public class ChromeDriverProvider extends DownloadableDriverProvider implements DriverProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(ChromeDriverProvider.class);

    private final DriverCapabilityRecord driverProperties;

    private final FixtureProviderService fixtureProviderService;

    public ChromeDriverProvider(FixtureProviderService fixtureProviderService) {
        this.fixtureProviderService = fixtureProviderService;
        this.driverProperties = SerenityInfrastructure.getDriverCapabilityRecord();
    }

    @Override
    public WebDriver newInstance(String options, EnvironmentVariables environmentVariables) {
        // If webdriver calls are suspended no need to create a new driver
        if (StepEventBus.getParallelEventBus().webdriverCallsAreSuspended()) {
            return new WebDriverStub();
        }
        // Download the driver using WebDriverManager if required
//        downloadDriverIfRequired("chrome", environmentVariables);
        //
        // Update the chromedriver binary path if necessary
        //
        UpdateDriverEnvironmentProperty.forDriverProperty(ChromeDriverService.CHROME_DRIVER_EXE_PROPERTY);
        //
        // Load the ChromeDriver capabilities from the serenity.conf file
        //
        ChromeOptions chromeOptions = W3CCapabilities.definedIn(environmentVariables).withPrefix("webdriver.capabilities").chromeOptions();
        //
        // Check for extended classes to add extra ChromeOptions configuration
        //
        ChromeOptions enhancedOptions = ConfigureChromiumOptions.from(environmentVariables).into(chromeOptions);
        EnhanceCapabilitiesWithFixtures.using(fixtureProviderService).into(enhancedOptions);
        //
        // Record browser and platform
        //
        TestContext.forTheCurrentTest().recordBrowserConfiguration(enhancedOptions);
        TestContext.forTheCurrentTest().recordCurrentPlatform();
        //
        // Add any arguments passed from the test itself
        //
        enhancedOptions.addArguments(argumentsIn(options));
        //
        // Record the driver capabilities for reporting
        //
        driverProperties.registerCapabilities("chrome", capabilitiesToProperties(enhancedOptions));

        LOGGER.info("Starting Chrome driver instance with capabilities:");
        LOGGER.info(enhancedOptions.toString());

        return new ChromeDriver(enhancedOptions);
    }
}
