/*
 * Decompiled with CFR 0.152.
 */
package net.serenitybdd.plugins.jira;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import net.serenitybdd.plugins.jira.TestResultTally;
import net.serenitybdd.plugins.jira.domain.IssueComment;
import net.serenitybdd.plugins.jira.model.IssueTracker;
import net.serenitybdd.plugins.jira.model.NamedTestResult;
import net.serenitybdd.plugins.jira.model.TestResultComment;
import net.serenitybdd.plugins.jira.service.JIRAConfiguration;
import net.serenitybdd.plugins.jira.service.JIRAInfrastructure;
import net.serenitybdd.plugins.jira.service.NoSuchIssueException;
import net.serenitybdd.plugins.jira.workflow.Workflow;
import net.serenitybdd.plugins.jira.workflow.WorkflowLoader;
import net.thucydides.model.ThucydidesSystemProperty;
import net.thucydides.model.domain.TestOutcomeSummary;
import net.thucydides.model.domain.TestResult;
import net.thucydides.model.util.EnvironmentVariables;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JiraUpdater {
    static int DEFAULT_MAX_THREADS = 4;
    private final IssueTracker issueTracker;
    private final EnvironmentVariables environmentVariables;
    private static final Logger LOGGER = LoggerFactory.getLogger(JiraUpdater.class);
    private final String projectPrefix;
    private final Workflow workflow;
    private final JIRAConfiguration configuration;

    public JiraUpdater(IssueTracker issueTracker, EnvironmentVariables environmentVariables, WorkflowLoader loader) {
        this.issueTracker = issueTracker;
        this.environmentVariables = environmentVariables;
        this.configuration = JIRAInfrastructure.getConfiguration();
        this.workflow = loader.load();
        this.projectPrefix = environmentVariables.getProperty(ThucydidesSystemProperty.JIRA_PROJECT.getPropertyName());
        this.logStatus(environmentVariables);
    }

    private void logStatus(EnvironmentVariables environmentVariables) {
        String jiraUrl = environmentVariables.getProperty(ThucydidesSystemProperty.JIRA_URL.getPropertyName());
        String reportUrl = ThucydidesSystemProperty.SERENITY_PUBLIC_URL.from(environmentVariables, "");
        LOGGER.debug("JIRA LISTENER STATUS");
        LOGGER.debug("JIRA URL: {} ", (Object)jiraUrl);
        LOGGER.debug("REPORT URL: {} ", (Object)reportUrl);
        LOGGER.debug("WORKFLOW ACTIVE: {} ", (Object)this.workflow.isActive());
    }

    public void updateIssueStatus(Set<String> issues, TestResultTally<TestOutcomeSummary> resultTally) {
        issues.parallelStream().forEach(issue -> this.updateIssue((String)issue, resultTally.getTestOutcomesForIssue((String)issue)));
    }

    public boolean shouldUpdateIssues() {
        if (this.dryRun()) {
            return false;
        }
        String jiraUrl = this.environmentVariables.getProperty(ThucydidesSystemProperty.JIRA_URL.getPropertyName());
        String reportUrl = ThucydidesSystemProperty.SERENITY_PUBLIC_URL.from(this.environmentVariables, "");
        if (this.workflow.isActive()) {
            LOGGER.debug("WORKFLOW TRANSITIONS: {}", (Object)this.workflow.getTransitions());
        }
        return !StringUtils.isEmpty((CharSequence)jiraUrl) && !StringUtils.isEmpty((CharSequence)reportUrl);
    }

    private void updateIssue(String issueId, List<TestOutcomeSummary> testOutcomes) {
        try {
            TestResultComment testResultComment = this.newOrUpdatedCommentFor(issueId, testOutcomes);
            if (this.getWorkflow().isActive() && this.shouldUpdateWorkflow()) {
                this.updateIssueStatusFor(issueId, testResultComment.getOverallResult());
            }
        }
        catch (NoSuchIssueException e) {
            LOGGER.error("No JIRA issue found with ID {}", (Object)issueId);
        }
    }

    private void updateIssueStatusFor(String issueId, TestResult testResult) {
        LOGGER.info("Updating status for issue {} with test result {}", (Object)issueId, (Object)testResult);
        String currentStatus = this.issueTracker.getStatusFor(issueId);
        LOGGER.info("Issue {} currently has status '{}'", (Object)issueId, (Object)currentStatus);
        List<String> transitions = this.getWorkflow().getTransitions().forTestResult(testResult).whenIssueIs(currentStatus);
        LOGGER.info("Found transitions {} for issue {}", transitions, (Object)issueId);
        for (String transition : transitions) {
            this.issueTracker.doTransition(issueId, transition);
        }
    }

    private List<NamedTestResult> namedTestResultsFrom(List<TestOutcomeSummary> testOutcomes) {
        return testOutcomes.stream().map(testOutcome -> new NamedTestResult(testOutcome.getTitle(), testOutcome.getTestResult())).collect(Collectors.toList());
    }

    private TestResultComment newOrUpdatedCommentFor(String issueId, List<TestOutcomeSummary> testOutcomes) {
        TestResultComment testResultComment;
        LOGGER.info("Updating comments for issue {}", (Object)issueId);
        LOGGER.info("WIKI Rendering activated: {}", (Object)this.isWikiRenderedActive());
        List<IssueComment> comments = this.issueTracker.getCommentsFor(issueId);
        Optional<IssueComment> existingComment = this.findExistingSerenityCommentIn(comments);
        String testRunNumber = this.environmentVariables.getProperty("build.id");
        List<NamedTestResult> newTestResults = this.namedTestResultsFrom(testOutcomes);
        if (!existingComment.isPresent() || this.createNewCommentForEachUpdate()) {
            testResultComment = TestResultComment.comment(this.isWikiRenderedActive()).withResults(this.namedTestResultsFrom(testOutcomes)).withReportUrl(this.linkToReport(testOutcomes)).forTestsExecutedAt(LocalDateTime.now()).withTestRun(testRunNumber).asComment();
            if (!this.dryRun()) {
                this.issueTracker.addComment(issueId, testResultComment.asText());
            }
        } else {
            testResultComment = TestResultComment.fromText(existingComment.get().getBody()).withWikiRendering(this.isWikiRenderedActive()).withUpdatedTestResults(newTestResults).withUpdatedReportUrl(this.linkToReport(testOutcomes)).forTestsExecutedAt(LocalDateTime.now()).withUpdatedTestRunNumber(testRunNumber);
            IssueComment updatedComment = existingComment.get().withText(testResultComment.asText());
            if (!this.dryRun()) {
                this.issueTracker.updateComment(issueId, updatedComment);
            }
        }
        return testResultComment;
    }

    private boolean createNewCommentForEachUpdate() {
        return this.environmentVariables.getPropertyAsBoolean("serenity.jira.always.create.new.comment", false);
    }

    private Optional<IssueComment> findExistingSerenityCommentIn(List<IssueComment> comments) {
        return comments.stream().filter(comment -> comment.getBody().contains("Serenity BDD Automated Acceptance Tests")).findFirst();
    }

    private void logIssueTracking(String issueId) {
        if (this.dryRun()) {
            LOGGER.info("--- DRY RUN ONLY: JIRA WILL NOT BE UPDATED ---");
        }
        LOGGER.info("Updating JIRA issue: " + issueId);
        LOGGER.info("JIRA server: " + this.issueTracker.toString());
    }

    private boolean dryRun() {
        return Boolean.parseBoolean(this.environmentVariables.getProperty("serenity.skip.jira.updates"));
    }

    private String linkToReport(List<TestOutcomeSummary> testOutcomes) {
        TestOutcomeSummary firstTestOutcome = testOutcomes.get(0);
        String reportUrl = ThucydidesSystemProperty.SERENITY_PUBLIC_URL.from(this.environmentVariables, "");
        String reportName = firstTestOutcome.getReportName() + ".html";
        return this.formatTestResultsLink(reportUrl, reportName);
    }

    private String formatTestResultsLink(String reportUrl, String reportName) {
        return reportUrl + "/" + reportName;
    }

    private boolean isWikiRenderedActive() {
        return this.configuration.isWikiRenderedActive();
    }

    public List<String> getPrefixedIssuesWithoutHashes(TestOutcomeSummary result) {
        return this.addPrefixesIfRequired(this.stripInitialHashesFrom(this.issueReferencesIn(result)));
    }

    private List<String> addPrefixesIfRequired(List<String> issueNumbers) {
        return issueNumbers.stream().map(this::toIssueNumbersWithPrefixes).collect(Collectors.toList());
    }

    private List<String> issueReferencesIn(TestOutcomeSummary result) {
        return result.getIssues();
    }

    private String toIssueNumbersWithPrefixes(String issueNumber) {
        if (StringUtils.isEmpty((CharSequence)this.projectPrefix)) {
            return issueNumber;
        }
        if (issueNumber.startsWith(this.projectPrefix)) {
            return issueNumber;
        }
        return this.projectPrefix + "-" + issueNumber;
    }

    private List<String> stripInitialHashesFrom(List<String> issueNumbers) {
        return issueNumbers.stream().map(this::toIssueNumbersWithoutHashes).collect(Collectors.toList());
    }

    public int hashCode() {
        return super.hashCode();
    }

    private String toIssueNumbersWithoutHashes(String issueNumber) {
        if (issueNumber.startsWith("#")) {
            return issueNumber.substring(1);
        }
        return issueNumber;
    }

    private int getMaxJobs() {
        return this.environmentVariables.getPropertyAsInteger("jira.max.threads", Integer.valueOf(DEFAULT_MAX_THREADS));
    }

    protected Workflow getWorkflow() {
        return this.workflow;
    }

    protected boolean shouldUpdateWorkflow() {
        return Boolean.valueOf(this.environmentVariables.getProperty("serenity.jira.workflow.active"));
    }

    public IssueTracker getIssueTracker() {
        return this.issueTracker;
    }
}

