package net.sf.cuf.fw2;

import net.sf.cuf.appevent.AppEvent;
import net.sf.cuf.fw.Dc;

import java.util.Map;

/**
 * A CloseDialogEvent signals that a dialog should be closed.<br>
 * For dialogs following the SimpleLifeCycle interface this will
 * trigger a call to the doPassivate() method.
 * The dialog is referenced either by a Dc object or a symbolic name.
 * The close event contains also the information if the dialog should
 * be disposed, if the dialog was closed and the result of the doPassivate()
 * call.
 * If the dialog implements the SimpleLifeCycle interface, the
 * CloseDialogEvent should only be fired if the dialog is active.
 * A dialog may choose this event because he has no direct or complete
 * control about its frame/MDI frame, and therefore delegates this
 * task to someone else.
 */
public class CloseDialogEvent extends AppEvent
{
    /** either a String or a Dc, never null */
    private Object              mDialog;
    /** map containing the result, never null */
    private Map<String, Object> mResult;
    /** true if the close should also trigger a dispose of the dialog, the default is false */
    private boolean             mDispose;
    /** true if the close should skip asking the Dc if it can passivate, the default is false */
    private boolean             mForceClose;
    /** true if the dialog was closed */
    private boolean             mWasClosed;

    /**
     * Creates a new close dialog.
     * @param pSource the sender
     * @param pDialogName the dialog name
     */
    public CloseDialogEvent(final Object pSource, final String pDialogName)
    {
        this(pSource, (Object)pDialogName, false, false);
    }

    /**
     * Creates a new close dialog.
     * @param pSource the sender
     * @param pDialogName the dialog name
     * @param pDispose flag if we should also dispose the dialog
     */
    public CloseDialogEvent(final Object pSource, final String pDialogName, final boolean pDispose)
    {
        this(pSource, (Object)pDialogName, pDispose, false);
    }

    /**
     * Creates a new close dialog.
     * @param pSource the sender
     * @param pDialogName the dialog name
     * @param pDispose flag if we should also dispose the dialog
     * @param pForceClose flag if we should skip asking the Dc if it can passivate
     */
    public CloseDialogEvent(final Object pSource, final String pDialogName,
                            final boolean pDispose, final boolean pForceClose)
    {
        this(pSource, (Object)pDialogName, pDispose, pForceClose);
    }

    /**
     * Creates a new close dialog.
     * @param pSource the sender
     * @param pDialog the dialog object
     */
    public CloseDialogEvent(final Object pSource, final Dc pDialog)
    {
        this(pSource, pDialog, false, false);
    }

    /**
     * Creates a new close dialog.
     * @param pSource the sender
     * @param pDialog the dialog object
     * @param pDispose flag if we should also dispose the dialog
     */
    public CloseDialogEvent(final Object pSource, final Dc pDialog, final boolean pDispose)
    {
        this(pSource, pDialog, pDispose, false);
    }

    /**
     * Creates a new close dialog.
     * @param pSource the sender
     * @param pDialog the dialog object
     * @param pDispose flag if we should also dispose the dialog
     * @param pForceClose flag if we should skip asking the Dc if it can passivate
     */
    public CloseDialogEvent(final Object pSource, final Dc pDialog,
                            final boolean pDispose, final boolean pForceClose)
    {
        this(pSource, (Object)pDialog, pDispose, pForceClose);
    }

    /**
     * Creates a new close dialog.
     * @param pSource the sender
     * @param pDialog the dialog string or object we should close
     * @param pDispose flag if we should also dispose the dialog
     * @param pForceClose flag if we should skip asking the Dc if it can passivate
     */
    private CloseDialogEvent(final Object pSource, final Object pDialog,
                             final boolean pDispose, final boolean pForceClose)
    {
        super(pSource);

        if (pDialog==null)
            throw new IllegalArgumentException("dialog (name) must not be null");
        mDialog    = pDialog;
        mDispose   = pDispose;
        mForceClose= pForceClose;
        mWasClosed = false;
    }

    /**
     * Checks if the close event contains a dialog id and not a Dc object.
     * @return true if the event was generated with a dialog id
     */
    public boolean hasDialogId()
    {
        return (mDialog instanceof String);
    }

    /**
     * The dialog id of this close event.
     * @return dialog id, never null
     * @throws IllegalStateException if this event contains no dialog string
     */
    public String getDialogId()
    {
        if (!hasDialogId())
        {
            throw new IllegalStateException("This CloseDialogEvent was created with no String but the Dc "+mDialog);
        }
        return (String)mDialog;
    }

    /**
     * The dialog Dc of this close event.
     * @return Dc object, never null
     * @throws IllegalStateException if this event contains no Dc object
     */
    public Dc getDialog()
    {
        if (hasDialogId())
        {
            throw new IllegalStateException("This CloseDialogEvent was created with no Dc but the String "+mDialog);
        }
        return (Dc)mDialog;
    }

    /**
     * Returns if the dialog should be disposed after de-activating.
     * @return true for dispose
     */
    public boolean isDispose()
    {
        return mDispose;
    }

    /**
     * Returns if the close is forced.
     * @return true if we should not ask the Dc if it can passivate
     */
    public boolean isForceClose()
    {
        return mForceClose;
    }

    /**
     * Return the result of a close.
     * @return the result, never null
     */
    public Map<String, Object>  getResult()
    {
        return mResult;
    }

    /**
     * Set the result of a close.
     * @param pResult the result, must not be null
     */
    public void setResult(final Map<String, Object> pResult)
    {
        if (pResult==null)
            throw new IllegalArgumentException("dialog result must not be null");

        mResult= pResult;
    }

    /**
     * Was the dialog really closed by this event?
     * @return true if the dialog was closed
     */
    public boolean getWasClosed()
    {
        return mWasClosed;
    }

    /**
     * Set the flag if the dialog was really closed.
     * @param pWasClosed true if the dialog was closed
     */
    public void setWasClosed(final boolean pWasClosed)
    {
        mWasClosed= pWasClosed;
    }
}
