package net.sf.cuf.fw2;

import java.util.Properties;
import java.io.IOException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;

/**
 * Utility for reading and writing properties to users home directory.
 *
 * @author  J&ouml;rg Eichhorn, sd&amp;m AG
 */
public class UserProperties
{
    /** Fileseparator on this particular platform (e.g. "/" on Unix, "\" on Windows) */
    private static final String FILE_SEPARATOR = System.getProperty("file.separator", "/");

    /** Filename to use for userProps, if no one given as parameter */
    protected static final String DEFAULT_USER_PROPERTIES_FILENAME = "user.properties";

    /**
     * We provide only private helper methods.
     */
    private UserProperties()
    {
    }

    /**
     * Read user properties out of a file in users home-directory.
     *
     * @param pDefaultProps the default properties name (will read by the class loader), may be <code>null</code>
     * @param pPath optional path underneath users home-directory, may be <code>null</code>
     * @param pFle optional filename for user properties. If <code>null</code>, a default filename will be used.
     * @return users properties
     * @throws IOException if we have IO problems
     */
    public static Properties read(final String pDefaultProps, final String pPath, final String pFle)
        throws IOException
    {
        Properties  userProps= new Properties();

        // create user properties with default, we use a ClassLoader so we can read
        // from a jar File
        if (pDefaultProps!=null)
        {
            ClassLoader loader   = Thread.currentThread().getContextClassLoader();
            InputStream defaultIn= loader.getResourceAsStream(pDefaultProps);
            if (defaultIn!=null)
            {
                userProps.load(defaultIn);
                defaultIn.close();
            }
        }

        // use pFle as name if given, use default otherwise
        String fileName = pFle != null ? pFle : DEFAULT_USER_PROPERTIES_FILENAME;

        // construct complete filename for userProps
        StringBuilder userPropsFilename = new StringBuilder(System.getProperty("user.home", "."));
        userPropsFilename.append(FILE_SEPARATOR);
        if (pPath != null)
        {
            userPropsFilename.append(pPath);
            userPropsFilename.append(FILE_SEPARATOR);
        }
        userPropsFilename.append(fileName);

        // create fileobject to work with
        File userPropsFile = new File(userPropsFilename.toString());

        try
        {
            // now load properties from pFle
            FileInputStream userIn= new FileInputStream(userPropsFile);
            userProps.load(userIn);
            userIn.close();
        }
        catch (FileNotFoundException ex)
        {
            // do nothing ...
        }

        return userProps;
    }

    /**
     * Write user properties into a file in users home-directory.
     *
     * @param pUserProps the user properties to write into a file
     * @param pPath optional path underneath users home-directory, may be <code>null</code>
     * @param pFile optional filename for user properties. If <code>null</code>, a default filename will be used.
     * @throws IOException if we have IO problems
     */
    public static void write(final Properties pUserProps, final String pPath, final String pFile) throws IOException
    {
        // use pFile as name if given, use default otherwise
        String fileName = pFile != null ? pFile : DEFAULT_USER_PROPERTIES_FILENAME;

        // construct complete filename for userProps
        StringBuilder userPropsFilename = new StringBuilder(System.getProperty("user.home", "."));
        if (pPath != null)
        {
            userPropsFilename.append(FILE_SEPARATOR);
            userPropsFilename.append(pPath);
        }
        userPropsFilename.append(FILE_SEPARATOR);
        userPropsFilename.append(fileName);

        // create fileobject to work with
        File userPropsFile = new File(userPropsFilename.toString());

        // get fileobject containing pPath for userProps
        File userPropsPath = userPropsFile.getParentFile();

        if (userPropsPath != null && !userPropsPath.exists())
        {
            // try to create directories for userPropsFile
            //noinspection ResultOfMethodCallIgnored
            userPropsPath.mkdirs();
        }

        FileOutputStream out = new FileOutputStream(userPropsFile);
        if (pUserProps != null)
        {
            pUserProps.store(out, System.getProperty("user.name", ""));
        }
        out.close();
    }
}
