package net.sf.cuf.singleapp;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;

/**
 * This class allows a simple, one time socket communication between instances of the same application.
 * This is intended for:
 * <UL>
 * <LI>Ensuring only instance of the application is run at the same time.</LI>
 * <LI>Passing a data object from a newly started instance to the first instance started 
 * so that the first instance may do the work while the newly started instance quits.</LI>
 * </UL>
 * It is not intended for:
 * <UL>
 * <LI>Continuous communication between processes.
 * <LI>A robust or fast server implementation.
 * <LI>Making ice cream and other things that may be on your mind.
 * </UL>
 * <P>
 * To use:
 * <OL>
 * <LI>Create an instance with {@link #StartupSocketCommunicator(String, int, ContactHandler, Serializable)}.</LI>
 * <LI>Determine which mode we entered by evaluating {@link #isServerMode()}.</LI>
 * <LI>If you are in server mode (first instance of the application), you will receive calls on {@link ContactHandler#handleContactFromOtherInstance(Serializable)}.
 * If you are in client mode (not first instance of the application), you have to check if the communication was successful ({@link #getException()} is null) and
 * then get the result that was returned from the server by calling {@link #getClientResult()}.
 * </OL>
 * See {@link #StartupSocketCommunicator(String, int, ContactHandler, Serializable)} for further details.
 * <P>
 * This implementation is meant to be thread safe and may be used from any thread.
 */
public class StartupSocketCommunicator
{
    
    /**
     * some long value that we use to determine if we are communicating with a counter part implementation.
     */
    private static final long MAGIC_NUMBER = 8920712192410910312L;

    /**
     * a string that is sent from the server to the client to acknowledge the correct receipt of a message or data
     */
    private static final String OK_STRING = "OK";
    
    /**
     * the prefix of a string that is sent from the server to the client to notify the client of an error.
     * Following the prefix is a error message that can be used for debug purposes.
     */
    private static final String NOK_STRING_PREFIX = "NOK:";
    
    /**
     * An id that is compared when two instances create contact.
     * If the application id does not match, we assume that a different application that uses the same technique
     * is configured to use the same port.
     */
    private String mApplicationId;
    
    /**
     * true if we have entered the server mode, read only after the constructor
     */
    private boolean mServerMode;

    /**
     * true if the server has been stopped - only meaningful if {@link #mServerMode} is true
     */
    private boolean mServerStopped;
    
    /**
     * true if a stop of the server has been requested - only meaningful if {@link #mServerMode} is true
     */
    private boolean mServerStopRequested;
    
    /**
     * the shutdown hook we register in server mode, null otherwise.
     */
    private Thread mShutdownHook;

    /**
     * the thread we use for listening to the server socket
     */
    private ServerCommunicatorSocketThread mServerSocketThread;
    
    /**
     * the socket that we listen to (server mode only), read only after the constructor
     */
    private ServerSocket mServerSocket;
    
    /**
     * the contact handler that is called by other instances, must not be null, read only after the constructor
     */
    private ContactHandler mContactHandler;

    /**
     * the last exception encountered (both in client and server mode)
     */
    private Throwable mException;
    
    /**
     * the result we obtained from the server when we were running in client mode.
     */
    private Serializable mClientResult;
    
    /**
     * Initiates the communication for this application instance. After the object has been created you can
     * query the result of the communication attempt.
     * <P>
     * First, we attempt to register a server socket on the given port.
     * If the socket is successfully registered, we are the first instance of the application and 
     * act as server to other instances. Otherwise we are an additional instance of the application and
     * act as client.
     * <P>
     * The following situations may happen:
     * <UL>
     * <LI><B>First instance - Server mode ({@link #isServerMode()} is true)</B><BR>
     * 
     * If we are the first to register the socket, other application instances will call the 
     * given {@link ContactHandler}. The calls will happen in arbitrary threads.
     * <P>
     * If an exception in server mode occurs, you can access it with the {@link #getException()}.
     * If that value is null in server mode, then the server is running (or ran) without problem.
     * <P>
     * If you want to stop the server explicitly, call {@link #stopServer()}. Otherwise the server will be stopped
     * when the shutdown handlers are executed (see {@link Runtime#addShutdownHook(Thread)}).
     * 
     * <LI><B>Not first instance - Client mode ({@link #isServerMode()} is true)</B><BR>
     * 
     * If we can not register the socket, we try to contact the application instance that has registered the socket.
     * The protocol that is used is described in the private inner class ClientContactHandlerThread.
     * <P>
     * If an exception in client mode occurs, you can access it with the {@link #getException()} - it will not be thrown!.
     * If that value is null in client mode, then the request ran without problems and we have a result
     * stored in {@link #getClientResult()}.
     * </UL>
     * <P>
     * Note that the various threads started by the {@link StartupSocketCommunicator} are all daemon threads, i.e. they
     * will not stop the application from exiting, even if a client is already connected. 
     * The client will notice that by an exception in its communication attempts.
     * 
     * @param pApplicationId an id that is compared when two instances create contact.
     * If the application id does not match, we assume that a different application that uses the same technique
     * is configured to use the same port.
     * 
     * @param pPort the port to use for communication
     * 
     * @param pContactHandler the contact handler that is called by other instances, must not be null.
     * This handler is only used when this application instance is the one that
     * gets the port and thus acts as a server. Otherwise it is ignored.
     * 
     * @param pData the data to pass to the first application instance if we are not the first.
     * This may be any {@link Serializable} but you should certain that
     * either both application instances are using the same code base or that the objects have 
     * custom serialization mechanisms that allow communication across different versions.
     */
    public StartupSocketCommunicator( final String pApplicationId, final int pPort, final ContactHandler pContactHandler, final Serializable pData)
    {
        if (pContactHandler==null)
        {
            throw new IllegalArgumentException("pContactHandler must not be null");
        }
        mApplicationId = pApplicationId;
        mContactHandler = pContactHandler;
        try
        {
            mServerSocket = new ServerSocket(pPort);
        } 
        catch (IOException e)
        {
            // we could not create the socket
            mServerSocket = null;
        }
        if (mServerSocket==null)
        {
            runClientMode( pPort, pData);
        }
        else
        {
            initServerMode();
        }
    }

    /**
     * initializes the client mode and attempts contact with the application instance that
     * owns the port. The result of the contact will be written to member variables.
     * See {@link #StartupSocketCommunicator(String, int, ContactHandler, Serializable)}
     * for details.
     * <P>
     * This method will never throw an exception. In case of an error, the exception is
     * stored in {@link #mException}.
     * 
     * @param pPort the port to use for communication
     * @param pData the data to pass to the first application instance
     */
    private void runClientMode(final int pPort, final Serializable pData)
    {
        mServerMode = false;
        mServerStopRequested = true; // well - we're not running anyway
        mServerStopped = true;
        Socket clientSocket = null;
        ObjectOutputStream out = null;
        ObjectInputStream in = null;
        try
        {
            // create socket
            clientSocket = new Socket(InetAddress.getLocalHost().getHostAddress(), pPort);
//            InputStream testIn = clientSocket.getInputStream();
//            OutputStream testOut = clientSocket.getOutputStream();
//            for(int i=0; i<10; i++) testOut.write(i);
//            testOut.close();
//            testIn.close();
//            if (true) return;
            // get streams
            out = new ObjectOutputStream( clientSocket.getOutputStream());
            in = new ObjectInputStream( clientSocket.getInputStream());
            // call the server
            handleConversationForClient(out, in, pData);
            // close streams and socket
            in.close();
            out.close();
            clientSocket.close();
        }
        catch (Throwable ex)
        {
            setException(ex);
            // try to close everything
            if (in!=null)
            {
                try
                {
                    in.close();
                } 
                catch (IOException e)
                {
                    // ignore this, it is a best effort
                }
            }
            if (out!=null)
            {
                try
                {
                    out.close();
                }
                catch (IOException e)
                {
                    // ignore this, it is a best effort
                }
            }
            if (clientSocket!=null && !clientSocket.isClosed())
            {
                try
                {
                    clientSocket.close();
                } 
                catch (IOException e)
                {
                    // ignore this, it is a best effort
                }
            }
        }
    }

    /**
     * handles the client side of the conversation between the client and the server as described in ClientContactHandlerThread.
     * This corresponds to  ClientContactHandlerThread#handleConversationForServer(ObjectInputStream, ObjectOutputStream).
     * 
     * @param pOut the output stream to the server
     * @param pIn the input stream from the server
     * @param pData the data object to send to the server
     * @throws RuntimeException if the protocol is not met
     * @throws IOException if we have a technical communication error
     * @throws ClassNotFoundException if we have a technical communication error
     */
    private void handleConversationForClient(final ObjectOutputStream pOut, final ObjectInputStream pIn, final Serializable pData) throws IOException, ClassNotFoundException
    {
        // write magic number
        pOut.writeLong(MAGIC_NUMBER);
        pOut.flush();
        // check for OK
        checkOkResponse(pIn);
        // send application id
        pOut.writeObject(mApplicationId);
        pOut.flush();
        // check for OK
        checkOkResponse(pIn);
        // write the data object
        pOut.writeObject(pData);
        pOut.flush();
        // check for OK
        checkOkResponse(pIn);
        // check for OK (again - this happens after the server handled the query)
        checkOkResponse(pIn);
        // read the result
        mClientResult = (Serializable)pIn.readObject();
        // we're done - the streams will be closed by our caller
    }
    
    /**
     * Reads an object from the stream and checks if it is the String {@link #OK_STRING}.
     * If it is {@link #OK_STRING}, we continue, if not, we throw an exception.
     * @param pIn the input stream to read from
     * @throws RuntimeException if the next object read from the stream is not the {@link #OK_STRING}
     * @throws IOException if we have a technical communication error
     * @throws ClassNotFoundException if we have a technical communication error
     */
    private void checkOkResponse(final ObjectInputStream pIn) throws IOException, ClassNotFoundException
    {
        Object obj = pIn.readObject();
        if (obj instanceof String)
        {
            if (OK_STRING.equals( obj))
            {
                // ok
                return;
            }
        }
        throw new RuntimeException("Error in communication to server: "+obj);
    }

    /**
     * initializes the server mode, calls from other instances will be forwarded to the {@link ContactHandler}.
     * For details see {@link #StartupSocketCommunicator(String, int, ContactHandler, Serializable)}.
     */
    private void initServerMode()
    {
        mServerMode = true;
        mServerStopRequested = false;
        mServerStopped = false;
        mServerSocketThread = new ServerCommunicatorSocketThread();
        mServerSocketThread.start();
    }

    /**
     * @return true if we have entered the server mode, see {@link #StartupSocketCommunicator(String, int, ContactHandler, Serializable)}
     * for details
     */
    public synchronized boolean isServerMode()
    {
        return mServerMode;
    }
    
    /**
     * @return true if the server has been stopped, makes only sense if {@link #isServerMode()} is true
     */
    public synchronized boolean isServerStopped()
    {
        return mServerStopped;
    }
    
    /**
     * @return the last exception (if any) that occurred
     */
    public synchronized Throwable getException()
    {
        return mException;
    }
    
    /**
     * @return the result returned from the server when it called {@link ContactHandler#handleContactFromOtherInstance(Serializable)}
     * with our data (only meaningfull if {@link #isServerMode()} is false).
     */
    public Serializable getClientResult()
    {
        return mClientResult;
    }
    
    /**
     * sets the exception
     * @param pException the exception
     */
    private synchronized void setException(final Throwable pException)
    {
        mException = pException;
    }
    
    /**
     * stops the server and frees the port. If we are not in server mode, we ignore this call.
     * When this method returns, the server may not be stopped, yet, but it will be on its way.
     */
    public void stopServer()
    {
        if (!mServerMode)
        {
            return;
        }
        synchronized (this)
        {
            if (mServerStopped)
            {
                return;
            }
            if (mServerStopRequested)
            {
                return;
            }
            mServerStopRequested = true;
        }
        // Ok, we're now the only thread in this piece of code (though the server thread is still running and others may access stuff.)
        // clean up shutdown hook (safe to use here since it is only accessed in the constructor)
        if (mShutdownHook!=null)
        {
            try
            {
                Runtime.getRuntime().removeShutdownHook(mShutdownHook);
            }
            catch (Throwable ex)
            {
                // ignore - it may happen when the VM-shutdown was already initiated
            }
            mShutdownHook = null;
        }
        // now "kill" the thread
        // This happens through closing the socket since the socket io blocks and does not use the InterruptedException.
        try
        {
            mServerSocket.close();
        } 
        catch (IOException e)
        {
            // ignore this exception, it is an effect of what we want to accomplish
        }
        // the mServerStopped will be set to true by the ServerCommunicatorSocketThread when it finishes 
    }

    /**
     * stops the server, frees the port and waits until that happens. If we are not in server mode, we ignore this call.
     * When this method returns, the server is stopped.
     * @throws InterruptedException if we get interrupted during sleep
     */
    public void waitServerStopped() throws InterruptedException
    {
        stopServer();
        while (!isServerStopped())
        {
            Thread.sleep(100);
        }
    }
    
    /**
     * the thread that drives the server side of the {@link StartupSocketCommunicator}
     */
    private class ServerCommunicatorSocketThread extends Thread
    {
        
        /**
         * Constructor, private to restrict access.
         */
        private ServerCommunicatorSocketThread()
        {
            setDaemon(true);
        }
        
        /**
         * Loop and accept contacts
         */
        public void run()
        {
            try
            {
                boolean stopRequested;
                synchronized (StartupSocketCommunicator.this)
                {
                    stopRequested = mServerStopRequested;
                }
                while (!stopRequested)
                {
                    Socket socket;
                    try
                    {
                        socket = mServerSocket.accept();
                    } 
                    catch (IOException e)
                    {
                        synchronized (StartupSocketCommunicator.this)
                        {
                            if (mServerStopRequested)
                            {
                                // that is the cause for the exception, ignore it and quit the server
                                return;
                            }
                            else
                            {
                                setException(e);
                                return;
                            }
                        }
                    }
                    if (socket!=null)
                    {
                        // ok, handle the caller
                        new ClientContactHandlerThread( socket).start();
                    }
                    
                    synchronized (StartupSocketCommunicator.this)
                    {
                        stopRequested = mServerStopRequested;
                    }
                }
            }
            finally
            {
                try
                {
                    mServerSocket.close();
                }
                catch (Throwable ex)
                {
                    // this is to be expected, but the cleanup is a good idea
                }
                synchronized (StartupSocketCommunicator.this)
                {
                    mServerStopped = true;
                }
            }
        }

    }
    
    /**
     * A thread that handles a socket contact from a client, created and started by {@link ServerCommunicatorSocketThread}.
     * <P>
     * The protocol used is as follows:
     * <OL>
     * <LI>Both sides use an object stream (corresponding {@link ObjectInputStream} and {@link ObjectOutputStream}).
     * <LI>The client sends the (long){@link StartupSocketCommunicator#MAGIC_NUMBER}.
     * <LI>The server sends OK (an {@link String} that is {@link StartupSocketCommunicator#OK_STRING}).
     * <LI>The client sends the application id
     * <LI>The server sends OK.
     * <LI>The client sends the data object (the parameter to {@link StartupSocketCommunicator#StartupSocketCommunicator(String, int, ContactHandler, Serializable)}).
     * <LI>The server sends OK. (This means 'ok, I got the request, I'm working on it'.) 
     * <LI>The server queries the {@link ContactHandler#handleContactFromOtherInstance(Serializable)}. If this
     * query returns without exception, the server sends OK. (This means 'ok, I processed the request, here follows the result'.)
     * If the query throws an exception, a NOK-String is sent and the commection terminated (as with any other error.)
     * <LI>The server sends the result from {@link ContactHandler#handleContactFromOtherInstance(Serializable)}.
     * </OL>
     * If an error occurs during this communication, the socket streams and connection will be closed. If possible, the server will send an error
     * message to the client starting with {@link StartupSocketCommunicator#NOK_STRING_PREFIX}. 
     * The error will be stored
     * in {@link StartupSocketCommunicator#getException()} (though it may be replaced by the next exception occurring on the server).
     */
    private class ClientContactHandlerThread extends Thread
    {
        /**
         * the socket that was opened
         */
        private Socket mSocket;
        /**
         * Constructor
         * @param pSocket the socket that was opened
         */
        private ClientContactHandlerThread( final Socket pSocket)
        {
            setDaemon(true);
            mSocket = pSocket;
        }
        
        /**
         * Handles the socket communication with the client
         */
        public void run()
        {
            ObjectInputStream in = null;
            ObjectOutputStream out = null;
            
            try
            {
//                InputStream intest = mSocket.getInputStream();
//                OutputStream outtest = mSocket.getOutputStream();
//                int read = 0;
//                while (read>=0)
//                {
//                    read = intest.read();
//                    System.out.println("In: "+read);
//                }
//                if (true) return;
                // get streams
                in = new ObjectInputStream( mSocket.getInputStream());
                out = new ObjectOutputStream( mSocket.getOutputStream());
                // do our stuff
                handleConversationForServer(in,out);
                // close streams and socket
                in.close();
                out.close();
                mSocket.close();
            }
            catch (Throwable ex)
            {
                setException(ex);
                // try to close everything
                if (in!=null)
                {
                    try
                    {
                        in.close();
                    } 
                    catch (IOException e)
                    {
                        // ignore this, it is a best effort
                    }
                }
                if (out!=null)
                {
                    try
                    {
                        out.close();
                    }
                    catch (IOException e)
                    {
                        // ignore this, it is a best effort
                    }
                }
                if (!mSocket.isClosed())
                {
                    try
                    {
                        mSocket.close();
                    } 
                    catch (IOException e)
                    {
                        // ignore this, it is a best effort
                    }
                }
            }
        }

        /**
         * handles the server side of the conversation between the client and the server as described in the private inner class ClientContactHandlerThread.
         * This corresponds to {@link StartupSocketCommunicator#handleConversationForClient(java.io.ObjectOutputStream, java.io.ObjectInputStream, java.io.Serializable)}.
         * 
         * @param pIn the input stream from the client
         * @param pOut the output stream to the client
         * @throws RuntimeException if the protocol is not met
         * @throws IOException if we have a technical communication error
         * @throws ClassNotFoundException if we have a technical communication error
         */
        private void handleConversationForServer(final ObjectInputStream pIn, final ObjectOutputStream pOut) throws IOException, ClassNotFoundException
        {
            // read magic number
            long clientMagicNumber = pIn.readLong();
            if (clientMagicNumber!=MAGIC_NUMBER)
            {
                pOut.writeObject( NOK_STRING_PREFIX+"Magic Number does not match");
                pOut.flush();
                throw new RuntimeException("Client sent different magic number: got "+clientMagicNumber+" expected "+MAGIC_NUMBER);
            }
            // send OK
            pOut.writeObject(OK_STRING);
            pOut.flush();
            // read application id
            String clientAppId = (String) pIn.readObject();
            if (!mApplicationId.equals(clientAppId))
            {
                pOut.writeObject( NOK_STRING_PREFIX+"Application id does not match");
                pOut.flush();
                throw new RuntimeException("Client sent different application id: got "+clientAppId+" expected "+mApplicationId);
            }
            // send OK
            pOut.writeObject(OK_STRING);
            pOut.flush();
            // read the data object
            Serializable data = (Serializable) pIn.readObject();
            // send OK
            pOut.writeObject(OK_STRING);
            pOut.flush();
            // process the data object
            Serializable result;
            try
            {
                result = mContactHandler.handleContactFromOtherInstance(data);
            }
            catch (RuntimeException ex)
            {
                pOut.writeObject( NOK_STRING_PREFIX+"Handler returned error: "+ex);
                pOut.flush();
                throw ex;
            }
            // send OK
            pOut.writeObject(OK_STRING);
            pOut.flush();
            // send the result
            pOut.writeObject(result);
            pOut.flush();
            // we're done - the streams will be closed by our caller
        }
    }

}
