package net.sf.cuf.xfer;

/**
 * This class provides the common stuff of all Request classes.
 * It provides delegate and dispatch target handling.
 * In most cases, a derived class must only implement the execute() method.
 */
public abstract class AbstractRequest<T> implements Request<T>
{
    /** null or our delegate */
    private RequestDelegate<T> mDelegate;

    /** null or our target */
    private DispatchTarget<T>  mTarget;

    /**
     * Derived classes must call this constructor via super() if they want to get
     * the benefit of delegate handling.
     */
    public AbstractRequest()
    {
        mDelegate= null;
        mTarget  = null;

        String delegateClassName= System.getProperty(this.getClass().getName()+".delegate");
        if (delegateClassName != null)
        {
            //noinspection EmptyCatchBlock
            try
            {
                Object delegate;
                if (delegateClassName.equals(this.getClass().getName()))
                {
                    delegate= this;
                }
                else
                {
                    delegate= Class.forName(delegateClassName).newInstance();
                }
                if (delegate instanceof RequestDelegate)
                {
                    //noinspection unchecked
                    mDelegate= (RequestDelegate)delegate;
                }
            }
            catch (Exception ignored)
            {
                // if the delegate extraction does not work, silently ignore it
            }
        }
    }

    @Override
    public void setDispatchTarget(final DispatchTarget<T> pTarget)
    {
        mTarget= pTarget;
    }

    @Override
    public void setDispatchTarget(final Object pTargetObject, final String pMethodName)
    {
        DispatchTarget<T> target= new DefaultDispatchTarget<>(pTargetObject, pMethodName);
        setDispatchTarget(target);
    }

    @Override
    public DispatchTarget<T> getDispatchTarget()
    {
        return mTarget;
    }

    @Override
    public RequestDelegate<T> getDelegate()
    {
        return mDelegate;
    }

    @Override
    public boolean canBeCancelled()
    {
        return false;
    }

    /**
     * Cancel the current request; this implementation will always throw an
     * IllegalStateException.
     * @throws IllegalStateException always thrown
     */
    public void cancel() throws IllegalStateException
    {
        throw new IllegalStateException("this request can't be canceled");
    }

    @Override
    public boolean equals(final Object pOther)
    {
        if (this == pOther)
            return true;
        if (!(pOther instanceof AbstractRequest))
            return false;

        final AbstractRequest abstractRequest = (AbstractRequest) pOther;

        if (mDelegate!=null ? !mDelegate.equals(abstractRequest.mDelegate) : abstractRequest.mDelegate!=null) return false;
        //noinspection RedundantIfStatement
        if (mTarget  !=null ? !mTarget.equals(abstractRequest.mTarget)     : abstractRequest.mTarget  !=null) return false;

        return true;
    }

    @Override
    public int hashCode()
    {
        int result;
        result = (mDelegate != null ? mDelegate.hashCode() : 0);
        result = 29 * result + (mTarget != null ? mTarget.hashCode() : 0);
        return result;
    }

    @Override
    public String toString()
    {
        return super.toString() + "[target=" + mTarget + ", delegate=" + mDelegate + ']';
    }
}
