package net.sf.cuf.xfer;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

/**
 * DefaultDispatchTarget is a default implementation for the DispatchTarget interface
 * that uses Java reflection to deliver the callback.
 */
public class DefaultDispatchTarget<T> implements DispatchTarget<T>
{
    /** our target object */
    private Object mTargetObject;
    /** the method of our target */
    private Method mTargetMethod;

    /**
     * Create a callback wrapper for the given object and method
     * @param pTargetObject the object to call back
     * @param pMethodName the method to call
     * @throws IllegalArgumentException if pTargetObject/pMethodName have issues
     */
    public  DefaultDispatchTarget(final Object pTargetObject, final String pMethodName) throws IllegalArgumentException
    {
        if (pTargetObject==null)
            throw new IllegalArgumentException("pTargetObject must not be null");
        if (pMethodName==null)
            throw new IllegalArgumentException("pMethodName must not be null");
        if (!Modifier.isPublic(pTargetObject.getClass().getModifiers()))
            throw new IllegalArgumentException("pTargetObject must be an object from a public class");

        /*
         * create Method Object, the signature is always
         * void <pMethodName>(Response pResponse)
         */
        Class<?>[] params= {Response.class};
        Method method;
        try
        {
            method= pTargetObject.getClass().getMethod(pMethodName, params);
        }
        catch (NoSuchMethodException e)
        {
            throw new IllegalArgumentException("No method: " + e.getMessage(), e);
        }

        mTargetObject= pTargetObject;
        mTargetMethod= method;
    }

    @Override
    public void callback(final Response<T> pResponse)
    {
        try
        {
            Object[] args= {pResponse};

            // do it: call the method with the generated response
            mTargetMethod.invoke(mTargetObject, args);
        }
        catch (IllegalAccessException e)
        {
            // map e to a IllegalArgumentException
            throw new IllegalArgumentException(e.getMessage(), e);
        }
        catch (InvocationTargetException e)
        {
            // map the cause of e to a IllegalArgumentException, if it is
            // not a RuntimeException or an Error
            Throwable cause= e.getTargetException();

            if (cause instanceof RuntimeException)
            {
                throw (RuntimeException) cause;
            }
            if (cause instanceof Error)
            {
                throw (Error) cause;
            }

            throw new IllegalArgumentException(cause!=null ? cause.getMessage() : e.getMessage(),
                                               cause!=null ? cause : e);
        }
    }

    @Override
    public boolean equals(final Object pObject)
    {
        if (pObject==this)
            return true;
        if (!(pObject instanceof DefaultDispatchTarget))
            return false;

        DefaultDispatchTarget target= (DefaultDispatchTarget) pObject;
        if (target.mTargetObject != mTargetObject)
        {
            return false;
        }

        if (mTargetMethod==null)
        {
            // target.mTargetMethod must be null in this case
            return (target.mTargetMethod==null);
        }

        return target.mTargetMethod.getName().equals(mTargetMethod.getName());
    }

    @Override
    public int hashCode()
    {
        return (mTargetObject.hashCode() | mTargetMethod.hashCode());
    }

    @Override
    public String toString()
    {
        return super.toString() + "[targetObject=" + mTargetObject + ",targetMethod=" + mTargetMethod + ']';
    }

}
