///////////////////////////////////////////////////////////////////////////////
//
// JTOpen (IBM Toolbox for Java - OSS version)
//
// Filename: AS400JDBCPreparedStatement.java
//
// The source code contained herein is licensed under the IBM Public License
// Version 1.0, which has been approved by the Open Source Initiative.
// Copyright (C) 1997-2006 International Business Machines Corporation and
// others. All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

package com.ibm.as400.access;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.BatchUpdateException;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.DataTruncation;
import java.sql.Date;
import java.sql.SQLWarning;
/* ifdef JDBC40
 import java.sql.NClob;
 endif */
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
/* ifdef JDBC40
 import java.sql.RowId;
 endif */
import java.sql.SQLException;
/* ifdef JDBC42
import java.sql.SQLType;
import java.sql.JDBCType;

endif */
/* ifdef JDBC40
 import java.sql.SQLXML;
 import java.sql.SQLFeatureNotSupportedException;
 endif */
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.Vector;

/**
 * <p>
 * The AS400JDBCPreparedStatement class precompiles and stores an SQL statement.
 * This provides the ability to efficiently run the statement multiple times. In
 * addition, the statement may contain parameters. Use
 * Connection.prepareStatement() to create new PreparedStatement objects.
 * 
 * <p>
 * When setting input parameter values, the caller must specify types that are
 * compatible with the defined SQL type of the input parameter. For example, if
 * the input parameter has SQL type INTEGER, then the caller must call setInt()
 * to set the IN parameter value. If arbitrary type conversions are required,
 * then use setObject() with a target SQL type.
 * 
 * <p>
 * For method that sets parameters, the application should not modify the
 * parameter value until after the execute completes. Modifying a value between
 * the setXXXX method and the execute method may result in unpredictable
 * behavior.
 **/
//
// Implementation notes:
//
// This class is used to form a common parent for
// AS400JDBCPreparedStatementImpl and AS400JDBCPreparedStatementRedirect.
// This is needed because some clients use the DB2 specific methods. 
// 
public abstract class AS400JDBCPreparedStatement extends AS400JDBCStatement implements
    PreparedStatement {
  static final String copyright2 = "Copyright (C) 1997-2006 International Business Machines Corporation and others.";

  
  // Any method that can deal with extremely large data values must be prepared
  // to deal with them in blocks instead of as one giant unit. This value is
  // used to determine the size of each block. Eventually, we might externalize
  // this value so that users can set it as they see fit.
  static final int LOB_BLOCK_SIZE = 1000000; // @pdc Match Native JDBC Driver
                                             // for IBM i

  
  /**
   * Constructs an AS400JDBCPreparedStatement object.
   * 
   * @param connection
   *          The connection to the system.
   * @param id
   *          The id.
   * @param transactionManager
   *          The transaction manager for the connection.
   * @param packageManager
   *          The package manager for the connection.
   * @param blockCriteria
   *          The block criteria.
   * @param blockSize
   *          The block size (in KB).
   * @param prefetch
   *          Indicates if prefetching data.
   * @param sqlStatement
   *          The SQL statement.
   * @param outputParametersExpected
   *          Indicates if output parameters are expected.
   * @param packageCriteria
   *          The package criteria.
   * @param resultSetType
   *          The result set type.
   * @param resultSetConcurrency
   *          The result set concurrency.
   * @param resultSetHoldability
   *          The result set holdability.
   * @param autoGeneratedKeys
   *          The auto-generated keys requested
   * @exception SQLException
   *              If the SQL statement contains a syntax error or an error
   *              occurs.
   **/
  AS400JDBCPreparedStatement(AS400JDBCConnection connection, int id,
      JDTransactionManager transactionManager, JDPackageManager packageManager,
      String blockCriteria, int blockSize, boolean prefetch,
       String packageCriteria, int resultSetType, int resultSetConcurrency,
      int resultSetHoldability, // @G4A
      int autoGeneratedKeys) // @G4A
      throws SQLException {
    super(connection, id, transactionManager, packageManager, blockCriteria,
        blockSize, prefetch, packageCriteria, resultSetType,
        resultSetConcurrency, resultSetHoldability, autoGeneratedKeys);
  }

  // Dummy constructor
  AS400JDBCPreparedStatement() {
    
  }
  // JDBC 2.0
  /**
   * Adds the set of parameters to the current batch.
   * 
   * @exception SQLException
   *              If the statement is not open or an input parameter has not
   *              been set.
   **/
  public abstract void addBatch() throws SQLException;

  // JDBC 2.0
  /**
   * Adds an SQL statement to the current batch of SQL statements.
   * 
   * <p>
   * Do not use this form of addBatch() on a prepared statement.
   * 
   * @param sql
   *          The SQL statement to be added to the current batch. This can be
   *          any SQL statement that does not return a result set.
   * @exception SQLException
   *              This exception is always thrown.
   **/
  public abstract void addBatch(String sql) throws SQLException;

  // @BAA

  /**
   * Releases the resources used by the current input parameter values. In
   * general, input parameter values remain in effect for repeated executions of
   * the prepared statement. Setting an input parameter value to a new value
   * automatically clears its previous value.
   * 
   * @exception SQLException
   *              If the statement is not open.
   **/
  public abstract void clearParameters() throws SQLException;


 
  /**
   * Runs an SQL statement that may return multiple result sets. This closes the
   * current result set and clears warnings before executing the SQL statement
   * again.
   * 
   * <p>
   * Under some situations, a single SQL statement may return multiple result
   * sets, an update count, or both. This might occur either when executing a
   * stored procedure that returns multiple result sets or when dynamically
   * executing an unknown SQL string.
   * 
   * <p>
   * Use Statement.getMoreResults(), Statement.getResultSet(), and
   * Statement.getUpdateCount() to navigate through multiple result sets, an
   * update count, or both.
   * 
   * @return true if a result set was returned; false if an update count was
   *         returned or nothing was returned.
   * @exception SQLException
   *              If the statement is not open, the query timeout limit is
   *              exceeded, or an error occurs.
   **/
  public abstract boolean execute() throws SQLException;

  /**
   * Runs an SQL statement that may return multiple result sets. This closes the
   * current result set and clears warnings before executing a new SQL
   * statement.
   * 
   * <p>
   * Do not use this form of execute() on a prepared statement.
   * 
   * @param sql
   *          The SQL statement.
   * @return true if a result set was returned, false if an update count was
   *         returned or nothing was returned.
   * @exception SQLException
   *              This exception is always thrown.
   **/
  public abstract boolean execute(String sql) throws SQLException;

  // @GAA
  /**
   * Runs an SQL statement that may return multiple result sets and makes any
   * auto-generated keys available for retrieval using
   * Statement.getGeneratedKeys(). This closes the current result set and clears
   * warnings before executing the new SQL statement.
   * 
   * <p>
   * Do not use this form of execute() on a prepared statement.
   * 
   * @param sql
   *          The SQL statement.
   * @param autoGeneratedKeys
   *          Indicates whether auto-generated keys should be made available for
   *          retrieval. Valid values are Statement.RETURN_GENERATED_KEYS and
   *          Statement.NO_GENERATED_KEYS.
   * @return true if a result set was returned, false if an update count was
   *         returned or nothing was returned.
   * @exception SQLException
   *              This exception is always thrown.
   * @since Modification 5
   **/
  public abstract boolean execute(String sql, int autoGeneratedKeys) throws SQLException;

  /**
   * Runs the batch of SQL statements. Batch updates can be used to submit a set
   * of SQL statements together as a single unit. The SQL statements are run in
   * the order in which they were added to the batch. The batch is cleared after
   * the SQL statements are run. In addition, this closes the current result set
   * and clears warnings before executing the new SQL statement.
   * 
   * <p>
   * When batch updates are run, autocommit should usually be turned off. This
   * allows the caller to decide whether or not to commit the transaction in the
   * event that an error occurs and some of the SQL statements in a batch fail
   * to run.
   * 
   * @return An array of row counts for the SQL statements that are run. The
   *         array contains one element for each statement in the batch of SQL
   *         statements. The array is ordered according to the order in which
   *         the SQL statements were added to the batch.
   * @exception SQLException
   *              If the statement is not open, an SQL statement contains a
   *              syntax error, the query timeout limit is exceeded, an SQL
   *              statement returns a result set, or an error occurs.
   **/
  public abstract int[] executeBatch() throws SQLException;

  /**
   * Runs the SQL statement that returns a single result set. This closes the
   * current result set and clears warnings before executing the SQL statement
   * again.
   * 
   * @return The result set that contains the data produced by the query.
   * @exception SQLException
   *              If the statement is not open, no result set is returned by the
   *              database, the query timeout limit is exceeded, an input
   *              parameter has not been set, or an error occurs.
   **/
  public abstract ResultSet executeQuery() throws SQLException;

  /**
   * Runs an SQL statement that returns a single result set. This closes the
   * current result set and clears warnings before executing a new SQL
   * statement.
   * 
   * <p>
   * Do not use this form of executeQuery() on a prepared statement.
   * 
   * @param sql
   *          The SQL statement.
   * @return The result set that contains the data produced by the query.
   * @exception SQLException
   *              This exception is always thrown.
   **/
  public abstract ResultSet executeQuery(String sql) throws SQLException;

  /**
   * Runs an SQL INSERT, UPDATE, or DELETE statement, or any SQL statement that
   * does not return a result set. This closes the current result set and clears
   * warnings before executing the SQL statement again.
   * 
   * @return Either the row count for INSERT, UPDATE, or DELETE, or 0 for SQL
   *         statements that return nothing.
   * @exception SQLException
   *              If the statement is not open, the query timeout limit is
   *              exceeded, the statement returns a result set, an input
   *              parameter has not been set, or an error occurs.
   **/
  public abstract int executeUpdate() throws SQLException;

  /**
   * Runs an SQL INSERT, UPDATE, or DELETE statement, or any SQL statement that
   * does not return a result set. This closes the current result set and clears
   * warnings before executing a new SQL statement.
   * 
   * <p>
   * Do not use this form of executeUpdate() on a prepared statement.
   * 
   * @param sql
   *          The SQL statement.
   * @return Either the row count for INSERT, UPDATE, or DELETE, or 0 for SQL
   *         statements that return nothing.
   * @exception SQLException
   *              This exception is always thrown.
   **/
  public abstract int executeUpdate(String sql) throws SQLException;

  // @GAA
  /**
   * Runs an SQL INSERT, UPDATE, or DELETE statement, or any SQL statement that
   * does not return a result set and makes any auto-generated keys available
   * for retrieval using Statement.getGeneratedKeys(). This closes the current
   * result set and clears warnings before executing the new SQL statement.
   * 
   * <p>
   * Do not use this form of executeUpdate() on a prepared statement.
   * 
   * @param sql
   *          The SQL statement.
   * @return Either the row count for INSERT, UPDATE, or DELETE, or 0 for SQL
   *         statements that return nothing.
   * @exception SQLException
   *              This exception is always thrown.
   * @since Modification 5
   **/
  public abstract int executeUpdate(String sql, int autoGeneratedKeys)
      throws SQLException;

  // JDBC 2.0
  /**
   * Returns the ResultSetMetaData object that describes the result set's
   * columns. Null is returned if the statement does not return a result set. In
   * the following example rsmd is null since the statement does not return a
   * result set.
   * 
   * <PRE>
   * PreparedStatement ps = connection
   *     .prepareStatement(&quot;INSERT INTO COLLECTION.TABLE VALUES(?)&quot;);
   * ResultSetMetaData rsmd = ps.getMetaData();
   * </PRE>
   * 
   * @return The metadata object, or null if the statement does not return a
   *         result set.
   * @exception SQLException
   *              If the statement is not open.
   **/
  public abstract ResultSetMetaData getMetaData() throws SQLException;


  
  
  /**
   * Returns the number, types, and properties of a PreparedStatement object's
   * parameters.
   * 
   * @return The ParameterMetaData object that describes this prepared statement
   *         object.
   * @exception SQLException
   *              If the statement is not open.
   * @since Modification 5
   **/
  public abstract ParameterMetaData getParameterMetaData() throws SQLException;

 
  // JDBC 2.0
  /**
   * Sets an input parameter to an Array value. DB2 for IBM i only supports
   * arrays in stored procedures.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value.
   * @exception SQLException
   *              Always thrown because DB2 for IBM i does not support arrays.
   **/
  public abstract void setArray(int parameterIndex, Array parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to an ASCII stream value. The driver reads the data
   * from the stream as needed until no more bytes are available. The driver
   * converts this to an SQL VARCHAR value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @param length
   *          The number of bytes in the stream.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, the length is not valid,
   *              the input stream does not contain all ASCII characters, or an
   *              error occurs while reading the input stream.
   **/
  public abstract void setAsciiStream(int parameterIndex, InputStream parameterValue,
      int length) throws SQLException;

  /**
   * Sets an input parameter to a BigDecimal value. The driver converts this to
   * an SQL NUMERIC value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   **/
  public abstract void setBigDecimal(int parameterIndex, BigDecimal parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to a binary stream value. The driver reads the data
   * from the stream as needed until no more bytes are available. The driver
   * converts this to an SQL VARBINARY value.
   * 
   * <br>
   * If a parameter is set using setBinaryStream, then the parameter must be
   * reset prior to the second execute of the PreparedStatement object.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @param length
   *          The number of bytes in the stream.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, the length is not valid,
   *              or an error occurs while reading the input stream.
   **/
  public abstract void setBinaryStream(int parameterIndex, InputStream parameterValue,
      int length) throws SQLException;

  // JDBC 2.0
  /**
   * Sets an input parameter to a Blob value. The driver converts this to an SQL
   * BLOB value. <br>
   * If proxy support is in use, the Blob must be serializable.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, or the parameter is not
   *              serializable (when proxy support is in use).
   **/
  public abstract void setBlob(int parameterIndex, Blob parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to a Java boolean value. The driver converts this
   * to an SQL SMALLINT value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   **/
  //
  // Implementation note:
  //
  // The spec defines this in terms of SQL BIT, but DB2 for IBM i
  // does not support that.
  //
  public abstract void setBoolean(int parameterIndex, boolean parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to a Java byte value. The driver converts this to
   * an SQL SMALLINT value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   **/
  //
  // Implementation note:
  //
  // The spec defines this in terms of SQL TINYINT, but DB2 for IBM i
  // does not support that.
  //
  public abstract void setByte(int parameterIndex, byte parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to a Java byte array value. The driver converts
   * this to an SQL VARBINARY value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   **/
  public abstract void setBytes(int parameterIndex, byte[] parameterValue)
      throws SQLException;

  // JDBC 2.0
  /**
   * Sets an input parameter to a character stream value. The driver reads the
   * data from the character stream as needed until no more characters are
   * available. The driver converts this to an SQL VARCHAR value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @param length
   *          The number of characters to read from the reader.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, the length is not valid,
   *              or an error occurs while reading the character stream
   **/
  public abstract void setCharacterStream(int parameterIndex, Reader parameterValue,
      int length) throws SQLException;

  // JDBC 2.0
  /**
   * Sets an input parameter to a Clob value. The driver converts this to an SQL
   * CLOB value. <br>
   * If proxy support is in use, the Clob must be serializable.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, or the parameter is not
   *              serializable (when proxy support is in use).
   **/
  public abstract void setClob(int parameterIndex, Clob parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to a java.sql.Date value using the default
   * calendar. The driver converts this to an SQL DATE value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   **/
  public abstract void setDate(int parameterIndex, Date parameterValue)
      throws SQLException;

  // JDBC 2.0
  /**
   * Sets an input parameter to a java.sql.Date value using a calendar other
   * than the default. The driver converts this to an SQL DATE value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @param calendar
   *          The calendar.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, or the calendar is null.
   **/
  public abstract void setDate(int parameterIndex, Date parameterValue, Calendar calendar)
      throws SQLException;

  // @EIA 550 extended indicator defaults
  /**
   * Sets an input parameter to the default value
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter.
   **/
  public abstract void setDB2Default(int parameterIndex) throws SQLException;

  // @EIA 550 extended indicator defaults
  /**
   * Sets an input parameter to the default value. This is a the same as
   * setDB2Default.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter.
   **/
  public abstract void setDBDefault(int parameterIndex) throws SQLException;

  // @EIA 550 extended indicator defaults
  /**
   * Sets an input parameter to unassigned
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter.
   **/
  public abstract void setDB2Unassigned(int parameterIndex) throws SQLException;

  // @EIA 550 extended indicator defaults
  /**
   * Sets an input parameter to unassigned. This is a the same as
   * setDB2Unassigned.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter.
   **/
  public abstract void setDBUnassigned(int parameterIndex) throws SQLException;

  /**
   * Sets an input parameter to a Java double value. The driver converts this to
   * an SQL DOUBLE value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value.
   * @exception SQLException
   *              If the statement is not open, the index is not valid or the
   *              parameter is not an input parameter.
   **/
  public abstract void setDouble(int parameterIndex, double parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to a Java float value. The driver converts this to
   * an SQL REAL value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   **/
  //
  // Note: The JDBC 1.22 specification states that this
  // method should set an SQL FLOAT value. However,
  // all tables map float to REAL. Otherwise,
  // nothing is symmetrical and certain INOUT
  // parameters do not work.
  //
  public abstract void setFloat(int parameterIndex, float parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to a Java int value. The driver converts this to an
   * SQL INTEGER value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value.
   * @exception SQLException
   *              If the statement is not open, the index is not valid or the
   *              parameter is not an input parameter.
   **/
  public abstract void setInt(int parameterIndex, int parameterValue)
      throws SQLException;

  // @D0C
  /**
   * Sets an input parameter to a Java long value. If the connected system
   * supports SQL BIGINT data, the driver converts this to an SQL BIGINT value.
   * Otherwise, the driver converts this to an SQL INTEGER value. SQL BIGINT
   * data is supported on V4R5 and later.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   **/
  //
  // Implementation note:
  //
  // The spec defines this in terms of SQL BIGINT, but DB2 for IBM i
  // does not support that until V4R5.
  //
  public abstract void setLong(int parameterIndex, long parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to SQL NULL.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param sqlType
   *          The SQL type code defined in java.sql.Types.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, or the SQL type is not
   *              valid.
   **/
  public abstract void setNull(int parameterIndex, int sqlType) throws SQLException;

  // @B4 - Added for JDK 2.0RC1 - typeName can be ignored, since it is not
  // relevant to IBM i.
  /**
   * Sets an input parameter to SQL NULL.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param sqlType
   *          The SQL type code defined in java.sql.Types.
   * @param typeName
   *          The fully-qualified name of an SQL structured type. This value
   *          will be ignored.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, or the SQL type is not
   *              valid.
   **/
  public abstract void setNull(int parameterIndex, int sqlType, String typeName)
      throws SQLException;

  /**
   * Sets an input parameter to an Object value. The driver converts this to a
   * value of an SQL type, depending on the type of the specified value. The
   * JDBC specification defines a standard mapping from Java types to SQL types.
   * In the cases where a SQL type is not supported by DB2 for IBM i, the <a
   * href="doc-files/SQLTypes.html#unsupported">next closest matching type</a>
   * is used. <br>
   * If proxy support is in use, the Object must be serializable.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, the type of value is not
   *              supported, or the parameter is not serializable (when proxy
   *              support is in use).
   **/
  public abstract void setObject(int parameterIndex, Object parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to an Object value. The driver converts this to a
   * value with the specified SQL type. <br>
   * If proxy support is in use, the Object must be serializable.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @param sqlType
   *          The SQL type code defined in java.sql.Types.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, the SQL type is not
   *              valid, or the parameter is not serializable (when proxy
   *              support is in use).
   **/
  public abstract void setObject(int parameterIndex, Object parameterValue, int sqlType)
      throws SQLException;

  /**
   * Sets an input parameter to an Object value. The driver converts this to a
   * value with the specified SQL type. <br>
   * If proxy support is in use, the Object must be serializable.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @param sqlType
   *          The SQL type code defined in java.sql.Types.
   * @param scale
   *          The number of digits after the decimal if sqlType is DECIMAL or
   *          NUMERIC.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, the SQL type is not
   *              valid, the scale is not valid, or the parameter is not
   *              serializable (when proxy support is in use).
   **/
  public abstract void setObject(int parameterIndex, Object parameterValue, int sqlType,
      int scale) throws SQLException;

  // JDBC 2.0
  /**
    Sets an input parameter to a Ref value.  DB2 for IBM i
    does not support structured types.

    @param  parameterIndex  The parameter index (1-based).
    @param  parameterValue  The parameter value.

    @exception  SQLException    Always thrown because DB2 for IBM i does not support structured types.
   **/
  public abstract void setRef(int parameterIndex, Ref parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to a Java short value. The driver converts this to
   * an SQL SMALLINT value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value.
   * @exception SQLException
   *              If the statement is not open, the index is not valid or the
   *              parameter is not an input parameter.
   **/
  public abstract void setShort(int parameterIndex, short parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to a String value. The driver converts this to an
   * SQL VARCHAR value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   **/
  public abstract void setString(int parameterIndex, String parameterValue)
      throws SQLException;

  /**
   * Sets an input parameter to a java.sql.Time value using the default
   * calendar. The driver converts this to an SQL TIME value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   **/
  public abstract void setTime(int parameterIndex, Time parameterValue)
      throws SQLException;

  // JDBC 2.0
  /**
   * Sets an input parameter to a java.sql.Time value using a calendar other
   * than the default. The driver converts this to an SQL TIME value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @param calendar
   *          The calendar.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, or the calendar is null.
   **/
  public abstract void setTime(int parameterIndex, Time parameterValue, Calendar calendar)
      throws SQLException;

  /**
   * Sets an input parameter to a java.sql.Timestamp value using the default
   * calendar. The driver converts this to an SQL TIMESTAMP value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   **/
  public abstract void setTimestamp(int parameterIndex, Timestamp parameterValue)
      throws SQLException;

  // JDBC 2.0
  /**
   * Sets an input parameter to a java.sql.Timestamp value using a calendar
   * other than the default. The driver converts this to an SQL TIMESTAMP value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @param calendar
   *          The calendar.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, or the calendar is null.
   **/
  public abstract void setTimestamp(int parameterIndex, Timestamp parameterValue,
      Calendar calendar) throws SQLException;

  /**
   * Sets an input parameter to a Unicode stream value. The driver reads the
   * data from the stream as needed until no more bytes are available. The
   * driver converts this to an SQL VARCHAR value.
   * <p>
   * Note that the number of bytes in a Unicode stream can be computed as 2
   * multiplied by the number of characters plus 2 bytes for the byte-order
   * mark. If an uneven number of bytes is specified, then Java will convert
   * this to an empty String.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @param length
   *          The number of bytes in the stream.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, the
   *              parameter is not an input parameter, the length is not valid,
   *              the input stream does not contain all Unicode characters, or
   *              an error occurs while reading the input stream
   * @deprecated Use setCharacterStream(int, Reader, int) instead.
   * @see #setCharacterStream
   **/
  public abstract void setUnicodeStream(int parameterIndex, InputStream parameterValue,
      int length) throws SQLException;

  // @G4A JDBC 3.0
  /**
   * Sets an input parameter to a URL value. The driver converts this to an SQL
   * DATALINK value.
   * 
   * @param parameterIndex
   *          The parameter index (1-based).
   * @param parameterValue
   *          The parameter value or null to set the value to SQL NULL.
   * @exception SQLException
   *              If the statement is not open, the index is not valid, or the
   *              parameter is not an input parameter.
   * @since Modification 5
   **/
  public abstract void setURL(int parameterIndex, URL parameterValue)
      throws SQLException;

    // @PDA jdbc40
  // JDBC40DOC /**
 // JDBC40DOC     * Sets the designated parameter to the given <code>java.sql.RowId</code> object. The
 // JDBC40DOC     * driver converts this to a SQL <code>ROWID</code> value when it sends it
  // JDBC40DOC * to the database
  // JDBC40DOC *
  // JDBC40DOC * @param parameterIndex
  // JDBC40DOC * @param x the parameter value
  // JDBC40DOC * @throws SQLException if a database access error occurs
  // JDBC40DOC *
  // JDBC40DOC */
/* ifdef JDBC40 
    public abstract void setRowId(int parameterIndex, RowId x) throws SQLException;
endif */
  // @PDA jdbc40
  /**
   * Sets the designated paramter to the given <code>String</code> object. The
   * driver converts this to a SQL <code>NCHAR</code> or <code>NVARCHAR</code>
   * or <code>LONGNVARCHAR</code> value (depending on the argument's size
   * relative to the driver's limits on <code>NVARCHAR</code> values) when it
   * sends it to the database.
   * 
   * @param parameterIndex
   * @param value
   *          the parameter value
   * @throws SQLException
   *           if the driver does not support national character sets; if the
   *           driver can detect that a data conversion error could occur ; or
   *           if a database access error occurs
   */
  public abstract void setNString(int parameterIndex, String value) throws SQLException;

  // @PDA jdbc40
  /**
   * Sets the designated parameter to a <code>Reader</code> object. The
   * <code>Reader</code> reads the data till end-of-file is reached. The driver
   * does the necessary conversion from Java character format to the national
   * character set in the database.
   * 
   * @param parameterIndex
   * @param value
   *          the parameter value
   * @param length
   *          the number of characters in the parameter data.
   * @throws SQLException
   *           if the driver does not support national character sets; if the
   *           driver can detect that a data conversion error could occur ; or
   *           if a database access error occurs
   */
  public abstract void setNCharacterStream(int parameterIndex, Reader value, long length)
      throws SQLException;

  // @PDA jdbc40
  // JDBC40DOC /**
  // JDBC40DOC     * Sets the designated parameter to a <code>java.sql.NClob</code> object. The driver converts this to a
  // JDBC40DOC * SQL <code>NCLOB</code> value when it sends it to the database.
  // JDBC40DOC * @param parameterIndex
  // JDBC40DOC * @param value the parameter value
  // JDBC40DOC * @throws SQLException if the driver does not support national
  // JDBC40DOC * character sets; if the driver can detect that a data conversion
  // JDBC40DOC * error could occur ; or if a database access error occurs
  // JDBC40DOC */
/* ifdef JDBC40 
     public abstract void setNClob(int parameterIndex, NClob value) throws SQLException;
    
endif */

  // @PDA jdbc40
  /**
   * Sets the designated parameter to a <code>Reader</code> object. The reader
   * must contain the number of characters specified by length otherwise a
   * <code>SQLException</code> will be generated when the
   * <code>PreparedStatement</code> is executed.
   * 
   * @param parameterIndex
   * @param reader
   *          An object that contains the data to set the parameter value to.
   * @param length
   *          the number of characters in the parameter data.
   * @throws SQLException
   *           if parameterIndex does not correspond to a parameter marker in
   *           the SQL statement, or if the length specified is less than zero.
   * 
   */
  public abstract void setClob(int parameterIndex, Reader reader, long length)
      throws SQLException;

  // @PDA jdbc40
  /**
   * Sets the designated parameter to an <code>InputStream</code> object. The
   * inputStream must contain the number of characters specified by length
   * otherwise a <code>SQLException</code> will be generated when the
   * <code>PreparedStatement</code> is executed.
   * 
   * @param parameterIndex
   * @param inputStream
   *          An object that contains the data to set the parameter value to.
   * @param length
   *          the number of bytes in the parameter data.
   * @throws SQLException
   *           if parameterIndex does not correspond to a parameter marker in
   *           the SQL statement, if the length specified is less than zero or
   *           if the number of bytes in the inputstream does not match the
   *           specfied length.
   * 
   */
  public abstract void setBlob(int parameterIndex, InputStream inputStream, long length)
      throws SQLException;

  // @PDA jdbc40
  /**
   * Sets the designated parameter to a <code>Reader</code> object. The reader
   * must contain the number of characters specified by length otherwise a
   * <code>SQLException</code> will be generated when the
   * <code>PreparedStatement</code> is executed.
   * 
   * @param parameterIndex
   * @param reader
   *          An object that contains the data to set the parameter value to.
   * @param length
   *          the number of characters in the parameter data.
   * @throws SQLException
   *           if parameterIndex does not correspond to a parameter marker in
   *           the SQL statement; if the length specified is less than zero; if
   *           the driver does not support national character sets; if the
   *           driver can detect that a data conversion error could occur; or if
   *           a database access error occurs
   * 
   */
  public abstract void setNClob(int parameterIndex, Reader reader, long length)
      throws SQLException;

  // @PDA jdbc40
  // JDBC40DOC /**
  // JDBC40DOC      * Sets the designated parameter to the given <code>java.sql.SQLXML</code> object.
  // JDBC40DOC * @param parameterIndex
  // JDBC40DOC      * @param xmlObject a <code>SQLXML</code> object that maps an SQL <code>XML</code> value
  // JDBC40DOC * @throws SQLException if a database access error occurs
  // JDBC40DOC */
  /* ifdef JDBC40 
     public abstract void setSQLXML(int parameterIndex, SQLXML xmlObject) throws SQLException;
     endif */

  // @pda jdbc40
  protected String[] getValidWrappedList() {
    return new String[] { "com.ibm.as400.access.AS400JDBCPreparedStatement",
        "java.sql.PreparedStatement" };
  }

  // @PDA jdbc40
  /**
   * Sets the designated parameter to the given input stream, which will have
   * the specified number of bytes. When a very large ASCII value is input to a
   * <code>LONGVARCHAR</code> parameter, it may be more practical to send it via
   * a <code>java.io.InputStream</code>. Data will be read from the stream as
   * needed until end-of-file is reached. The JDBC driver will do any necessary
   * conversion from ASCII to the database char format.
   * 
   * <P>
   * <B>Note:</B> This stream object can either be a standard Java stream object
   * or your own subclass that implements the standard interface.
   * 
   * @param parameterIndex
   * @param x
   *          the Java input stream that contains the ASCII parameter value
   * @param length
   *          the number of bytes in the stream
   * @exception SQLException
   *              if a database access error occurs or this method is called on
   *              a closed <code>PreparedStatement</code>
   */
  public abstract void setAsciiStream(int parameterIndex, InputStream x, long length)
      throws SQLException;

  // @PDA jdbc40
  /**
   * Sets the designated parameter to the given input stream, which will have
   * the specified number of bytes. When a very large binary value is input to a
   * <code>LONGVARBINARY</code> parameter, it may be more practical to send it
   * via a <code>java.io.InputStream</code> object. The data will be read from
   * the stream as needed until end-of-file is reached.
   * 
   * <P>
   * <B>Note:</B> This stream object can either be a standard Java stream object
   * or your own subclass that implements the standard interface.
   * 
   * @param parameterIndex
   * @param x
   *          the java input stream which contains the binary parameter value
   * @param length
   *          the number of bytes in the stream
   * @exception SQLException
   *              if a database access error occurs or this method is called on
   *              a closed <code>PreparedStatement</code>
   */
  public abstract void setBinaryStream(int parameterIndex, InputStream x, long length)
      throws SQLException;

  // @PDA jdbc40
  /**
   * Sets the designated parameter to the given <code>Reader</code> object,
   * which is the given number of characters long. When a very large UNICODE
   * value is input to a <code>LONGVARCHAR</code> parameter, it may be more
   * practical to send it via a <code>java.io.Reader</code> object. The data
   * will be read from the stream as needed until end-of-file is reached. The
   * JDBC driver will do any necessary conversion from UNICODE to the database
   * char format.
   * 
   * <P>
   * <B>Note:</B> This stream object can either be a standard Java stream object
   * or your own subclass that implements the standard interface.
   * 
   * @param parameterIndex
   * @param reader
   *          the <code>java.io.Reader</code> object that contains the Unicode
   *          data
   * @param length
   *          the number of characters in the stream
   * @exception SQLException
   *              if a database access error occurs or this method is called on
   *              a closed <code>PreparedStatement</code>
   */
  public abstract void setCharacterStream(int parameterIndex, Reader reader, long length)
      throws SQLException;

  /**
   * Return the name of the parameter for a stored procedure call. 
   * @param parm the parameter number to get the name for
   * @return the parameter name. 
   * @throws SQLException
   */
  public abstract String getDB2ParameterName(int parm) throws SQLException;
 
  // @PDA jdbc40
  /**
   * Sets the designated parameter to the given input stream. When a very large
   * ASCII value is input to a <code>LONGVARCHAR</code> parameter, it may be
   * more practical to send it via a <code>java.io.InputStream</code>. Data will
   * be read from the stream as needed until end-of-file is reached. The JDBC
   * driver will do any necessary conversion from ASCII to the database char
   * format.
   * 
   * <P>
   * <B>Note:</B> This stream object can either be a standard Java stream object
   * or your own subclass that implements the standard interface.
   * 
   * @param parameterIndex
   * @param x
   *          the Java input stream that contains the ASCII parameter value
   * @exception SQLException
   *              if parameterIndex does not correspond to a parameter marker in
   *              the SQL statement; if a database access error occurs or this
   *              method is called on a closed <code>PreparedStatement</code>
   */
  public abstract void setAsciiStream(int parameterIndex, InputStream x)
      throws SQLException;

  // @PDA jdbc40
  /**
   * Sets the designated parameter to the given input stream. When a very large
   * binary value is input to a <code>LONGVARBINARY</code> parameter, it may be
   * more practical to send it via a <code>java.io.InputStream</code> object.
   * The data will be read from the stream as needed until end-of-file is
   * reached.
   * 
   * <P>
   * <B>Note:</B> This stream object can either be a standard Java stream object
   * or your own subclass that implements the standard interface.
   * 
   * @param parameterIndex
   * @param x
   *          the java input stream which contains the binary parameter value
   * @exception SQLException
   *              if parameterIndex does not correspond to a parameter marker in
   *              the SQL statement; if a database access error occurs or this
   *              method is called on a closed <code>PreparedStatement</code>
   */
  public abstract void setBinaryStream(int parameterIndex, InputStream x)
      throws SQLException;

  // @PDA jdbc40
  /**
   * Sets the designated parameter to a <code>InputStream</code> object. This
   * method differs from the <code>setBinaryStream (int, InputStream)</code>
   * method because it informs the driver that the parameter value should be
   * sent to the server as a <code>BLOB</code>. When the
   * <code>setBinaryStream</code> method is used, the driver may have to do
   * extra work to determine whether the parameter data should be sent to the
   * server as a <code>LONGVARBINARY</code> or a <code>BLOB</code>
   * 
   * @param parameterIndex
   * @param inputStream
   *          An object that contains the data to set the parameter value to.
   * @throws SQLException
   *           if parameterIndex does not correspond to a parameter marker in
   *           the SQL statement; if a database access error occurs; this method
   *           is called on a closed <code>PreparedStatement</code> or if
   *           parameterIndex does not correspond to a parameter marker in the
   *           SQL statement,
   * 
   */
  public abstract void setBlob(int parameterIndex, InputStream inputStream)
      throws SQLException;

  /**
   * Sets the designated parameter to the given <code>Reader</code> object. When
   * a very large UNICODE value is input to a <code>LONGVARCHAR</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.Reader</code> object. The data will be read from the stream
   * as needed until end-of-file is reached. The JDBC driver will do any
   * necessary conversion from UNICODE to the database char format.
   * 
   * <P>
   * <B>Note:</B> This stream object can either be a standard Java stream object
   * or your own subclass that implements the standard interface.
   * <P>
   * <B>Note:</B> Consult your JDBC driver documentation to determine if it
   * might be more efficient to use a version of <code>setCharacterStream</code>
   * which takes a length parameter.
   * 
   * @param parameterIndex
   * @param reader
   *          the <code>java.io.Reader</code> object that contains the Unicode
   *          data
   * @exception SQLException
   *              if parameterIndex does not correspond to a parameter marker in
   *              the SQL statement; if a database access error occurs or this
   *              method is called on a closed <code>PreparedStatement</code>
   */
  public abstract void setCharacterStream(int parameterIndex, Reader reader)
      throws SQLException;

  // @PDA jdbc40
  /**
   * Sets the designated parameter to a <code>Reader</code> object. This method
   * differs from the <code>setCharacterStream (int, Reader)</code> method
   * because it informs the driver that the parameter value should be sent to
   * the server as a <code>CLOB</code>. When the <code>setCharacterStream</code>
   * method is used, the driver may have to do extra work to determine whether
   * the parameter data should be sent to the server as a
   * <code>LONGVARCHAR</code> or a <code>CLOB</code>
   * 
   * <P>
   * <B>Note:</B> Consult your JDBC driver documentation to determine if it
   * might be more efficient to use a version of <code>setClob</code> which
   * takes a length parameter.
   * 
   * @param parameterIndex
   * @param reader
   *          An object that contains the data to set the parameter value to.
   * @throws SQLException
   *           if parameterIndex does not correspond to a parameter marker in
   *           the SQL statement; if a database access error occurs; this method
   *           is called on a closed <code>PreparedStatement</code>or if
   *           parameterIndex does not correspond to a parameter marker in the
   *           SQL statement
   * 
   */
  public abstract void setClob(int parameterIndex, Reader reader) throws SQLException;

  // @PDA jdbc40
  /**
   * Sets the designated parameter to a <code>Reader</code> object. The
   * <code>Reader</code> reads the data till end-of-file is reached. The driver
   * does the necessary conversion from Java character format to the national
   * character set in the database.
   * 
   * <P>
   * <B>Note:</B> This stream object can either be a standard Java stream object
   * or your own subclass that implements the standard interface.
   * <P>
   * <B>Note:</B> Consult your JDBC driver documentation to determine if it
   * might be more efficient to use a version of
   * <code>setNCharacterStream</code> which takes a length parameter.
   * 
   * @param parameterIndex
   * @param value
   *          the parameter value
   * @throws SQLException
   *           if parameterIndex does not correspond to a parameter marker in
   *           the SQL statement; if the driver does not support national
   *           character sets; if the driver can detect that a data conversion
   *           error could occur; if a database access error occurs; or this
   *           method is called on a closed <code>PreparedStatement</code>
   */
  public abstract void setNCharacterStream(int parameterIndex, Reader value)
      throws SQLException;

  // @PDA jdbc40
  /**
   * Sets the designated parameter to a <code>Reader</code> object. This method
   * differs from the <code>setCharacterStream (int, Reader)</code> method
   * because it informs the driver that the parameter value should be sent to
   * the server as a <code>NCLOB</code>. When the
   * <code>setCharacterStream</code> method is used, the driver may have to do
   * extra work to determine whether the parameter data should be sent to the
   * server as a <code>LONGNVARCHAR</code> or a <code>NCLOB</code>
   * <P>
   * <B>Note:</B> Consult your JDBC driver documentation to determine if it
   * might be more efficient to use a version of <code>setNClob</code> which
   * takes a length parameter.
   * 
   * @param parameterIndex
   * @param reader
   *          An object that contains the data to set the parameter value to.
   * @throws SQLException
   *           if parameterIndex does not correspond to a parameter marker in
   *           the SQL statement; if the driver does not support national
   *           character sets; if the driver can detect that a data conversion
   *           error could occur; if a database access error occurs or this
   *           method is called on a closed <code>PreparedStatement</code>
   */
  public abstract void setNClob(int parameterIndex, Reader reader) throws SQLException;
  
   
/**
 * Sets the value of the designated parameter with the given object. If the second 
 * argument is an InputStream then the stream must contain the number of bytes 
 * specified by scaleOrLength. If the second argument is a Reader then the reader 
 * must contain the number of characters specified by scaleOrLength. If these 
 * conditions are not true the driver will generate a SQLException when the 
 * prepared statement is executed.
 * <p>The given Java object will be converted to the given targetSqlType 
 * before being sent to the database. If the object has a custom mapping 
 * (is of a class implementing the interface SQLData), the JDBC driver 
 * should call the method SQLData.writeSQL to write it to the SQL data stream. 
 * If, on the other hand, the object is of a class implementing Ref, Blob, 
 * Clob, NClob, Struct, java.net.URL, or Array, the driver should pass it 
 * to the database as a value of the corresponding SQL type.
 * <p>Note that this method may be used to pass database-specific abstract data types.
 * 
 * @param parameterIndex - the first parameter is 1, the second is 2, ...
 * @param x - the object containing the input parameter value
 * @param targetSqlType - the SQL type to be sent to the database. The scale argument may
 *  further qualify this type.
 * @param scaleOrLength - for java.sql.JDBCType.DECIMAL or java.sql.JDBCType.NUMERIC 
 * types, this is the number of digits after the decimal point. For Java Object types 
 * InputStream and Reader, this is the length of the data in the stream or reader. 
 * For all other types, this value will be ignored.
 * @throws SQLException - if parameterIndex does not correspond to a parameter 
 * marker in the SQL statement; if a database access error occurs or this method 
 * is called on a closed PreparedStatement or if the Java Object specified by x 
 * is an InputStream or Reader object and the value of the scale parameter is 
 * less than zero
 */
  public abstract void setObject(int parameterIndex,
        Object x,
/* ifdef JDBC42        
        SQLType  
endif*/ 
/* ifndef JDBC42 */
Object
/* endif */
        targetSqlType,
        int scaleOrLength)
 throws SQLException ;
  



  /**
   * Sets the value of the designated parameter with the given object. This method 
   * is similar to setObject(int parameterIndex, Object x, SQLType targetSqlType, 
   * int scaleOrLength), except that it assumes a scale of zero.
   * @param parameterIndex - the first parameter is 1, the second is 2, ...
   * @param x - the object containing the input parameter value
   * @param targetSqlType  - the SQL type to be sent to the database parameterIndex
   * @throws SQLException - if parameterIndex does not correspond to a parameter marker 
   * in the SQL statement; if a database access error occurs or this method is called 
   * on a closed PreparedStatement
   */
  public abstract void setObject(int parameterIndex,
                         Object x,
                         /* ifdef JDBC42        
                         SQLType  
                   endif*/ 
                   /* ifndef JDBC42 */
                   Object
                   /* endif */
                         targetSqlType)
                  throws SQLException;
  
 /**
  *  Executes the SQL statement in this PreparedStatement object, which 
  *  must be an SQL Data Manipulation Language (DML) statement, such as 
  *  INSERT, UPDATE or DELETE; or an SQL statement that returns nothing, 
  *  such as a DDL statement.
  * <p>This method should be used when the returned row count may exceed Integer.MAX_VALUE.
  * @return either (1) the row count for SQL Data Manipulation Language (DML) statements or 
  * (2) 0 for SQL statements that return nothing
  * @throws SQLException - if a database access error occurs; this method is called on a 
  * closed PreparedStatement or the SQL statement returns a ResultSet object
  */

  public abstract long executeLargeUpdate() throws SQLException;
  
  
  // Internal methods used by RowSet and parameter metadata
  abstract int findParameterIndex(String s) throws SQLException; 
  abstract int getParameterCcsid(int p) throws SQLException; 
  abstract String getParameterClassName(int p) throws SQLException; 
  abstract int getParameterCount() throws SQLException; 
  abstract int getParameterMode(int param) throws SQLException; 
  abstract int getParameterType(int param) throws SQLException; 
  abstract String getParameterTypeName(int param) throws SQLException ; 
  abstract int getPrecision(int param) throws SQLException; 
  abstract JDServerRow getResultRow(); 
  abstract int getScale(int param) throws SQLException;
  abstract int isNullable(int param) throws SQLException;
  abstract boolean isSigned(int param) throws SQLException; 
  
  abstract void setSaveParameterValues(boolean saveParameterValues); 
  
}
