/* Copyright (c) The m-m-m Team, Licensed under the Apache License, Version 2.0
 * http://www.apache.org/licenses/LICENSE-2.0 */
package net.sf.mmm.util.collection.api;

import java.util.Map;

/**
 * This is the interface for a factory of {@link Map maps}. It allows to abstract from {@link Map} implementations. <br>
 * A {@link Map} instance can be used for different purposes such as a cache or with different aspects such as a
 * thread-safe map. If you write a generic component different users of that component may expect different aspects of
 * your component and therefore the underlying {@link Map}. <br>
 * If you use this interface and allow the user to {@link javax.inject.Inject inject} an instance of this interface to
 * override the default, your code will increase usability. <br>
 * <b>Why passing a {@link MapFactory} rather than a {@link Map} instance to the constructor?</b><br>
 * Since java 5 you want to use generics for type-safe code. If these generic types change slightly over the time of
 * development (e.g. from {@code Class} to {@literal Class<?>}) you would break compatibility of the users of your code.
 * Additionally you may want to express that the {@link Map} should be empty and/or NOT shared with others. Anyways the
 * interface can obviously NOT guarantee this.
 *
 * @see net.sf.mmm.util.collection.base.HashMapFactory#INSTANCE
 *
 * @param <MAP> is the generic {@link Map}-type.
 * @author Joerg Hohwiller (hohwille at users.sourceforge.net)
 * @since 1.0.0
 */
@SuppressWarnings("rawtypes")
public interface MapFactory<MAP extends Map> {

  /**
   * This method gets the interface of the {@link Map} managed by this factory.
   *
   * @return the {@link Map} interface.
   */
  Class<MAP> getMapInterface();

  /**
   * This method gets the implementation of the {@link #getMapInterface() map-interface} used by this factory.
   *
   * @return the {@link Map} implementation.
   */
  Class<? extends MAP> getMapImplementation();

  /**
   * This method creates a new {@link Map} instance. <br>
   * It is explicitly typed and respects the generic key and value type of the map. Therefore the type of the
   * {@link Map} can NOT be bound to the generic type {@literal <MAP>} because of limitations in Java's generic type
   * system. You need to work on the actual sub-interface (e.g. {@link SortedMapFactory}) to get a more specific result
   * type.
   *
   * @param <K> the type of keys maintained by the map.
   * @param <V> the type of mapped values.
   * @return the new map instance.
   */
  <K, V> Map<K, V> create();

  /**
   * This method creates a new {@link Map} instance with the given {@code capacity}. For a regular map this will be the
   * initial capacity while a cache may never grow beyond this capacity limit and if reached force out entries last
   * recently of frequently used.
   *
   * @param <K> the type of keys maintained by the map.
   * @param <V> the type of mapped values.
   * @param capacity is the capacity of the map to create.
   * @return the new map instance.
   */
  <K, V> Map<K, V> create(int capacity);

  /**
   * This method creates a new instance of the generic {@link Map} type {@literal <MAP>}.
   *
   * @return the new {@link Map} instance.
   */
  MAP createGeneric();

  /**
   * This method creates a new instance of the generic {@link Map} type {@literal <MAP>}.
   *
   * @param capacity is the capacity of the map to create.
   * @return the new {@link Map} instance.
   */
  MAP createGeneric(int capacity);

}
