/* Copyright (c) The m-m-m Team, Licensed under the Apache License, Version 2.0
 * http://www.apache.org/licenses/LICENSE-2.0 */
package net.sf.mmm.util.pojo.path.base;

import java.util.HashMap;
import java.util.Map;

import net.sf.mmm.util.component.base.AbstractComponent;
import net.sf.mmm.util.pojo.path.api.PojoPathFunction;
import net.sf.mmm.util.pojo.path.api.PojoPathFunctionManager;

/**
 * This is the default implementation of the {@link PojoPathFunctionManager}.
 *
 * @author Joerg Hohwiller (hohwille at users.sourceforge.net)
 * @since 1.1.0
 */
@SuppressWarnings("rawtypes")
public class DefaultPojoPathFunctionManager extends AbstractComponent implements PojoPathFunctionManager {

  private final Map<String, PojoPathFunction> functionMap;

  /**
   * The constructor.
   */
  public DefaultPojoPathFunctionManager() {

    this(new HashMap<String, PojoPathFunction>());
  }

  /**
   * The constructor.
   *
   * @param functionMap is the underlying {@link Map} with the {@link #getFunction(String) functions}.
   */
  public DefaultPojoPathFunctionManager(Map<String, PojoPathFunction> functionMap) {

    super();
    this.functionMap = functionMap;
  }

  @Override
  public PojoPathFunction getFunction(String functionName) {

    return this.functionMap.get(functionName);
  }

  /**
   * This method registers the given {@code function} for the given {@code functionName} so it is available via
   * {@link #getFunction(String)}.
   *
   * @see #getFunction(String)
   *
   * @param function is the {@link PojoPathFunction} to register.
   * @param functionName is the {@link #getFunction(String) associated name}.
   * @return the {@link PojoPathFunction} that was registered before for the given {@code functionName} and has now been
   *         replaced.
   */
  public PojoPathFunction registerFunction(PojoPathFunction function, String functionName) {

    if (function instanceof AbstractComponent) {
      // make things easier for users, duplicate initialization will do nothing
      ((AbstractComponent) function).initialize();
    }
    return this.functionMap.put(functionName, function);
  }

}
