package com.saxonica.xqj.pull;

import net.sf.saxon.event.PipelineConfiguration;
import net.sf.saxon.expr.parser.ExplicitLocation;
import net.sf.saxon.expr.parser.Location;
import net.sf.saxon.om.*;
import net.sf.saxon.pull.PullProvider;
import net.sf.saxon.pull.UnparsedEntity;
import net.sf.saxon.trans.XPathException;
import net.sf.saxon.tree.tiny.*;
import net.sf.saxon.tree.util.AttributeCollectionImpl;
import net.sf.saxon.tree.util.NamespaceIterator;
import net.sf.saxon.type.SchemaType;
import net.sf.saxon.type.Type;
import net.sf.saxon.type.Untyped;
import net.sf.saxon.value.AtomicValue;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * This implementation of the Saxon pull interface starts from a document, element,
 * text, comment, or processing-instruction node in a TinyTree,
 * and returns the events corresponding to that node and its descendants (including
 * their attributes and namespaces). The class performs the same function as
 * the general-purpose {@link com.saxonica.xqj.pull.TreeWalker} class, but is
 * specialized to exploit the TinyTree data structure: in particular, it never
 * materializes any Node objects.
 */

public class TinyTreeWalker implements PullProvider, Location {

    private int startNode;
    private int currentNode;
    private int currentEvent;
    private TinyTree tree;
    private int[] treeNameCodes;
    private short[] treeDepth;
    private SchemaType[] treeTypes;
    private int[] treeNext;
    private int[] treeAlpha;
    private PipelineConfiguration pipe;
    private NamespaceBinding[] nsBuffer = new NamespaceBinding[10];

    /**
     * Create a TinyTreeWalker to return events associated with a tree or subtree
     *
     * @param startNode the root of the tree or subtree. Must be a document, element, text,
     *                  comment, or processing-instruction node.
     * @throws IllegalArgumentException if the start node is an attribute or namespace node.
     */

    public TinyTreeWalker(TinyNodeImpl startNode) {
        int kind = startNode.getNodeKind();
        if (kind == Type.ATTRIBUTE || kind == Type.NAMESPACE) {
            throw new IllegalArgumentException("TinyTreeWalker cannot start at an attribute or namespace node");
        }
        this.startNode = startNode.getNodeNumber();
        tree = startNode.getTree();
        treeNameCodes = tree.getNameCodeArray();
        treeNext = tree.getNextPointerArray();
        treeDepth = tree.getNodeDepthArray();
        treeTypes = tree.getTypeArray();
        treeAlpha = tree.getAlphaArray();
    }

    /**
     * Set configuration information. This must only be called before any events
     * have been read.
     */

    public void setPipelineConfiguration(PipelineConfiguration pipe) {
        this.pipe = pipe;
    }

    /**
     * Get configuration information.
     */

    public PipelineConfiguration getPipelineConfiguration() {
        return pipe;
    }

    /**
     * Get the next event
     *
     * @return an integer code indicating the type of event. The code
     *         {@link #END_OF_INPUT} is returned if there are no more events to return.
     */

    public int next() throws XPathException {
        switch (currentEvent) {
            case START_OF_INPUT:
                currentNode = startNode;
                switch (tree.nodeKind[currentNode]) {
                    case Type.DOCUMENT:
                        currentEvent = START_DOCUMENT;
                        break;
                    case Type.ELEMENT:
                    case Type.TEXTUAL_ELEMENT:
                        currentEvent = START_ELEMENT;
                        break;
                    case Type.TEXT:
                    case Type.WHITESPACE_TEXT:
                        currentEvent = TEXT;
                        break;
                    case Type.COMMENT:
                        currentEvent = COMMENT;
                        break;
                    case Type.PROCESSING_INSTRUCTION:
                        currentEvent = PROCESSING_INSTRUCTION;
                        break;
                    case Type.PARENT_POINTER:
                        throw new IllegalStateException("Current node is a parent-pointer pseudo-node");
                    case Type.EXTERNAL_NODE_REFERENCE:
                        // TODO: implement me
                        throw new UnsupportedOperationException("Pull-mode walking of grafted TinyTree is not implemented");
                }
                return currentEvent;

            case START_DOCUMENT:
            case START_ELEMENT:
                if (tree.nodeKind[currentNode] == Type.TEXTUAL_ELEMENT) {
                    // move to the implicit child text node without changing currentNode
                    return currentEvent = TEXT;
                } else if (treeDepth[currentNode + 1] > treeDepth[currentNode]) {
                    // the current element or document has children: move to the first child
                    switch (tree.nodeKind[++currentNode]) {
                        case Type.ELEMENT:
                        case Type.TEXTUAL_ELEMENT:
                            currentEvent = START_ELEMENT;
                            break;
                        case Type.TEXT:
                        case Type.WHITESPACE_TEXT:
                            currentEvent = TEXT;
                            break;
                        case Type.COMMENT:
                            currentEvent = COMMENT;
                            break;
                        case Type.PROCESSING_INSTRUCTION:
                            currentEvent = PROCESSING_INSTRUCTION;
                            break;
                        case Type.PARENT_POINTER:
                            throw new IllegalStateException("First child node must not be a parent-pointer pseudo-node");
                        case Type.EXTERNAL_NODE_REFERENCE:
                            // TODO: implement me
                            throw new UnsupportedOperationException("Pull-mode walking of grafted TinyTree is not implemented");
                    }
                    return currentEvent;
                } else {
                    if (currentEvent == START_DOCUMENT) {
                        currentEvent = END_DOCUMENT;
                    } else {
                        currentEvent = END_ELEMENT;
                    }
                    return currentEvent;
                }
            case TEXT:
                if (tree.nodeKind[currentNode] == Type.TEXTUAL_ELEMENT) {
                    return currentEvent = END_ELEMENT;
                }
                // else fall through
            case END_ELEMENT:
            case COMMENT:
            case PROCESSING_INSTRUCTION:

                    if (currentNode == startNode) {
                        currentEvent = END_OF_INPUT;
                        return currentEvent;
                    }
                    int next = treeNext[currentNode];
                    if (next > currentNode /*&& tree.nodeKind[currentNode] != Type.TEXTUAL_ELEMENT*/) {
                        // this node has a following sibling
                        currentNode = treeNext[currentNode];
                        do {
                            switch (tree.nodeKind[currentNode]) {
                                case Type.ELEMENT:
                                    currentEvent = START_ELEMENT;
                                    break;
                                case Type.TEXT:
                                case Type.WHITESPACE_TEXT:
                                    currentEvent = TEXT;
                                    break;
                                case Type.COMMENT:
                                    currentEvent = COMMENT;
                                    break;
                                case Type.PROCESSING_INSTRUCTION:
                                    currentEvent = PROCESSING_INSTRUCTION;
                                    break;
                                case Type.PARENT_POINTER:
                                    // skip this pseudo-node
                                    currentEvent = -1;
                                    currentNode++;
                                    break;
                            }
                        } while (currentEvent == -1);
                        return currentEvent;
                    } else {
                        // return to the parent element or document
                        if (next == -1) {
                            // indicates we were at the END_ELEMENT of a parentless element node
                            currentEvent = END_OF_INPUT;
                            currentNode = next;
                            return currentEvent;
                        }
                        switch (tree.nodeKind[next]) {
                            case Type.ELEMENT:
                            case Type.TEXTUAL_ELEMENT:
                                currentEvent = END_ELEMENT;
                                break;
                            case Type.DOCUMENT:
                                currentEvent = END_DOCUMENT;
                                break;
                        }
                        currentNode = next;
                        return currentEvent;
                    }


            case ATTRIBUTE:
            case NAMESPACE:
            case END_DOCUMENT:
                currentEvent = END_OF_INPUT;
                return currentEvent;

            case END_OF_INPUT:
                throw new IllegalStateException("Cannot call next() when input is exhausted");

            default:
                throw new IllegalStateException("Unrecognized event " + currentEvent);

        }
    }

    /**
     * Get the event most recently returned by next(), or by other calls that change
     * the position, for example getStringValue() and skipToMatchingEnd(). This
     * method does not change the position of the PullProvider.
     *
     * @return the current event
     */

    public int current() {
        return currentEvent;
    }

    /**
     * Get the attributes associated with the current element. This method must
     * be called only after a START_ELEMENT event has been notified. The contents
     * of the returned AttributeCollection are guaranteed to remain unchanged
     * until the next START_ELEMENT event, but may be modified thereafter. The object
     * should not be modified by the client.
     * <p>Attributes may be read before or after reading the namespaces of an element,
     * but must not be read after the first child node has been read, or after calling
     * one of the methods skipToEnd(), getStringValue(), or getTypedValue().</p>
     *
     * @return an AttributeCollection representing the attributes of the element
     *         that has just been notified.
     */

    public AttributeCollection getAttributes() {
        if ((tree.nodeKind[currentNode] & 0x0f) == Type.ELEMENT) {
            if (treeAlpha[currentNode] == -1) {
                return AttributeCollectionImpl.EMPTY_ATTRIBUTE_COLLECTION;
            }
            return new TinyAttributeCollection(tree, currentNode);
        } else {
            throw new IllegalStateException("getAttributes() called when current event is not ELEMENT_START");
        }
    }

    /**
     * Get the namespace declarations associated with the current element. This method must
     * be called only after a START_ELEMENT event has been notified. In the case of a top-level
     * START_ELEMENT event (that is, an element that either has no parent node, or whose parent
     * is not included in the sequence being read), the NamespaceDeclarations object returned
     * will contain a namespace declaration for each namespace that is in-scope for this element
     * node. In the case of a non-top-level element, the NamespaceDeclarations will contain
     * a set of namespace declarations and undeclarations, representing the differences between
     * this element and its parent.
     * <p>It is permissible for this method to return namespace declarations that are redundant.</p>
     * <p>The NamespaceDeclarations object is guaranteed to remain unchanged until the next START_ELEMENT
     * event, but may then be overwritten. The object should not be modified by the client.</p>
     * <p>Namespaces may be read before or after reading the attributes of an element,
     * but must not be read after the first child node has been read, or after calling
     * one of the methods skipToEnd(), getStringValue(), or getTypedValue().</p>
     */

    public NamespaceBinding[] getNamespaceDeclarations() {
        if ((tree.nodeKind[currentNode] & 0x0f) == Type.ELEMENT) {
            if (currentNode == startNode) {
                // get all inscope namespaces for a top-level element in the sequence.
                List<NamespaceBinding> list = new ArrayList<>();
                Iterator<NamespaceBinding> iter = NamespaceIterator.iterateNamespaces(tree.getNode(currentNode));
                iter.forEachRemaining(list::add);
                return list.toArray(NamespaceBinding.EMPTY_ARRAY);
            } else {
                // only namespace declarations (and undeclarations) on this element are required
                return TinyElementImpl.getDeclaredNamespaces(tree, currentNode, nsBuffer);
            }
        }
        throw new IllegalStateException("getNamespaceDeclarations() called when current event is not START_ELEMENT");
    }

    /**
     * Skip the current subtree. This method may be called only immediately after
     * a START_DOCUMENT or START_ELEMENT event. This call returns the matching
     * END_DOCUMENT or END_ELEMENT event; the next call on next() will return
     * the event following the END_DOCUMENT or END_ELEMENT.
     */

    public int skipToMatchingEnd() throws XPathException {
        // For this implementation, we simply leave the current node unchanged, and change
        // the current event
        switch (currentEvent) {
            case START_DOCUMENT:
                currentEvent = END_DOCUMENT;
                return currentEvent;
            case START_ELEMENT:
                currentEvent = END_ELEMENT;
                return currentEvent;
            default:
                throw new IllegalStateException(
                        "Cannot call skipToMatchingEnd() except when at start of element or document");

        }
    }

    /**
     * Close the event reader. This indicates that no further events are required.
     * It is not necessary to close an event reader after {@link #END_OF_INPUT} has
     * been reported, but it is recommended to close it if reading terminates
     * prematurely. Once an event reader has been closed, the effect of further
     * calls on next() is undefined.
     */

    public void close() {
        // no action
    }

    /**
     * Get the namePool used to lookup all name codes and namespace codes
     *
     * @return the namePool
     */

    public NamePool getNamePool() {
        return pipe.getConfiguration().getNamePool();
    }

    /**
     * Get a nodeName identifying the name of the current node. This method
     * can be used after the {@link #START_ELEMENT}, {@link #PROCESSING_INSTRUCTION},
     * {@link #ATTRIBUTE}, or {@link #NAMESPACE} events. With some PullProvider implementations,
     * including this one, it can also be used after {@link #END_ELEMENT}.
     * If called at other times, the result is undefined and may result in an IllegalStateException.
     * If called when the current node is an unnamed namespace node (a node representing the default namespace)
     * the returned value is null.
     *
     * @return the node name.
     */

    public NodeName getNodeName() {
        switch (currentEvent) {
            case START_ELEMENT:
            case PROCESSING_INSTRUCTION:
            case END_ELEMENT:
                return new CodedName(treeNameCodes[currentNode], "", getNamePool());
            default:
                throw new IllegalStateException("getNodeName() called when its value is undefined");
        }

    }

    /**
     * Get the string value of the current attribute, text node, processing-instruction,
     * or atomic value.
     * This method cannot be used to obtain the string value of an element, or of a namespace
     * node. If the most recent event was anything other than {@link #START_ELEMENT}, {@link #TEXT},
     * {@link #PROCESSING_INSTRUCTION}, or {@link #ATOMIC_VALUE}, the result is undefined.
     */

    public CharSequence getStringValue() {
        switch (tree.nodeKind[currentNode]) {
            case Type.TEXT:
                return TinyTextImpl.getStringValue(tree, currentNode);
            case Type.WHITESPACE_TEXT:
                return WhitespaceTextImpl.getStringValueCS(tree, currentNode);
            case Type.COMMENT:
            case Type.PROCESSING_INSTRUCTION:
                // sufficiently rare that instantiating the node is OK
                return tree.getNode(currentNode).getStringValue();
            case Type.ELEMENT:
            case Type.TEXTUAL_ELEMENT:
                currentEvent = END_ELEMENT;
                return TinyParentNodeImpl.getStringValueCS(tree, currentNode);
            case Type.PARENT_POINTER:
                throw new IllegalStateException("Trying to get string value of a parent-pointer pseudo node");
        }
        return null;
    }

    /**
     * Get an atomic value. This call may be used only when the last event reported was
     * ATOMIC_VALUE. This indicates that the PullProvider is reading a sequence that contains
     * a free-standing atomic value; it is never used when reading the content of a node.
     */

    public AtomicValue getAtomicValue() {
        throw new IllegalStateException();
    }

    /**
     * Get the type annotation of the current attribute or element node, or atomic value.
     * The result of this method is undefined unless the most recent event was START_ELEMENT,
     * START_CONTENT, ATTRIBUTE, or ATOMIC_VALUE.
     *
     * @return the type code. This code is the fingerprint of a type name, which may be
     *         resolved to a {@link net.sf.saxon.type.SchemaType} by access to the Configuration.
     */

    public SchemaType getSchemaType() {
        if (tree.nodeKind[currentNode] != Type.ELEMENT) {
            throw new IllegalStateException("getTypeAnnotation() called when current event is not ELEMENT_START or ");
        }
        if (treeTypes == null) {
            return Untyped.getInstance();
        }
        return treeTypes[currentNode];
    }

    /**
     * Get the location of the current event.
     * For an event stream representing a real document, the location information
     * should identify the location in the lexical XML source. For a constructed document, it should
     * identify the location in the query or stylesheet that caused the node to be created.
     * A value of null can be returned if no location information is available.
     */

    public Location getSourceLocator() {
        return this;
    }

    /**
     * Return the public identifier for the current document event.
     * <p>The return value is the public identifier of the document
     * entity or of the external parsed entity in which the markup that
     * triggered the event appears.</p>
     *
     * @return A string containing the public identifier, or
     *         null if none is available.
     * @see #getSystemId
     */
    public String getPublicId() {
        return null;
    }

    /**
     * Return the system identifier for the current document event.
     * <p>The return value is the system identifier of the document
     * entity or of the external parsed entity in which the markup that
     * triggered the event appears.</p>
     * <p>If the system identifier is a URL, the parser must resolve it
     * fully before passing it to the application.</p>
     *
     * @return A string containing the system identifier, or null
     *         if none is available.
     * @see #getPublicId
     */
    public String getSystemId() {
        return tree.getSystemId(currentNode);
    }

    /**
     * Return the line number where the current document event ends.
     * <p><strong>Warning:</strong> The return value from the method
     * is intended only as an approximation for the sake of error
     * reporting; it is not intended to provide sufficient information
     * to edit the character content of the original XML document.</p>
     * <p>The return value is an approximation of the line number
     * in the document entity or external parsed entity where the
     * markup that triggered the event appears.</p>
     *
     * @return The line number, or -1 if none is available.
     * @see #getColumnNumber
     */
    public int getLineNumber() {
        return tree.getLineNumber(currentNode);
    }

    /**
     * Return the character position where the current document event ends.
     * <p><strong>Warning:</strong> The return value from the method
     * is intended only as an approximation for the sake of error
     * reporting; it is not intended to provide sufficient information
     * to edit the character content of the original XML document.</p>
     * <p>The return value is an approximation of the column number
     * in the document entity or external parsed entity where the
     * markup that triggered the event appears.</p>
     *
     * @return The column number, or -1 if none is available.
     * @see #getLineNumber
     */
    public int getColumnNumber() {
        return -1;
    }

    /**
     * Get an immutable copy of this Location object. By default Location objects may be mutable, so they
     * should not be saved for later use. The result of this operation holds the same location information,
     * but in an immutable form.
     */

    public Location saveLocation() {
        return new ExplicitLocation(getSystemId(), getLineNumber(), getColumnNumber());
    }

    /**
     * Get a list of unparsed entities.
     *
     * @return a list of unparsed entities, or null if the information is not available, or
     *         an empty list if there are no unparsed entities.
     */

    /*@Nullable*/
    public List<UnparsedEntity> getUnparsedEntities() {
        return null;
    }

//    public static void main(String[] args) throws Exception {
//
//        Controller controller = new Controller(new Configuration());
//        TinyBuilder tb = (TinyBuilder)controller.makeBuilder();
//        tb.open();
//        NamePool p = tb.getConfiguration().getNamePool();
//        int code = p.allocate("a", "b", "c");
//        tb.startElement(code, -1, -1, -1);
//        tb.endElement();
//        tb.startDocument(-1);
//        tb.startElement(code, -1, -1, -1);
//        tb.endElement();
//        tb.endDocument();
//        tb.close();
//        NodeInfo node = tb.getCurrentRoot();
//
//        TinyTreeWalker walker = new TinyTreeWalker((TinyNodeImpl)node);
//        System.out.println(walker.next());
//        System.out.println(walker.next());
//        System.out.println(walker.next());
//        System.out.println(walker.next());
//        System.out.println(walker.next());
//    }
}

// Copyright (c) 2009 Saxonica Limited.
