/*
 * Copyright (c) 2014 Snowflake Computing Inc. All right reserved.
 */
package net.snowflake.common.core;

import com.fasterxml.jackson.annotation.JsonIgnore;
import java.util.EnumSet;

import java.util.Map;

/**
 *
 * @author Johnston Chu
 *
 */

public class ClientAuthnDTO implements SubmittedAuthnDTO
{

  public enum RequestType
  {

    RENEW, // Renew an expired session token
    CLONE, // Clone an existing session, and its token
    ISSUE  // Issue new session token based on rememberme token

  }
  
  public enum ReauthnType
  {
    USERNAME_PASSWORD,
    MFA,
    FEDERATED,
    FEDERATED_MFA,
  }
  
  public enum AuthenticatorType
  {
    /*
     * regular login username+password via Snowflake, may or may not have MFA
     */
    SNOWFLAKE,
    
    /*
     * federated authentication, OKTA as IDP
     */
    OKTA,

    /*
     * Web browser based authenticator for SAML 2.0 compliant
     * service/application
     */
    EXTERNALBROWSER,

    /*
     * OAUTH 2.0 flow
     */
    OAUTH,
    
    /*
     * Snowflake local authentication using jwt token as a user credential
     */
    //TODO rename other snowflake local authenticator types
    SNOWFLAKE_JWT
  }

  // contains all the required data for current authn step
  private Map<String, Object> data;

  /*
   * current state
   * tokenized string with all current parameters and the authn step
   */
  private String inFlightCtx;

//  public static final String SF_ODBC_APP_ID = "ODBC";
//  public static final String SF_JDBC_APP_ID = "JDBC";
//  public static final String SF_CONSOLE_APP_ID = "Snowflake UI";

  private static final ResourceBundleManager versionResourceBundleManager =
      ResourceBundleManager.getSingleton("net.snowflake.common.version");

  /**
   * Required by Jackson
   */
  public ClientAuthnDTO()
  {
  }

  public Map<String, Object> getData()
  {
    return data;
  }

  public void setData(Map<String, Object> data)
  {
    this.data = data;
  }

  @Override
  public String getInFlightCtx()
  {
    return inFlightCtx;
  }

  public void setInFlightCtx(String inFlightCtx)
  {
    this.inFlightCtx = inFlightCtx;
  }

  @JsonIgnore
  public String getLoginName()
  {
    return getParamAsString(ClientAuthnParameter.LOGIN_NAME.name());
  }

  @JsonIgnore
  public String getPassword()
  {
    return getParamAsString(ClientAuthnParameter.PASSWORD.name());
  }

  @JsonIgnore
  public String getSamlResponse()
  {
    return getParamAsString(ClientAuthnParameter.SAML_RESPONSE.name());
  }

  @JsonIgnore
  public String getRawSamlResponse()
  {
    return getParamAsString(ClientAuthnParameter.RAW_SAML_RESPONSE.name());
  }

  @JsonIgnore
  public String getAccountName()
  {
    return getParamAsString(ClientAuthnParameter.ACCOUNT_NAME.name());
  }

  @JsonIgnore
  public String getClientAppId()
  {
    return getParamAsNonNullString(ClientAuthnParameter.CLIENT_APP_ID.name());
  }

  @JsonIgnore
  public String getClientAppVersion()
  {
    return getParamAsNonNullString(
        ClientAuthnParameter.CLIENT_APP_VERSION.name());
  }

  @JsonIgnore
  public String getChosenNewPassword()
  {
    return getParamAsString(ClientAuthnParameter.CHOSEN_NEW_PASSWORD.name());
  }

  @JsonIgnore
  public String getClientBuildId()
  {
    // return client build id if set
    String clientBuildId = getParamAsString(
            ClientAuthnParameter.CLIENT_BUILD_ID.name());

    if (clientBuildId != null)
    {
      return clientBuildId;
    }

    // if client build id is not set, use svn revision for backward
    // compatibility
    return getParamAsString(ClientAuthnParameter.SVN_REVISION.name());
  }

  @JsonIgnore
  public static String getLatestJDBCAppVersion()
  {
    return versionResourceBundleManager.getLocalizedMessage("jdbc.version");
  }

  @JsonIgnore
  public static String getLatestODBCAppVersion()
  {
    return versionResourceBundleManager.getLocalizedMessage("odbc.version");
  }

  @JsonIgnore
  public boolean isLicenseAccepted()
  {
    Object o = getParam(ClientAuthnParameter.LICENSE_ACCEPTED.name());
    if (o != null)
    {
      return (Boolean) o;
    }
    else
    {
      return false;
    }
  }

  @JsonIgnore
  public String getExtAuthnMethod()
  {
    return getParamAsString(ClientAuthnParameter.EXT_AUTHN_DUO_METHOD.name());
  }

  @JsonIgnore
  public String getPasscode()
  {
    return getParamAsString(ClientAuthnParameter.PASSCODE.name());
  }

  @JsonIgnore
  public String getOldMasterToken()
  {
    return getParamAsString(ClientAuthnParameter.OLD_MASTER_TOKEN.name());
  }

  @JsonIgnore
  public String getOldSessionToken()
  {
    return getParamAsString(ClientAuthnParameter.OLD_SESSION_TOKEN.name());
  }

  @JsonIgnore
  public RequestType getRequestType()
  {
    Object o = getParam(ClientAuthnParameter.REQUEST_TYPE.name());
    return valueOf(RequestType.class, o);
  }

  @JsonIgnore
  public ReauthnType getReauthnType()
  {
    Object o = getParam(ClientAuthnParameter.REAUTHENTICATION_TYPE.name());
    return valueOf(ReauthnType.class, o);
  }

  @JsonIgnore
  public String getAuthnRequestId()
  {
    return getParamAsString(ClientAuthnParameter.AUTHENTICATION_REQUEST_ID.name());
  }

  @JsonIgnore
  public String getAuthnMethod()
  {
    return getParamAsString(ClientAuthnParameter.AUTHENTICATION_METHOD.name());
  }

  @JsonIgnore
  public Map<String, Object> getClientEnvironment()
  {
    // return client environment if set
    return (Map<String, Object>)
                       getParam(ClientAuthnParameter.CLIENT_ENVIRONMENT.name());
  }

  @JsonIgnore
  public Map<String, Object> getSessionParameters()
  {
    // return session parameters if set
    Object sessionParams = getParam(ClientAuthnParameter.SESSION_PARAMETERS.name());
    if (sessionParams instanceof Map)
    {
      return (Map<String, Object>)sessionParams;
    }
    return null;
  }
  
  @JsonIgnore
  public String getReauthPendingCount()
  {
    return getParamAsString(ClientAuthnParameter.FED_REAUTH_PENDING_COUNT.name());
  }

  @JsonIgnore
  public String getAuthenticator()
  {
    return getParamAsString(ClientAuthnParameter.AUTHENTICATOR.name());
  }

  @JsonIgnore
  public String getBrowserModeRedirectPort()
  {
    return getParamAsString(ClientAuthnParameter.BROWSER_MODE_REDIRECT_PORT.name());
  }

  @JsonIgnore
  public String getProofKey()
  {
    return getParamAsString(ClientAuthnParameter.PROOF_KEY.name());
  }

  @JsonIgnore
  public String getToken()
  {
    return getParamAsString(ClientAuthnParameter.TOKEN.name());
  }

  /**
   * Gets the OAuth Federated CTX, if present.
   */
  @JsonIgnore
  public String getOAuthFederatedCtx()
  {
    return getParamAsString(ClientAuthnParameter.OAUTH_FEDERATED_CTX.name());
  }

  /**
   * Gets the Relay State, if present.
   */
  @JsonIgnore
  public String getRelayState()
  {
    return getParamAsString(ClientAuthnParameter.RELAY_STATE.name());
  }

  private String getParamAsString(String param)
  {
    Object obj = getParam(param);
    if (obj != null)
    {
      return obj.toString();
    }
    else
    {
      return null;
    }
  }

  private String getParamAsNonNullString(String param)
  {
    // not using Optional (JDK1.8) in common
    String s = getParamAsString(param);
    return s != null ? s : "";
  }

  private Object getParam(String param)
  {
    if (data != null)
    {
      return data.get(param);
    }
    return null;
  }
  
  private <T extends Enum<T>> T valueOf(Class<T> enumType, 
                                        Object o)
  {
    if (o != null)
    {
      try
      {
        return T.valueOf(enumType, o.toString());
      }
      catch (IllegalArgumentException e)
      {
        return null;
      }
    }
    return null;
  }

}
