/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.ingest.streaming.internal;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import net.snowflake.client.jdbc.internal.google.common.collect.Sets;
import net.snowflake.client.jdbc.internal.snowflake.common.util.Power10;
import net.snowflake.ingest.streaming.internal.BinaryStringUtils;
import net.snowflake.ingest.streaming.internal.TimestampWrapper;
import net.snowflake.ingest.streaming.internal.serialization.ByteArraySerializer;
import net.snowflake.ingest.streaming.internal.serialization.ZonedDateTimeSerializer;
import net.snowflake.ingest.utils.ErrorCode;
import net.snowflake.ingest.utils.SFException;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;

class DataValidationUtil {
    private static final long SECONDS_LIMIT_FOR_EPOCH = 31536000000L;
    private static final long MILLISECONDS_LIMIT_FOR_EPOCH = 31536000000000L;
    private static final long MICROSECONDS_LIMIT_FOR_EPOCH = 31536000000000000L;
    public static final int BYTES_8_MB = 0x800000;
    public static final int BYTES_16_MB = 0x1000000;
    static final int MAX_SEMI_STRUCTURED_LENGTH = 0xFFFFC0;
    private static final ObjectMapper objectMapper = new ObjectMapper();
    static Set<String> allowedBooleanStringsLowerCased;

    DataValidationUtil() {
    }

    private static JsonNode validateAndParseSemiStructuredAsJsonTree(String columnName, Object input, String snowflakeType) {
        if (input instanceof String) {
            String stringInput = (String)input;
            DataValidationUtil.verifyValidUtf8(stringInput, columnName, snowflakeType);
            try {
                return objectMapper.readTree(stringInput);
            }
            catch (JsonProcessingException e) {
                throw DataValidationUtil.valueFormatNotAllowedException(columnName, input, snowflakeType, "Not a valid JSON");
            }
        }
        if (DataValidationUtil.isAllowedSemiStructuredType(input)) {
            return objectMapper.valueToTree(input);
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), snowflakeType, new String[]{"String", "Primitive data types and their arrays", "java.time.*", "List<T>", "Map<String, T>", "T[]"});
    }

    static String validateAndParseVariant(String columnName, Object input) {
        JsonNode node = DataValidationUtil.validateAndParseSemiStructuredAsJsonTree(columnName, input, "VARIANT");
        if (node.isMissingNode()) {
            return null;
        }
        String output = node.toString();
        int stringLength = output.getBytes(StandardCharsets.UTF_8).length;
        if (stringLength > 0xFFFFC0) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, output, "VARIANT", String.format("Variant too long: length=%d maxLength=%d", stringLength, 0xFFFFC0));
        }
        return output;
    }

    static boolean isAllowedSemiStructuredType(Object o) {
        if (o == null) {
            return true;
        }
        if (o instanceof String) {
            return true;
        }
        if (o instanceof Long || o instanceof Integer || o instanceof Short || o instanceof Byte || o instanceof Float || o instanceof Double || o instanceof Boolean || o instanceof Character) {
            return true;
        }
        if (o instanceof BigInteger || o instanceof BigDecimal) {
            return true;
        }
        if (o instanceof LocalTime || o instanceof OffsetTime || o instanceof LocalDate || o instanceof LocalDateTime || o instanceof ZonedDateTime || o instanceof OffsetDateTime) {
            return true;
        }
        if (o instanceof Map) {
            boolean allKeysAreStrings = ((Map)o).keySet().stream().allMatch(x -> x instanceof String);
            if (!allKeysAreStrings) {
                return false;
            }
            boolean allValuesAreAllowed = ((Map)o).values().stream().allMatch(DataValidationUtil::isAllowedSemiStructuredType);
            return allValuesAreAllowed;
        }
        if (o instanceof byte[] || o instanceof short[] || o instanceof int[] || o instanceof long[] || o instanceof float[] || o instanceof double[] || o instanceof boolean[] || o instanceof char[]) {
            return true;
        }
        if (o.getClass().isArray()) {
            return Arrays.stream((Object[])o).allMatch(DataValidationUtil::isAllowedSemiStructuredType);
        }
        if (o instanceof List) {
            return ((List)o).stream().allMatch(DataValidationUtil::isAllowedSemiStructuredType);
        }
        return false;
    }

    static String validateAndParseArray(String columnName, Object input) {
        String output;
        int stringLength;
        JsonNode jsonNode = DataValidationUtil.validateAndParseSemiStructuredAsJsonTree(columnName, input, "ARRAY");
        if (!jsonNode.isArray()) {
            jsonNode = objectMapper.createArrayNode().add(jsonNode);
        }
        if ((stringLength = (output = jsonNode.toString()).getBytes(StandardCharsets.UTF_8).length) > 0xFFFFC0) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, output, "ARRAY", String.format("Array too large. length=%d maxLength=%d", stringLength, 0xFFFFC0));
        }
        return output;
    }

    static String validateAndParseObject(String columnName, Object input) {
        JsonNode jsonNode = DataValidationUtil.validateAndParseSemiStructuredAsJsonTree(columnName, input, "OBJECT");
        if (!jsonNode.isObject()) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, jsonNode, "OBJECT", "Not an object");
        }
        String output = jsonNode.toString();
        int stringLength = output.getBytes(StandardCharsets.UTF_8).length;
        if (stringLength > 0xFFFFC0) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, output, "OBJECT", String.format("Object too large. length=%d maxLength=%d", stringLength, 0xFFFFC0));
        }
        return output;
    }

    private static OffsetDateTime inputToOffsetDateTime(String columnName, String typeName, Object input, ZoneId defaultTimezone) {
        if (input instanceof OffsetDateTime) {
            return (OffsetDateTime)input;
        }
        if (input instanceof ZonedDateTime) {
            return ((ZonedDateTime)input).toOffsetDateTime();
        }
        if (input instanceof LocalDateTime) {
            return ((LocalDateTime)input).atZone(defaultTimezone).toOffsetDateTime();
        }
        if (input instanceof LocalDate) {
            return ((LocalDate)input).atStartOfDay().atZone(defaultTimezone).toOffsetDateTime();
        }
        if (input instanceof Instant) {
            return ((Instant)input).atZone(ZoneOffset.UTC).toOffsetDateTime();
        }
        if (input instanceof String) {
            String stringInput = ((String)input).trim();
            ZonedDateTime zoned = DataValidationUtil.catchParsingError(() -> ZonedDateTime.parse(stringInput));
            if (zoned != null) {
                return zoned.toOffsetDateTime();
            }
            OffsetDateTime offset = DataValidationUtil.catchParsingError(() -> OffsetDateTime.parse(stringInput));
            if (offset != null) {
                return offset;
            }
            LocalDateTime localDateTime = DataValidationUtil.catchParsingError(() -> LocalDateTime.parse(stringInput));
            if (localDateTime != null) {
                return localDateTime.atZone(defaultTimezone).toOffsetDateTime();
            }
            LocalDate localDate = DataValidationUtil.catchParsingError(() -> LocalDate.parse(stringInput));
            if (localDate != null) {
                return localDate.atStartOfDay().atZone(defaultTimezone).toOffsetDateTime();
            }
            Instant instant = DataValidationUtil.catchParsingError(() -> DataValidationUtil.parseInstantGuessScale(stringInput));
            if (instant != null) {
                return instant.atOffset(ZoneOffset.UTC);
            }
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, input.toString(), typeName, "Not a valid value, see https://docs.snowflake.com/en/LIMITEDACCESS/snowpipe-streaming.html for the list of supported formats");
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), typeName, new String[]{"String", "LocalDate", "LocalDateTime", "ZonedDateTime", "OffsetDateTime"});
    }

    private static <T> T catchParsingError(Supplier<T> op) {
        try {
            return op.get();
        }
        catch (NumberFormatException | DateTimeParseException e) {
            return null;
        }
    }

    static TimestampWrapper validateAndParseTimestamp(String columnName, Object input, int scale, ZoneId defaultTimezone, boolean trimTimezone) {
        OffsetDateTime offsetDateTime = DataValidationUtil.inputToOffsetDateTime(columnName, "TIMESTAMP", input, defaultTimezone);
        if (trimTimezone) {
            offsetDateTime = offsetDateTime.withOffsetSameLocal(ZoneOffset.UTC);
        }
        return new TimestampWrapper(offsetDateTime, scale);
    }

    static String validateAndParseString(String columnName, Object input, Optional<Integer> maxLengthOptional) {
        String output;
        if (input instanceof String) {
            output = (String)input;
            DataValidationUtil.verifyValidUtf8(output, columnName, "STRING");
        } else if (input instanceof Number) {
            output = new BigDecimal(input.toString()).stripTrailingZeros().toPlainString();
        } else if (input instanceof Boolean || input instanceof Character) {
            output = input.toString();
        } else {
            throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "STRING", new String[]{"String", "Number", "boolean", "char"});
        }
        byte[] utf8Bytes = output.getBytes(StandardCharsets.UTF_8);
        if (utf8Bytes.length > 0x1000000) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, input, "STRING", String.format("String too long: length=%d bytes maxLength=%d bytes", utf8Bytes.length, 0x1000000));
        }
        maxLengthOptional.ifPresent(maxAllowedCharacters -> {
            int actualCharacters = BinaryStringUtils.unicodeCharactersCount(output);
            if (actualCharacters > maxAllowedCharacters) {
                throw DataValidationUtil.valueFormatNotAllowedException(columnName, input, "STRING", String.format("String too long: length=%d characters maxLength=%d characters", actualCharacters, maxAllowedCharacters));
            }
        });
        return output;
    }

    static BigDecimal validateAndParseBigDecimal(String columnName, Object input) {
        if (input instanceof BigDecimal) {
            return (BigDecimal)input;
        }
        if (input instanceof BigInteger) {
            return new BigDecimal((BigInteger)input);
        }
        if (input instanceof Byte || input instanceof Short || input instanceof Integer || input instanceof Long) {
            return BigDecimal.valueOf(((Number)input).longValue());
        }
        if (input instanceof Float || input instanceof Double) {
            return BigDecimal.valueOf(((Number)input).doubleValue());
        }
        if (input instanceof String) {
            try {
                String stringInput = ((String)input).trim();
                return new BigDecimal(stringInput);
            }
            catch (NumberFormatException e) {
                throw DataValidationUtil.valueFormatNotAllowedException(columnName, input, "NUMBER", "Not a valid number");
            }
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "NUMBER", new String[]{"int", "long", "byte", "short", "float", "double", "BigDecimal", "BigInteger", "String"});
    }

    static int validateAndParseDate(String columnName, Object input) {
        OffsetDateTime offsetDateTime = DataValidationUtil.inputToOffsetDateTime(columnName, "DATE", input, ZoneOffset.UTC);
        return Math.toIntExact(offsetDateTime.toLocalDate().toEpochDay());
    }

    static byte[] validateAndParseBinary(String columnName, Object input, Optional<Integer> maxLengthOptional) {
        byte[] output;
        if (input instanceof byte[]) {
            byte[] originalInputArray = (byte[])input;
            output = new byte[originalInputArray.length];
            System.arraycopy(originalInputArray, 0, output, 0, originalInputArray.length);
        } else if (input instanceof String) {
            try {
                String stringInput = ((String)input).trim();
                output = Hex.decodeHex((String)stringInput);
            }
            catch (DecoderException e) {
                throw DataValidationUtil.valueFormatNotAllowedException(columnName, input, "BINARY", "Not a valid hex string");
            }
        } else {
            throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "BINARY", new String[]{"byte[]", "String"});
        }
        int maxLength = maxLengthOptional.orElse(0x800000);
        if (output.length > maxLength) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, String.format("byte[%d]", output.length), "BINARY", String.format("Binary too long: length=%d maxLength=%d", output.length, maxLength));
        }
        return output;
    }

    static BigInteger validateAndParseTime(String columnName, Object input, int scale) {
        if (input instanceof LocalTime) {
            LocalTime localTime = (LocalTime)input;
            return BigInteger.valueOf(localTime.toNanoOfDay()).divide(Power10.sb16Table[9 - scale]);
        }
        if (input instanceof OffsetTime) {
            return DataValidationUtil.validateAndParseTime(columnName, ((OffsetTime)input).toLocalTime(), scale);
        }
        if (input instanceof String) {
            String stringInput = ((String)input).trim();
            LocalTime localTime = DataValidationUtil.catchParsingError(() -> LocalTime.parse(stringInput));
            if (localTime != null) {
                return DataValidationUtil.validateAndParseTime(columnName, localTime, scale);
            }
            OffsetTime offsetTime = DataValidationUtil.catchParsingError(() -> OffsetTime.parse(stringInput));
            if (offsetTime != null) {
                return DataValidationUtil.validateAndParseTime(columnName, offsetTime.toLocalTime(), scale);
            }
            Instant parsedInstant = DataValidationUtil.catchParsingError(() -> DataValidationUtil.parseInstantGuessScale(stringInput));
            if (parsedInstant != null) {
                return DataValidationUtil.validateAndParseTime(columnName, LocalDateTime.ofInstant(parsedInstant, ZoneOffset.UTC).toLocalTime(), scale);
            }
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, input, "TIME", "Not a valid time, see https://docs.snowflake.com/en/LIMITEDACCESS/snowpipe-streaming.html for the list of supported formats");
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "TIME", new String[]{"String", "LocalTime", "OffsetTime"});
    }

    private static Instant parseInstantGuessScale(String input) {
        long val = Long.parseLong(input);
        long epochNanos = val > -31536000000L && val < 31536000000L ? val * (long)Power10.intTable[9] : (val > -31536000000000L && val < 31536000000000L ? val * (long)Power10.intTable[6] : (val > -31536000000000000L && val < 31536000000000000L ? val * (long)Power10.intTable[3] : val));
        return Instant.ofEpochSecond(epochNanos / (long)Power10.intTable[9], epochNanos % (long)Power10.intTable[9]);
    }

    static double validateAndParseReal(String columnName, Object input) {
        if (input instanceof Float) {
            return Double.parseDouble(input.toString());
        }
        if (input instanceof Number) {
            return ((Number)input).doubleValue();
        }
        if (input instanceof String) {
            String stringInput = ((String)input).trim();
            try {
                return Double.parseDouble(stringInput);
            }
            catch (NumberFormatException err) {
                switch (stringInput = stringInput.toLowerCase()) {
                    case "nan": {
                        return Double.NaN;
                    }
                    case "inf": {
                        return Double.POSITIVE_INFINITY;
                    }
                    case "-inf": {
                        return Double.NEGATIVE_INFINITY;
                    }
                }
                throw DataValidationUtil.valueFormatNotAllowedException(columnName, input, "REAL", "Not a valid decimal number");
            }
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "REAL", new String[]{"Number", "String"});
    }

    static int validateAndParseBoolean(String columnName, Object input) {
        if (input instanceof Boolean) {
            return (Boolean)input != false ? 1 : 0;
        }
        if (input instanceof Number) {
            return new BigDecimal(input.toString()).compareTo(BigDecimal.ZERO) == 0 ? 0 : 1;
        }
        if (input instanceof String) {
            return DataValidationUtil.convertStringToBoolean(columnName, (String)input) ? 1 : 0;
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "BOOLEAN", new String[]{"boolean", "Number", "String"});
    }

    static void checkValueInRange(BigDecimal bigDecimalValue, int scale, int precision) {
        if (bigDecimalValue.abs().compareTo(BigDecimal.TEN.pow(precision - scale)) >= 0) {
            throw new SFException(ErrorCode.INVALID_ROW, bigDecimalValue, String.format("Number out of representable exclusive range of (-1e%s..1e%s)", precision - scale, precision - scale));
        }
    }

    private static boolean convertStringToBoolean(String columnName, String value) {
        String normalizedInput = value.toLowerCase().trim();
        if (!allowedBooleanStringsLowerCased.contains(normalizedInput)) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, value, "BOOLEAN", "Not a valid boolean, see https://docs.snowflake.com/en/sql-reference/data-types-logical.html#conversion-to-boolean for the list of supported formats");
        }
        return "1".equals(normalizedInput) || "yes".equals(normalizedInput) || "y".equals(normalizedInput) || "t".equals(normalizedInput) || "true".equals(normalizedInput) || "on".equals(normalizedInput);
    }

    private static SFException typeNotAllowedException(String columnName, Class<?> javaType, String snowflakeType, String[] allowedJavaTypes) {
        return new SFException(ErrorCode.INVALID_ROW, String.format("Object of type %s cannot be ingested into Snowflake column %s of type %s", javaType.getName(), columnName, snowflakeType), String.format(String.format("Allowed Java types: %s", String.join((CharSequence)", ", allowedJavaTypes)), new Object[0]));
    }

    private static SFException valueFormatNotAllowedException(String columnName, Object value, String snowflakeType, String reason) {
        return new SFException(ErrorCode.INVALID_ROW, DataValidationUtil.sanitizeValueForExceptionMessage(value), String.format("Value cannot be ingested into Snowflake column %s of type %s: %s", columnName, snowflakeType, reason));
    }

    private static String sanitizeValueForExceptionMessage(Object value) {
        int maxSize = 20;
        String valueString = value.toString();
        return valueString.length() <= maxSize ? valueString : valueString.substring(0, 20) + "...";
    }

    private static void verifyValidUtf8(String input, String columnName, String dataType) {
        CharsetEncoder charsetEncoder = StandardCharsets.UTF_8.newEncoder().onMalformedInput(CodingErrorAction.REPORT).onUnmappableCharacter(CodingErrorAction.REPORT);
        try {
            charsetEncoder.encode(CharBuffer.wrap(input));
        }
        catch (CharacterCodingException e) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, input, dataType, "Invalid Unicode string");
        }
    }

    static {
        SimpleModule module = new SimpleModule();
        module.addSerializer(byte[].class, (JsonSerializer)new ByteArraySerializer());
        module.addSerializer(ZonedDateTime.class, (JsonSerializer)new ZonedDateTimeSerializer());
        module.addSerializer(LocalTime.class, (JsonSerializer)new ToStringSerializer());
        module.addSerializer(OffsetTime.class, (JsonSerializer)new ToStringSerializer());
        module.addSerializer(LocalDate.class, (JsonSerializer)new ToStringSerializer());
        module.addSerializer(LocalDateTime.class, (JsonSerializer)new ToStringSerializer());
        module.addSerializer(OffsetDateTime.class, (JsonSerializer)new ToStringSerializer());
        objectMapper.registerModule((Module)module);
        allowedBooleanStringsLowerCased = Sets.newHashSet((Object[])new String[]{"1", "0", "yes", "no", "y", "n", "t", "f", "true", "false", "on", "off"});
    }
}

