/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.ingest.internal.apache.parquet.hadoop;

import java.lang.management.ManagementFactory;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import net.snowflake.ingest.internal.apache.parquet.hadoop.InternalParquetRecordWriter;
import net.snowflake.ingest.internal.apache.parquet.hadoop.ParquetMemoryManagerRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemoryManager {
    private static final Logger LOG = LoggerFactory.getLogger(MemoryManager.class);
    static final float DEFAULT_MEMORY_POOL_RATIO = 0.95f;
    static final long DEFAULT_MIN_MEMORY_ALLOCATION = 0x100000L;
    private final float memoryPoolRatio;
    private final long totalMemoryPool;
    private final long minMemoryAllocation;
    private final Map<InternalParquetRecordWriter<?>, Long> writerList = new HashMap();
    private final Map<String, Runnable> callBacks = new HashMap<String, Runnable>();
    private double scale = 1.0;

    public MemoryManager(float ratio, long minAllocation) {
        this.checkRatio(ratio);
        this.memoryPoolRatio = ratio;
        this.minMemoryAllocation = minAllocation;
        this.totalMemoryPool = Math.round((double)ManagementFactory.getMemoryMXBean().getHeapMemoryUsage().getMax() * (double)ratio);
        LOG.debug("Allocated total memory pool is: {}", (Object)this.totalMemoryPool);
    }

    private void checkRatio(float ratio) {
        if (ratio <= 0.0f || ratio > 1.0f) {
            throw new IllegalArgumentException("The configured memory pool ratio " + ratio + " is not between 0 and 1.");
        }
    }

    synchronized void addWriter(InternalParquetRecordWriter<?> writer, Long allocation) {
        Long oldValue = this.writerList.get(writer);
        if (oldValue != null) {
            throw new IllegalArgumentException("[BUG] The Parquet Memory Manager should not add an instance of InternalParquetRecordWriter more than once. The Manager already contains the writer: " + writer);
        }
        this.writerList.put(writer, allocation);
        this.updateAllocation();
    }

    synchronized void removeWriter(InternalParquetRecordWriter<?> writer) {
        this.writerList.remove(writer);
        if (!this.writerList.isEmpty()) {
            this.updateAllocation();
        }
    }

    private void updateAllocation() {
        long totalAllocations = 0L;
        for (Long l : this.writerList.values()) {
            totalAllocations += l.longValue();
        }
        if (totalAllocations <= this.totalMemoryPool) {
            this.scale = 1.0;
        } else {
            this.scale = (double)this.totalMemoryPool / (double)totalAllocations;
            LOG.warn(String.format("Total allocation exceeds %.2f%% (%,d bytes) of heap memory\nScaling row group sizes to %.2f%% for %d writers", Float.valueOf(100.0f * this.memoryPoolRatio), this.totalMemoryPool, 100.0 * this.scale, this.writerList.size()));
            for (Runnable runnable : this.callBacks.values()) {
                runnable.run();
            }
        }
        for (Map.Entry entry : this.writerList.entrySet()) {
            long newSize = (long)Math.floor((double)((Long)entry.getValue()).longValue() * this.scale);
            if (this.scale < 1.0 && this.minMemoryAllocation > 0L && newSize < this.minMemoryAllocation) {
                throw new ParquetMemoryManagerRuntimeException(String.format("New Memory allocation %d bytes is smaller than the minimum allocation size of %d bytes.", newSize, this.minMemoryAllocation));
            }
            ((InternalParquetRecordWriter)entry.getKey()).setRowGroupSizeThreshold(newSize);
            LOG.debug(String.format("Adjust block size from %,d to %,d for writer: %s", entry.getValue(), newSize, entry.getKey()));
        }
    }

    long getTotalMemoryPool() {
        return this.totalMemoryPool;
    }

    Map<InternalParquetRecordWriter<?>, Long> getWriterList() {
        return this.writerList;
    }

    float getMemoryPoolRatio() {
        return this.memoryPoolRatio;
    }

    public void registerScaleCallBack(String callBackName, Runnable callBack) {
        Objects.requireNonNull(callBackName, "callBackName cannot be null");
        Objects.requireNonNull(callBack, "callBack cannot be null");
        if (this.callBacks.containsKey(callBackName)) {
            throw new IllegalArgumentException("The callBackName " + callBackName + " is duplicated and has been registered already.");
        }
        this.callBacks.put(callBackName, callBack);
    }

    Map<String, Runnable> getScaleCallBacks() {
        return Collections.unmodifiableMap(this.callBacks);
    }

    double getScale() {
        return this.scale;
    }
}

