/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.ingest.streaming.internal;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import net.snowflake.client.jdbc.internal.google.common.collect.Sets;
import net.snowflake.client.jdbc.internal.snowflake.common.util.Power10;
import net.snowflake.ingest.internal.apache.commons.codec.DecoderException;
import net.snowflake.ingest.internal.apache.commons.codec.binary.Hex;
import net.snowflake.ingest.internal.fasterxml.jackson.core.JsonProcessingException;
import net.snowflake.ingest.internal.fasterxml.jackson.databind.JsonNode;
import net.snowflake.ingest.internal.fasterxml.jackson.databind.ObjectMapper;
import net.snowflake.ingest.internal.fasterxml.jackson.databind.module.SimpleModule;
import net.snowflake.ingest.internal.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import net.snowflake.ingest.streaming.internal.BinaryStringUtils;
import net.snowflake.ingest.streaming.internal.TimestampWrapper;
import net.snowflake.ingest.streaming.internal.serialization.ByteArraySerializer;
import net.snowflake.ingest.streaming.internal.serialization.ZonedDateTimeSerializer;
import net.snowflake.ingest.utils.ErrorCode;
import net.snowflake.ingest.utils.SFException;

class DataValidationUtil {
    private static final long SECONDS_LIMIT_FOR_EPOCH = 31536000000L;
    private static final long MILLISECONDS_LIMIT_FOR_EPOCH = 31536000000000L;
    private static final long MICROSECONDS_LIMIT_FOR_EPOCH = 31536000000000000L;
    public static final int BYTES_8_MB = 0x800000;
    public static final int BYTES_16_MB = 0x1000000;
    static final int MAX_SEMI_STRUCTURED_LENGTH = 0xFFFFC0;
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static final BigDecimal[] POWER_10;
    static Set<String> allowedBooleanStringsLowerCased;

    DataValidationUtil() {
    }

    private static BigDecimal[] makePower10Table() {
        BigDecimal[] power10 = new BigDecimal[39];
        for (int i = 0; i < 39; ++i) {
            power10[i] = new BigDecimal(Power10.sb16Table[i]);
        }
        return power10;
    }

    private static JsonNode validateAndParseSemiStructuredAsJsonTree(String columnName, Object input, String snowflakeType, long insertRowIndex) {
        if (input instanceof String) {
            String stringInput = (String)input;
            DataValidationUtil.verifyValidUtf8(stringInput, columnName, snowflakeType, insertRowIndex);
            try {
                return objectMapper.readTree(stringInput);
            }
            catch (JsonProcessingException e) {
                throw DataValidationUtil.valueFormatNotAllowedException(columnName, snowflakeType, "Not a valid JSON", insertRowIndex);
            }
        }
        if (DataValidationUtil.isAllowedSemiStructuredType(input)) {
            return objectMapper.valueToTree(input);
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), snowflakeType, new String[]{"String", "Primitive data types and their arrays", "java.time.*", "List<T>", "Map<String, T>", "T[]"}, insertRowIndex);
    }

    static String validateAndParseVariant(String columnName, Object input, long insertRowIndex) {
        JsonNode node = DataValidationUtil.validateAndParseSemiStructuredAsJsonTree(columnName, input, "VARIANT", insertRowIndex);
        if (node.isMissingNode()) {
            return null;
        }
        String output = node.toString();
        int stringLength = output.getBytes(StandardCharsets.UTF_8).length;
        if (stringLength > 0xFFFFC0) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, "VARIANT", String.format("Variant too long: length=%d maxLength=%d", stringLength, 0xFFFFC0), insertRowIndex);
        }
        return output;
    }

    static boolean isAllowedSemiStructuredType(Object o) {
        if (o == null) {
            return true;
        }
        if (o instanceof String) {
            return true;
        }
        if (o instanceof Long || o instanceof Integer || o instanceof Short || o instanceof Byte || o instanceof Float || o instanceof Double || o instanceof Boolean || o instanceof Character) {
            return true;
        }
        if (o instanceof BigInteger || o instanceof BigDecimal) {
            return true;
        }
        if (o instanceof LocalTime || o instanceof OffsetTime || o instanceof LocalDate || o instanceof LocalDateTime || o instanceof ZonedDateTime || o instanceof OffsetDateTime) {
            return true;
        }
        if (o instanceof Map) {
            boolean allKeysAreStrings = ((Map)o).keySet().stream().allMatch(x -> x instanceof String);
            if (!allKeysAreStrings) {
                return false;
            }
            boolean allValuesAreAllowed = ((Map)o).values().stream().allMatch(DataValidationUtil::isAllowedSemiStructuredType);
            return allValuesAreAllowed;
        }
        if (o instanceof byte[] || o instanceof short[] || o instanceof int[] || o instanceof long[] || o instanceof float[] || o instanceof double[] || o instanceof boolean[] || o instanceof char[]) {
            return true;
        }
        if (o.getClass().isArray()) {
            return Arrays.stream((Object[])o).allMatch(DataValidationUtil::isAllowedSemiStructuredType);
        }
        if (o instanceof List) {
            return ((List)o).stream().allMatch(DataValidationUtil::isAllowedSemiStructuredType);
        }
        return false;
    }

    static String validateAndParseArray(String columnName, Object input, long insertRowIndex) {
        String output;
        int stringLength;
        JsonNode jsonNode = DataValidationUtil.validateAndParseSemiStructuredAsJsonTree(columnName, input, "ARRAY", insertRowIndex);
        if (!jsonNode.isArray()) {
            jsonNode = objectMapper.createArrayNode().add(jsonNode);
        }
        if ((stringLength = (output = jsonNode.toString()).getBytes(StandardCharsets.UTF_8).length) > 0xFFFFC0) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, "ARRAY", String.format("Array too large. length=%d maxLength=%d", stringLength, 0xFFFFC0), insertRowIndex);
        }
        return output;
    }

    static String validateAndParseObject(String columnName, Object input, long insertRowIndex) {
        JsonNode jsonNode = DataValidationUtil.validateAndParseSemiStructuredAsJsonTree(columnName, input, "OBJECT", insertRowIndex);
        if (!jsonNode.isObject()) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, "OBJECT", "Not an object", insertRowIndex);
        }
        String output = jsonNode.toString();
        int stringLength = output.getBytes(StandardCharsets.UTF_8).length;
        if (stringLength > 0xFFFFC0) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, "OBJECT", String.format("Object too large. length=%d maxLength=%d", stringLength, 0xFFFFC0), insertRowIndex);
        }
        return output;
    }

    private static OffsetDateTime inputToOffsetDateTime(String columnName, String typeName, Object input, ZoneId defaultTimezone, long insertRowIndex) {
        if (input instanceof OffsetDateTime) {
            return (OffsetDateTime)input;
        }
        if (input instanceof ZonedDateTime) {
            return ((ZonedDateTime)input).toOffsetDateTime();
        }
        if (input instanceof LocalDateTime) {
            return ((LocalDateTime)input).atZone(defaultTimezone).toOffsetDateTime();
        }
        if (input instanceof LocalDate) {
            return ((LocalDate)input).atStartOfDay().atZone(defaultTimezone).toOffsetDateTime();
        }
        if (input instanceof Instant) {
            return ((Instant)input).atZone(ZoneOffset.UTC).toOffsetDateTime();
        }
        if (input instanceof String) {
            String stringInput = ((String)input).trim();
            ZonedDateTime zoned = DataValidationUtil.catchParsingError(() -> ZonedDateTime.parse(stringInput));
            if (zoned != null) {
                return zoned.toOffsetDateTime();
            }
            OffsetDateTime offset = DataValidationUtil.catchParsingError(() -> OffsetDateTime.parse(stringInput));
            if (offset != null) {
                return offset;
            }
            LocalDateTime localDateTime = DataValidationUtil.catchParsingError(() -> LocalDateTime.parse(stringInput));
            if (localDateTime != null) {
                return localDateTime.atZone(defaultTimezone).toOffsetDateTime();
            }
            LocalDate localDate = DataValidationUtil.catchParsingError(() -> LocalDate.parse(stringInput));
            if (localDate != null) {
                return localDate.atStartOfDay().atZone(defaultTimezone).toOffsetDateTime();
            }
            Instant instant = DataValidationUtil.catchParsingError(() -> DataValidationUtil.parseInstantGuessScale(stringInput));
            if (instant != null) {
                return instant.atOffset(ZoneOffset.UTC);
            }
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, typeName, "Not a valid value, see https://docs.snowflake.com/en/user-guide/data-load-snowpipe-streaming-overview for the list of supported formats", insertRowIndex);
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), typeName, new String[]{"String", "LocalDate", "LocalDateTime", "ZonedDateTime", "OffsetDateTime"}, insertRowIndex);
    }

    private static <T> T catchParsingError(Supplier<T> op) {
        try {
            return op.get();
        }
        catch (NumberFormatException | DateTimeParseException e) {
            return null;
        }
    }

    static TimestampWrapper validateAndParseTimestamp(String columnName, Object input, int scale, ZoneId defaultTimezone, boolean trimTimezone, long insertRowIndex) {
        OffsetDateTime offsetDateTime = DataValidationUtil.inputToOffsetDateTime(columnName, "TIMESTAMP", input, defaultTimezone, insertRowIndex);
        if (trimTimezone) {
            offsetDateTime = offsetDateTime.withOffsetSameLocal(ZoneOffset.UTC);
        }
        if (offsetDateTime.getYear() < 1 || offsetDateTime.getYear() > 9999) {
            throw new SFException(ErrorCode.INVALID_VALUE_ROW, String.format("Timestamp out of representable inclusive range of years between 1 and 9999, rowIndex:%d", insertRowIndex));
        }
        return new TimestampWrapper(offsetDateTime, scale);
    }

    static String validateAndParseString(String columnName, Object input, Optional<Integer> maxLengthOptional, long insertRowIndex) {
        String output;
        if (input instanceof String) {
            output = (String)input;
            DataValidationUtil.verifyValidUtf8(output, columnName, "STRING", insertRowIndex);
        } else if (input instanceof Number) {
            output = new BigDecimal(input.toString()).stripTrailingZeros().toPlainString();
        } else if (input instanceof Boolean || input instanceof Character) {
            output = input.toString();
        } else {
            throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "STRING", new String[]{"String", "Number", "boolean", "char"}, insertRowIndex);
        }
        byte[] utf8Bytes = output.getBytes(StandardCharsets.UTF_8);
        if (utf8Bytes.length > 0x1000000) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, "STRING", String.format("String too long: length=%d bytes maxLength=%d bytes", utf8Bytes.length, 0x1000000), insertRowIndex);
        }
        maxLengthOptional.ifPresent(maxAllowedCharacters -> {
            int actualCharacters = BinaryStringUtils.unicodeCharactersCount(output);
            if (actualCharacters > maxAllowedCharacters) {
                throw DataValidationUtil.valueFormatNotAllowedException(columnName, "STRING", String.format("String too long: length=%d characters maxLength=%d characters", actualCharacters, maxAllowedCharacters), insertRowIndex);
            }
        });
        return output;
    }

    static BigDecimal validateAndParseBigDecimal(String columnName, Object input, long insertRowIndex) {
        if (input instanceof BigDecimal) {
            return (BigDecimal)input;
        }
        if (input instanceof BigInteger) {
            return new BigDecimal((BigInteger)input);
        }
        if (input instanceof Byte || input instanceof Short || input instanceof Integer || input instanceof Long) {
            return BigDecimal.valueOf(((Number)input).longValue());
        }
        if (input instanceof Float || input instanceof Double) {
            return BigDecimal.valueOf(((Number)input).doubleValue());
        }
        if (input instanceof String) {
            try {
                String stringInput = ((String)input).trim();
                return new BigDecimal(stringInput);
            }
            catch (NumberFormatException e) {
                throw DataValidationUtil.valueFormatNotAllowedException(columnName, "NUMBER", "Not a valid number", insertRowIndex);
            }
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "NUMBER", new String[]{"int", "long", "byte", "short", "float", "double", "BigDecimal", "BigInteger", "String"}, insertRowIndex);
    }

    static int validateAndParseDate(String columnName, Object input, long insertRowIndex) {
        OffsetDateTime offsetDateTime = DataValidationUtil.inputToOffsetDateTime(columnName, "DATE", input, ZoneOffset.UTC, insertRowIndex);
        if (offsetDateTime.getYear() < -9999 || offsetDateTime.getYear() > 9999) {
            throw new SFException(ErrorCode.INVALID_VALUE_ROW, String.format("Date out of representable inclusive range of years between -9999 and 9999, rowIndex:%d", insertRowIndex));
        }
        return Math.toIntExact(offsetDateTime.toLocalDate().toEpochDay());
    }

    static byte[] validateAndParseBinary(String columnName, Object input, Optional<Integer> maxLengthOptional, long insertRowIndex) {
        byte[] output;
        if (input instanceof byte[]) {
            byte[] originalInputArray = (byte[])input;
            output = new byte[originalInputArray.length];
            System.arraycopy(originalInputArray, 0, output, 0, originalInputArray.length);
        } else if (input instanceof String) {
            try {
                String stringInput = ((String)input).trim();
                output = Hex.decodeHex(stringInput);
            }
            catch (DecoderException e) {
                throw DataValidationUtil.valueFormatNotAllowedException(columnName, "BINARY", "Not a valid hex string", insertRowIndex);
            }
        } else {
            throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "BINARY", new String[]{"byte[]", "String"}, insertRowIndex);
        }
        int maxLength = maxLengthOptional.orElse(0x800000);
        if (output.length > maxLength) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, "BINARY", String.format("Binary too long: length=%d maxLength=%d", output.length, maxLength), insertRowIndex);
        }
        return output;
    }

    static BigInteger validateAndParseTime(String columnName, Object input, int scale, long insertRowIndex) {
        if (input instanceof LocalTime) {
            LocalTime localTime = (LocalTime)input;
            return BigInteger.valueOf(localTime.toNanoOfDay()).divide(Power10.sb16Table[9 - scale]);
        }
        if (input instanceof OffsetTime) {
            return DataValidationUtil.validateAndParseTime(columnName, ((OffsetTime)input).toLocalTime(), scale, insertRowIndex);
        }
        if (input instanceof String) {
            String stringInput = ((String)input).trim();
            LocalTime localTime = DataValidationUtil.catchParsingError(() -> LocalTime.parse(stringInput));
            if (localTime != null) {
                return DataValidationUtil.validateAndParseTime(columnName, localTime, scale, insertRowIndex);
            }
            OffsetTime offsetTime = DataValidationUtil.catchParsingError(() -> OffsetTime.parse(stringInput));
            if (offsetTime != null) {
                return DataValidationUtil.validateAndParseTime(columnName, offsetTime.toLocalTime(), scale, insertRowIndex);
            }
            Instant parsedInstant = DataValidationUtil.catchParsingError(() -> DataValidationUtil.parseInstantGuessScale(stringInput));
            if (parsedInstant != null) {
                return DataValidationUtil.validateAndParseTime(columnName, LocalDateTime.ofInstant(parsedInstant, ZoneOffset.UTC).toLocalTime(), scale, insertRowIndex);
            }
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, "TIME", "Not a valid time, see https://docs.snowflake.com/en/user-guide/data-load-snowpipe-streaming-overview for the list of supported formats", insertRowIndex);
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "TIME", new String[]{"String", "LocalTime", "OffsetTime"}, insertRowIndex);
    }

    private static Instant parseInstantGuessScale(String input) {
        BigInteger epochNanos;
        try {
            long val = Long.parseLong(input);
            epochNanos = val > -31536000000L && val < 31536000000L ? BigInteger.valueOf(val).multiply(Power10.sb16Table[9]) : (val > -31536000000000L && val < 31536000000000L ? BigInteger.valueOf(val).multiply(Power10.sb16Table[6]) : (val > -31536000000000000L && val < 31536000000000000L ? BigInteger.valueOf(val).multiply(Power10.sb16Table[3]) : BigInteger.valueOf(val)));
        }
        catch (NumberFormatException e) {
            epochNanos = new BigInteger(input);
        }
        return Instant.ofEpochSecond(epochNanos.divide(Power10.sb16Table[9]).longValue(), epochNanos.remainder(Power10.sb16Table[9]).longValue());
    }

    static double validateAndParseReal(String columnName, Object input, long insertRowIndex) {
        if (input instanceof Float) {
            return Double.parseDouble(input.toString());
        }
        if (input instanceof Number) {
            return ((Number)input).doubleValue();
        }
        if (input instanceof String) {
            String stringInput = ((String)input).trim();
            try {
                return Double.parseDouble(stringInput);
            }
            catch (NumberFormatException err) {
                switch (stringInput = stringInput.toLowerCase()) {
                    case "nan": {
                        return Double.NaN;
                    }
                    case "inf": {
                        return Double.POSITIVE_INFINITY;
                    }
                    case "-inf": {
                        return Double.NEGATIVE_INFINITY;
                    }
                }
                throw DataValidationUtil.valueFormatNotAllowedException(columnName, "REAL", "Not a valid decimal number", insertRowIndex);
            }
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "REAL", new String[]{"Number", "String"}, insertRowIndex);
    }

    static int validateAndParseBoolean(String columnName, Object input, long insertRowIndex) {
        if (input instanceof Boolean) {
            return (Boolean)input != false ? 1 : 0;
        }
        if (input instanceof Number) {
            return new BigDecimal(input.toString()).compareTo(BigDecimal.ZERO) == 0 ? 0 : 1;
        }
        if (input instanceof String) {
            return DataValidationUtil.convertStringToBoolean(columnName, (String)input, insertRowIndex) ? 1 : 0;
        }
        throw DataValidationUtil.typeNotAllowedException(columnName, input.getClass(), "BOOLEAN", new String[]{"boolean", "Number", "String"}, insertRowIndex);
    }

    static void checkValueInRange(BigDecimal bigDecimalValue, int scale, int precision, long insertRowIndex) {
        BigDecimal comparand;
        BigDecimal bigDecimal = comparand = precision >= scale && precision - scale < POWER_10.length ? POWER_10[precision - scale] : BigDecimal.TEN.pow(precision - scale);
        if (bigDecimalValue.abs().compareTo(comparand) >= 0) {
            throw new SFException(ErrorCode.INVALID_FORMAT_ROW, String.format("Number out of representable exclusive range of (-1e%s..1e%s), rowIndex:%d", precision - scale, precision - scale, insertRowIndex));
        }
    }

    private static boolean convertStringToBoolean(String columnName, String value, long insertRowIndex) {
        String normalizedInput = value.toLowerCase().trim();
        if (!allowedBooleanStringsLowerCased.contains(normalizedInput)) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, "BOOLEAN", "Not a valid boolean, see https://docs.snowflake.com/en/sql-reference/data-types-logical.html#conversion-to-boolean for the list of supported formats", insertRowIndex);
        }
        return "1".equals(normalizedInput) || "yes".equals(normalizedInput) || "y".equals(normalizedInput) || "t".equals(normalizedInput) || "true".equals(normalizedInput) || "on".equals(normalizedInput);
    }

    private static SFException typeNotAllowedException(String columnName, Class<?> javaType, String snowflakeType, String[] allowedJavaTypes, long insertRowIndex) {
        return new SFException(ErrorCode.INVALID_FORMAT_ROW, String.format("Object of type %s cannot be ingested into Snowflake column %s of type %s, rowIndex:%d", javaType.getName(), columnName, snowflakeType, insertRowIndex), String.format(String.format("Allowed Java types: %s", String.join((CharSequence)", ", allowedJavaTypes)), new Object[0]));
    }

    private static SFException valueFormatNotAllowedException(String columnName, String snowflakeType, String reason, long rowIndex) {
        return new SFException(ErrorCode.INVALID_VALUE_ROW, String.format("Value cannot be ingested into Snowflake column %s of type %s, rowIndex:%d, reason: %s", columnName, snowflakeType, rowIndex, reason));
    }

    private static void verifyValidUtf8(String input, String columnName, String dataType, long insertRowIndex) {
        String roundTripStr = new String(input.getBytes(StandardCharsets.UTF_8), StandardCharsets.UTF_8);
        if (!input.equals(roundTripStr)) {
            throw DataValidationUtil.valueFormatNotAllowedException(columnName, dataType, "Invalid Unicode string", insertRowIndex);
        }
    }

    static {
        SimpleModule module = new SimpleModule();
        module.addSerializer(byte[].class, new ByteArraySerializer());
        module.addSerializer(ZonedDateTime.class, new ZonedDateTimeSerializer());
        module.addSerializer(LocalTime.class, new ToStringSerializer());
        module.addSerializer(OffsetTime.class, new ToStringSerializer());
        module.addSerializer(LocalDate.class, new ToStringSerializer());
        module.addSerializer(LocalDateTime.class, new ToStringSerializer());
        module.addSerializer(OffsetDateTime.class, new ToStringSerializer());
        objectMapper.registerModule(module);
        POWER_10 = DataValidationUtil.makePower10Table();
        allowedBooleanStringsLowerCased = Sets.newHashSet((Object[])new String[]{"1", "0", "yes", "no", "y", "n", "t", "f", "true", "false", "on", "off"});
    }
}

