/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.ingest.streaming.internal;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.nio.charset.StandardCharsets;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import net.snowflake.ingest.streaming.internal.DataValidationUtil;
import net.snowflake.ingest.streaming.internal.ParquetBufferValue;
import net.snowflake.ingest.streaming.internal.RowBufferStats;
import net.snowflake.ingest.utils.ErrorCode;
import net.snowflake.ingest.utils.SFException;
import net.snowflake.ingest.utils.Utils;
import org.apache.parquet.schema.GroupType;
import org.apache.parquet.schema.LogicalTypeAnnotation;
import org.apache.parquet.schema.PrimitiveType;
import org.apache.parquet.schema.Type;

class IcebergParquetValueParser {
    static final String THREE_LEVEL_MAP_GROUP_NAME = "key_value";
    static final String THREE_LEVEL_LIST_GROUP_NAME = "list";

    IcebergParquetValueParser() {
    }

    static ParquetBufferValue parseColumnValueToParquet(Object value, Type type, Map<String, RowBufferStats> statsMap, ZoneId defaultTimezone, long insertRowsCurrIndex) {
        Utils.assertNotNull("Parquet column stats map", statsMap);
        return IcebergParquetValueParser.parseColumnValueToParquet(value, type, statsMap, defaultTimezone, insertRowsCurrIndex, null, false);
    }

    private static ParquetBufferValue parseColumnValueToParquet(Object value, Type type, Map<String, RowBufferStats> statsMap, ZoneId defaultTimezone, long insertRowsCurrIndex, String path, boolean isDescendantsOfRepeatingGroup) {
        path = Utils.isNullOrEmpty(path) ? type.getName() : Utils.concatDotPath(path, type.getName());
        float estimatedParquetSize = 0.0f;
        if (type.isPrimitive() && !statsMap.containsKey(path)) {
            throw new SFException(ErrorCode.INTERNAL_ERROR, String.format("Stats not found for column: %s", path));
        }
        if (value != null) {
            if (type.isPrimitive()) {
                RowBufferStats stats = statsMap.get(path);
                estimatedParquetSize += 0.25f;
                estimatedParquetSize += isDescendantsOfRepeatingGroup ? 0.25f : 0.0f;
                PrimitiveType primitiveType = type.asPrimitiveType();
                switch (primitiveType.getPrimitiveTypeName()) {
                    case BOOLEAN: {
                        int intValue = DataValidationUtil.validateAndParseBoolean(path, value, insertRowsCurrIndex);
                        value = intValue > 0;
                        stats.addIntValue(BigInteger.valueOf(intValue));
                        estimatedParquetSize += 0.125f;
                        break;
                    }
                    case INT32: {
                        int intVal = IcebergParquetValueParser.getInt32Value(value, primitiveType, path, insertRowsCurrIndex);
                        value = intVal;
                        stats.addIntValue(BigInteger.valueOf(intVal));
                        estimatedParquetSize += 4.0f;
                        break;
                    }
                    case INT64: {
                        long longVal = IcebergParquetValueParser.getInt64Value(value, primitiveType, defaultTimezone, path, insertRowsCurrIndex);
                        value = longVal;
                        stats.addIntValue(BigInteger.valueOf(longVal));
                        estimatedParquetSize += 8.0f;
                        break;
                    }
                    case FLOAT: {
                        float floatVal = (float)DataValidationUtil.validateAndParseReal(path, value, insertRowsCurrIndex);
                        value = Float.valueOf(floatVal);
                        stats.addRealValue(Double.valueOf(floatVal));
                        estimatedParquetSize += 4.0f;
                        break;
                    }
                    case DOUBLE: {
                        double doubleVal = DataValidationUtil.validateAndParseReal(path, value, insertRowsCurrIndex);
                        value = doubleVal;
                        stats.addRealValue(doubleVal);
                        estimatedParquetSize += 8.0f;
                        break;
                    }
                    case BINARY: {
                        byte[] byteVal = IcebergParquetValueParser.getBinaryValue(value, primitiveType, stats, path, insertRowsCurrIndex);
                        value = byteVal;
                        estimatedParquetSize += (float)(4 + byteVal.length);
                        break;
                    }
                    case FIXED_LEN_BYTE_ARRAY: {
                        byte[] fixedLenByteArrayVal = IcebergParquetValueParser.getFixedLenByteArrayValue(value, primitiveType, stats, path, insertRowsCurrIndex);
                        value = fixedLenByteArrayVal;
                        estimatedParquetSize += (float)(4 + fixedLenByteArrayVal.length);
                        break;
                    }
                    default: {
                        throw new SFException(ErrorCode.UNKNOWN_DATA_TYPE, type.getLogicalTypeAnnotation(), primitiveType.getPrimitiveTypeName());
                    }
                }
            } else {
                return IcebergParquetValueParser.getGroupValue(value, type.asGroupType(), statsMap, defaultTimezone, insertRowsCurrIndex, path, isDescendantsOfRepeatingGroup);
            }
        }
        if (value == null) {
            if (type.isRepetition(Type.Repetition.REQUIRED)) {
                throw new SFException(ErrorCode.INVALID_FORMAT_ROW, path, "Passed null to non nullable field");
            }
            if (type.isPrimitive()) {
                statsMap.get(path).incCurrentNullCount();
            }
        }
        return new ParquetBufferValue(value, estimatedParquetSize);
    }

    private static int getInt32Value(Object value, PrimitiveType type, String path, long insertRowsCurrIndex) {
        LogicalTypeAnnotation logicalTypeAnnotation = type.getLogicalTypeAnnotation();
        if (logicalTypeAnnotation == null) {
            return DataValidationUtil.validateAndParseIcebergInt(path, value, insertRowsCurrIndex);
        }
        if (logicalTypeAnnotation instanceof LogicalTypeAnnotation.DecimalLogicalTypeAnnotation) {
            return IcebergParquetValueParser.getDecimalValue(value, type, path, insertRowsCurrIndex).unscaledValue().intValue();
        }
        if (logicalTypeAnnotation instanceof LogicalTypeAnnotation.DateLogicalTypeAnnotation) {
            return DataValidationUtil.validateAndParseDate(path, value, insertRowsCurrIndex);
        }
        throw new SFException(ErrorCode.UNKNOWN_DATA_TYPE, logicalTypeAnnotation, type.getPrimitiveTypeName());
    }

    private static long getInt64Value(Object value, PrimitiveType type, ZoneId defaultTimezone, String path, long insertRowsCurrIndex) {
        LogicalTypeAnnotation logicalTypeAnnotation = type.getLogicalTypeAnnotation();
        if (logicalTypeAnnotation == null) {
            return DataValidationUtil.validateAndParseIcebergLong(path, value, insertRowsCurrIndex);
        }
        if (logicalTypeAnnotation instanceof LogicalTypeAnnotation.DecimalLogicalTypeAnnotation) {
            return IcebergParquetValueParser.getDecimalValue(value, type, path, insertRowsCurrIndex).unscaledValue().longValue();
        }
        if (logicalTypeAnnotation instanceof LogicalTypeAnnotation.TimeLogicalTypeAnnotation) {
            return DataValidationUtil.validateAndParseTime(path, value, IcebergParquetValueParser.timeUnitToScale(((LogicalTypeAnnotation.TimeLogicalTypeAnnotation)logicalTypeAnnotation).getUnit()), insertRowsCurrIndex).longValue();
        }
        if (logicalTypeAnnotation instanceof LogicalTypeAnnotation.TimestampLogicalTypeAnnotation) {
            boolean includeTimeZone = ((LogicalTypeAnnotation.TimestampLogicalTypeAnnotation)logicalTypeAnnotation).isAdjustedToUTC();
            return DataValidationUtil.validateAndParseTimestamp(type.getName(), value, IcebergParquetValueParser.timeUnitToScale(((LogicalTypeAnnotation.TimestampLogicalTypeAnnotation)logicalTypeAnnotation).getUnit()), defaultTimezone, !includeTimeZone, insertRowsCurrIndex).toBinary(false).longValue();
        }
        throw new SFException(ErrorCode.UNKNOWN_DATA_TYPE, logicalTypeAnnotation, type.asPrimitiveType().getPrimitiveTypeName());
    }

    private static byte[] getBinaryValue(Object value, PrimitiveType type, RowBufferStats stats, String path, long insertRowsCurrIndex) {
        LogicalTypeAnnotation logicalTypeAnnotation = type.getLogicalTypeAnnotation();
        if (logicalTypeAnnotation == null) {
            byte[] bytes = DataValidationUtil.validateAndParseBinary(path, value, Optional.of(0x800000), insertRowsCurrIndex);
            stats.addBinaryValue(bytes);
            return bytes;
        }
        if (logicalTypeAnnotation instanceof LogicalTypeAnnotation.StringLogicalTypeAnnotation) {
            String string = DataValidationUtil.validateAndParseString(path, value, Optional.of(0x1000000), insertRowsCurrIndex);
            stats.addStrValue(string);
            return string.getBytes(StandardCharsets.UTF_8);
        }
        throw new SFException(ErrorCode.UNKNOWN_DATA_TYPE, logicalTypeAnnotation, type.getPrimitiveTypeName());
    }

    private static byte[] getFixedLenByteArrayValue(Object value, PrimitiveType type, RowBufferStats stats, String path, long insertRowsCurrIndex) {
        LogicalTypeAnnotation logicalTypeAnnotation = type.getLogicalTypeAnnotation();
        int length = type.getTypeLength();
        byte[] bytes = null;
        if (logicalTypeAnnotation == null) {
            bytes = DataValidationUtil.validateAndParseBinary(path, value, Optional.of(length), insertRowsCurrIndex);
            stats.addBinaryValue(bytes);
        }
        if (logicalTypeAnnotation instanceof LogicalTypeAnnotation.DecimalLogicalTypeAnnotation) {
            BigInteger bigIntegerVal = IcebergParquetValueParser.getDecimalValue(value, type, path, insertRowsCurrIndex).unscaledValue();
            stats.addIntValue(bigIntegerVal);
            bytes = bigIntegerVal.toByteArray();
            if (bytes.length < length) {
                byte[] newBytes = new byte[length];
                Arrays.fill(newBytes, (byte)(bytes[0] < 0 ? -1 : 0));
                System.arraycopy(bytes, 0, newBytes, length - bytes.length, bytes.length);
                bytes = newBytes;
            }
        }
        if (bytes != null) {
            DataValidationUtil.checkFixedLengthByteArray(bytes, length, insertRowsCurrIndex);
            return bytes;
        }
        throw new SFException(ErrorCode.UNKNOWN_DATA_TYPE, logicalTypeAnnotation, type.getPrimitiveTypeName());
    }

    private static BigDecimal getDecimalValue(Object value, PrimitiveType type, String path, long insertRowsCurrIndex) {
        int scale = ((LogicalTypeAnnotation.DecimalLogicalTypeAnnotation)type.getLogicalTypeAnnotation()).getScale();
        int precision = ((LogicalTypeAnnotation.DecimalLogicalTypeAnnotation)type.getLogicalTypeAnnotation()).getPrecision();
        BigDecimal bigDecimalValue = DataValidationUtil.validateAndParseBigDecimal(path, value, insertRowsCurrIndex);
        bigDecimalValue = bigDecimalValue.setScale(scale, RoundingMode.HALF_UP);
        DataValidationUtil.checkValueInRange(bigDecimalValue, scale, precision, insertRowsCurrIndex);
        return bigDecimalValue;
    }

    private static int timeUnitToScale(LogicalTypeAnnotation.TimeUnit timeUnit) {
        switch (timeUnit) {
            case MILLIS: {
                return 3;
            }
            case MICROS: {
                return 6;
            }
            case NANOS: {
                return 9;
            }
        }
        throw new SFException(ErrorCode.INTERNAL_ERROR, String.format("Unknown time unit: %s", timeUnit));
    }

    private static ParquetBufferValue getGroupValue(Object value, GroupType type, Map<String, RowBufferStats> statsMap, ZoneId defaultTimezone, long insertRowsCurrIndex, String path, boolean isDescendantsOfRepeatingGroup) {
        LogicalTypeAnnotation logicalTypeAnnotation = type.getLogicalTypeAnnotation();
        if (logicalTypeAnnotation == null) {
            return IcebergParquetValueParser.getStructValue(value, type, statsMap, defaultTimezone, insertRowsCurrIndex, path, isDescendantsOfRepeatingGroup);
        }
        if (logicalTypeAnnotation instanceof LogicalTypeAnnotation.ListLogicalTypeAnnotation) {
            return IcebergParquetValueParser.get3LevelListValue(value, type, statsMap, defaultTimezone, insertRowsCurrIndex, path);
        }
        if (logicalTypeAnnotation instanceof LogicalTypeAnnotation.MapLogicalTypeAnnotation) {
            return IcebergParquetValueParser.get3LevelMapValue(value, type, statsMap, defaultTimezone, insertRowsCurrIndex, path);
        }
        throw new SFException(ErrorCode.UNKNOWN_DATA_TYPE, logicalTypeAnnotation, type.getClass().getSimpleName());
    }

    private static ParquetBufferValue getStructValue(Object value, GroupType type, Map<String, RowBufferStats> statsMap, ZoneId defaultTimezone, long insertRowsCurrIndex, String path, boolean isDescendantsOfRepeatingGroup) {
        Map<String, ?> structVal = DataValidationUtil.validateAndParseIcebergStruct(path, value, insertRowsCurrIndex);
        HashSet<String> extraFields = new HashSet<String>(structVal.keySet());
        ArrayList<Object> listVal = new ArrayList<Object>(type.getFieldCount());
        float estimatedParquetSize = 0.0f;
        for (int i = 0; i < type.getFieldCount(); ++i) {
            ParquetBufferValue parsedValue = IcebergParquetValueParser.parseColumnValueToParquet(structVal.getOrDefault(type.getFieldName(i), null), type.getType(i), statsMap, defaultTimezone, insertRowsCurrIndex, path, isDescendantsOfRepeatingGroup);
            extraFields.remove(type.getFieldName(i));
            listVal.add(parsedValue.getValue());
            estimatedParquetSize += parsedValue.getSize();
        }
        if (!extraFields.isEmpty()) {
            String extraFieldsStr = extraFields.stream().map(f -> Utils.concatDotPath(path, f)).collect(Collectors.joining(", "));
            throw new SFException(ErrorCode.INVALID_FORMAT_ROW, "Extra fields: " + extraFieldsStr, String.format("Fields not present in the struct shouldn't be specified, rowIndex:%d", insertRowsCurrIndex));
        }
        return new ParquetBufferValue(listVal, estimatedParquetSize);
    }

    private static ParquetBufferValue get3LevelListValue(Object value, GroupType type, Map<String, RowBufferStats> statsMap, ZoneId defaultTimezone, long insertRowsCurrIndex, String path) {
        Iterable<?> iterableVal = DataValidationUtil.validateAndParseIcebergList(path, value, insertRowsCurrIndex);
        ArrayList<List<Object>> listVal = new ArrayList<List<Object>>();
        float estimatedParquetSize = 0.0f;
        String listGroupPath = Utils.concatDotPath(path, THREE_LEVEL_LIST_GROUP_NAME);
        for (Object val : iterableVal) {
            ParquetBufferValue parsedValue = IcebergParquetValueParser.parseColumnValueToParquet(val, type.getType(0).asGroupType().getType(0), statsMap, defaultTimezone, insertRowsCurrIndex, listGroupPath, true);
            listVal.add(Collections.singletonList(parsedValue.getValue()));
            estimatedParquetSize += parsedValue.getSize();
        }
        return new ParquetBufferValue(listVal, estimatedParquetSize);
    }

    private static ParquetBufferValue get3LevelMapValue(Object value, GroupType type, Map<String, RowBufferStats> statsMap, ZoneId defaultTimezone, long insertRowsCurrIndex, String path) {
        Map<?, ?> mapVal = DataValidationUtil.validateAndParseIcebergMap(path, value, insertRowsCurrIndex);
        ArrayList<List<Object>> listVal = new ArrayList<List<Object>>();
        float estimatedParquetSize = 0.0f;
        String mapGroupPath = Utils.concatDotPath(path, THREE_LEVEL_MAP_GROUP_NAME);
        for (Map.Entry<?, ?> entry : mapVal.entrySet()) {
            ParquetBufferValue parsedKey = IcebergParquetValueParser.parseColumnValueToParquet(entry.getKey(), type.getType(0).asGroupType().getType(0), statsMap, defaultTimezone, insertRowsCurrIndex, mapGroupPath, true);
            ParquetBufferValue parsedValue = IcebergParquetValueParser.parseColumnValueToParquet(entry.getValue(), type.getType(0).asGroupType().getType(1), statsMap, defaultTimezone, insertRowsCurrIndex, mapGroupPath, true);
            listVal.add(Arrays.asList(parsedKey.getValue(), parsedValue.getValue()));
            estimatedParquetSize += parsedKey.getSize() + parsedValue.getSize();
        }
        return new ParquetBufferValue(listVal, estimatedParquetSize);
    }
}

