/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core;

import com.google.common.base.Strings;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileOwnerAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Arrays;
import java.util.Collection;
import java.util.Set;
import net.snowflake.client.core.SnowflakeJdbcInternalApi;
import net.snowflake.client.jdbc.SnowflakeUtil;
import net.snowflake.client.log.SFLogger;
import net.snowflake.client.log.SFLoggerFactory;

@SnowflakeJdbcInternalApi
public class FileUtil {
    private static final SFLogger logger = SFLoggerFactory.getLogger(FileUtil.class);
    private static final Collection<PosixFilePermission> WRITE_BY_OTHERS = Arrays.asList(PosixFilePermission.GROUP_WRITE, PosixFilePermission.OTHERS_WRITE);
    private static final Collection<PosixFilePermission> READ_BY_OTHERS = Arrays.asList(PosixFilePermission.GROUP_READ, PosixFilePermission.OTHERS_READ);
    private static final Collection<PosixFilePermission> EXECUTABLE = Arrays.asList(PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OTHERS_EXECUTE);

    public static void logFileUsage(Path filePath, String context, boolean logReadAccess) {
        FileUtil.logWarnWhenAccessibleByOthers(filePath, context, logReadAccess);
    }

    public static void logFileUsage(File file, String context, boolean logReadAccess) {
        FileUtil.logFileUsage(file.toPath(), context, logReadAccess);
    }

    public static void logFileUsage(String stringPath, String context, boolean logReadAccess) {
        Path path = Paths.get(stringPath, new String[0]);
        FileUtil.logFileUsage(path, context, logReadAccess);
    }

    public static void throwWhenPermiossionDifferentThanReadWriteForOwner(File file, String context) {
        FileUtil.throwWhenPermiossionDifferentThanReadWriteForOwner(file.toPath(), context);
    }

    public static void throwWhenPermiossionDifferentThanReadWriteForOwner(Path filePath, String context) {
        if (SnowflakeUtil.isWindows()) {
            return;
        }
        try {
            Set<PosixFilePermission> filePermissions = Files.getPosixFilePermissions(filePath, new LinkOption[0]);
            boolean isWritableByOthers = FileUtil.isPermPresent(filePermissions, WRITE_BY_OTHERS);
            boolean isReadableByOthers = FileUtil.isPermPresent(filePermissions, READ_BY_OTHERS);
            boolean isExecutable = FileUtil.isPermPresent(filePermissions, EXECUTABLE);
            if (isWritableByOthers || isReadableByOthers || isExecutable) {
                logger.debug("{}File {} access rights: {}", FileUtil.getContextStr(context), filePath, filePermissions);
                throw new SecurityException(String.format("Access to file %s is wider than allowed only to the owner", filePath));
            }
        }
        catch (IOException e) {
            throw new SecurityException(String.format("%s Unable to access the file to check the permissions. Error: %s", filePath, e));
        }
    }

    private static void logWarnWhenAccessibleByOthers(Path filePath, String context, boolean logReadAccess) {
        if (SnowflakeUtil.isWindows()) {
            return;
        }
        try {
            Set<PosixFilePermission> filePermissions = Files.getPosixFilePermissions(filePath, new LinkOption[0]);
            logger.debug("{}File {} access rights: {}", FileUtil.getContextStr(context), filePath, filePermissions);
            boolean isWritableByOthers = FileUtil.isPermPresent(filePermissions, WRITE_BY_OTHERS);
            boolean isReadableByOthers = FileUtil.isPermPresent(filePermissions, READ_BY_OTHERS);
            boolean isExecutable = FileUtil.isPermPresent(filePermissions, EXECUTABLE);
            if (isWritableByOthers || isReadableByOthers || isExecutable) {
                logger.warn("{}File {} is accessible by others to:{}{}", FileUtil.getContextStr(context), filePath, isReadableByOthers && logReadAccess ? " read" : "", isWritableByOthers ? " write" : "", isExecutable ? " executable" : "");
            }
        }
        catch (IOException e) {
            logger.warn("{}Unable to access the file to check the permissions: {}. Error: {}", FileUtil.getContextStr(context), filePath, e);
        }
    }

    public static void throwWhenOwnerDifferentThanCurrentUser(File file, String context) {
        if (SnowflakeUtil.isWindows()) {
            return;
        }
        Path filePath = Paths.get(file.getPath(), new String[0]);
        try {
            String fileOwnerName = FileUtil.getFileOwnerName(filePath);
            String currentUser = System.getProperty("user.name");
            if (!currentUser.equalsIgnoreCase(fileOwnerName)) {
                logger.debug("The file owner: {} is different than current user: {}", fileOwnerName, currentUser);
                throw new SecurityException("The file owner is different than current user");
            }
        }
        catch (IOException e) {
            logger.warn("{}Unable to access the file to check the owner: {}. Error: {}", FileUtil.getContextStr(context), filePath, e);
        }
    }

    private static boolean isPermPresent(Collection<PosixFilePermission> filePerms, Collection<PosixFilePermission> permsToCheck) throws IOException {
        return filePerms.stream().anyMatch(permsToCheck::contains);
    }

    static String getFileOwnerName(Path filePath) throws IOException {
        FileOwnerAttributeView ownerAttributeView = Files.getFileAttributeView(filePath, FileOwnerAttributeView.class, new LinkOption[0]);
        return ownerAttributeView.getOwner().getName();
    }

    private static String getContextStr(String context) {
        return Strings.isNullOrEmpty((String)context) ? "" : context + ": ";
    }
}

