/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core;

import com.amazonaws.util.StringUtils;
import com.google.common.base.Strings;
import java.net.URI;
import net.snowflake.client.core.CachedCredentialType;
import net.snowflake.client.core.Constants;
import net.snowflake.client.core.SFException;
import net.snowflake.client.core.SFLoginInput;
import net.snowflake.client.core.SecureStorageAppleManager;
import net.snowflake.client.core.SecureStorageLinuxManager;
import net.snowflake.client.core.SecureStorageManager;
import net.snowflake.client.core.SecureStorageWindowsManager;
import net.snowflake.client.jdbc.ErrorCode;
import net.snowflake.client.log.SFLogger;
import net.snowflake.client.log.SFLoggerFactory;

public class CredentialManager {
    private static final SFLogger logger = SFLoggerFactory.getLogger(CredentialManager.class);
    private SecureStorageManager secureStorageManager;

    private CredentialManager() {
        this.initSecureStorageManager();
    }

    private void initSecureStorageManager() {
        try {
            if (Constants.getOS() == Constants.OS.MAC) {
                this.secureStorageManager = SecureStorageAppleManager.builder();
            } else if (Constants.getOS() == Constants.OS.WINDOWS) {
                this.secureStorageManager = SecureStorageWindowsManager.builder();
            } else if (Constants.getOS() == Constants.OS.LINUX) {
                this.secureStorageManager = SecureStorageLinuxManager.getInstance();
            } else {
                logger.error("Unsupported Operating System. Expected: OSX, Windows, Linux", false);
            }
        }
        catch (NoClassDefFoundError error) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
        }
    }

    static void resetSecureStorageManager() {
        logger.debug("Resetting the secure storage manager", new Object[0]);
        CredentialManager.getInstance().initSecureStorageManager();
    }

    static void injectSecureStorageManager(SecureStorageManager manager) {
        logger.debug("Injecting secure storage manager", new Object[0]);
        CredentialManager.getInstance().secureStorageManager = manager;
    }

    public static CredentialManager getInstance() {
        return CredentialManagerHolder.INSTANCE;
    }

    static void fillCachedIdToken(SFLoginInput loginInput) throws SFException {
        logger.debug("Looking for cached id token for user: {}, host: {}", loginInput.getUserName(), loginInput.getHostFromServerUrl());
        CredentialManager.getInstance().fillCachedCredential(loginInput, loginInput.getHostFromServerUrl(), loginInput.getUserName(), CachedCredentialType.ID_TOKEN);
    }

    static void fillCachedMfaToken(SFLoginInput loginInput) throws SFException {
        logger.debug("Looking for cached mfa token for user: {}, host: {}", loginInput.getUserName(), loginInput.getHostFromServerUrl());
        CredentialManager.getInstance().fillCachedCredential(loginInput, loginInput.getHostFromServerUrl(), loginInput.getUserName(), CachedCredentialType.MFA_TOKEN);
    }

    static void fillCachedOAuthAccessToken(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        logger.debug("Looking for cached OAuth access token for user: {}, host: {}", loginInput.getUserName(), host);
        CredentialManager.getInstance().fillCachedCredential(loginInput, host, loginInput.getUserName(), CachedCredentialType.OAUTH_ACCESS_TOKEN);
    }

    static void fillCachedOAuthRefreshToken(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        logger.debug("Looking for cached OAuth refresh token for user: {}, host: {}", loginInput.getUserName(), host);
        CredentialManager.getInstance().fillCachedCredential(loginInput, host, loginInput.getUserName(), CachedCredentialType.OAUTH_REFRESH_TOKEN);
    }

    synchronized void fillCachedCredential(SFLoginInput loginInput, String host, String username, CachedCredentialType credType) throws SFException {
        String cred;
        if (StringUtils.isNullOrEmpty((String)username)) {
            logger.debug("Missing username; Cannot read from credential cache", new Object[0]);
            return;
        }
        if (this.secureStorageManager == null) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
            return;
        }
        try {
            cred = this.secureStorageManager.getCredential(host, username, credType.getValue());
        }
        catch (NoClassDefFoundError error) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
            return;
        }
        if (cred == null) {
            logger.debug("Retrieved {} is null", new Object[]{credType});
        }
        logger.debug("Setting {}{} token for user: {}, host: {}", cred == null ? "null " : "", credType.getValue(), username, host);
        switch (credType) {
            case ID_TOKEN: {
                loginInput.setIdToken(cred);
                break;
            }
            case MFA_TOKEN: {
                loginInput.setMfaToken(cred);
                break;
            }
            case OAUTH_ACCESS_TOKEN: {
                loginInput.setOauthAccessToken(cred);
                break;
            }
            case OAUTH_REFRESH_TOKEN: {
                loginInput.setOauthRefreshToken(cred);
                break;
            }
            default: {
                throw new SFException(ErrorCode.INTERNAL_ERROR, new Object[]{"Unrecognized type {} for local cached credential", credType});
            }
        }
    }

    static void writeIdToken(SFLoginInput loginInput, String idToken) throws SFException {
        logger.debug("Caching id token in a secure storage for user: {}, host: {}", loginInput.getUserName(), loginInput.getHostFromServerUrl());
        CredentialManager.getInstance().writeTemporaryCredential(loginInput.getHostFromServerUrl(), loginInput.getUserName(), idToken, CachedCredentialType.ID_TOKEN);
    }

    static void writeMfaToken(SFLoginInput loginInput, String mfaToken) throws SFException {
        logger.debug("Caching mfa token in a secure storage for user: {}, host: {}", loginInput.getUserName(), loginInput.getHostFromServerUrl());
        CredentialManager.getInstance().writeTemporaryCredential(loginInput.getHostFromServerUrl(), loginInput.getUserName(), mfaToken, CachedCredentialType.MFA_TOKEN);
    }

    static void writeOAuthAccessToken(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        logger.debug("Caching OAuth access token in a secure storage for user: {}, host: {}", loginInput.getUserName(), host);
        CredentialManager.getInstance().writeTemporaryCredential(host, loginInput.getUserName(), loginInput.getOauthAccessToken(), CachedCredentialType.OAUTH_ACCESS_TOKEN);
    }

    static void writeOAuthRefreshToken(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        logger.debug("Caching OAuth refresh token in a secure storage for user: {}, host: {}", loginInput.getUserName(), host);
        CredentialManager.getInstance().writeTemporaryCredential(host, loginInput.getUserName(), loginInput.getOauthRefreshToken(), CachedCredentialType.OAUTH_REFRESH_TOKEN);
    }

    synchronized void writeTemporaryCredential(String host, String user, String cred, CachedCredentialType credType) {
        if (StringUtils.isNullOrEmpty((String)user)) {
            logger.debug("Missing username; Cannot write to credential cache", new Object[0]);
            return;
        }
        if (Strings.isNullOrEmpty((String)cred)) {
            logger.debug("No {} is given.", new Object[]{credType});
            return;
        }
        if (this.secureStorageManager == null) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
            return;
        }
        try {
            this.secureStorageManager.setCredential(host, user, credType.getValue(), cred);
        }
        catch (NoClassDefFoundError error) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
        }
    }

    static void deleteIdTokenCache(String host, String user) {
        logger.debug("Removing cached id token from a secure storage for user: {}, host: {}", user, host);
        CredentialManager.getInstance().deleteTemporaryCredential(host, user, CachedCredentialType.ID_TOKEN);
    }

    static void deleteMfaTokenCache(String host, String user) {
        logger.debug("Removing cached mfa token from a secure storage for user: {}, host: {}", user, host);
        CredentialManager.getInstance().deleteTemporaryCredential(host, user, CachedCredentialType.MFA_TOKEN);
    }

    static void deleteOAuthAccessTokenCache(String host, String user) {
        logger.debug("Removing cached mfa token from a secure storage for user: {}, host: {}", user, host);
        CredentialManager.getInstance().deleteTemporaryCredential(host, user, CachedCredentialType.OAUTH_ACCESS_TOKEN);
    }

    static void deleteOAuthAccessTokenCache(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        CredentialManager.deleteOAuthAccessTokenCache(host, loginInput.getUserName());
    }

    static void deleteOAuthRefreshTokenCache(SFLoginInput loginInput) throws SFException {
        String host = CredentialManager.getHostForOAuthCacheKey(loginInput);
        CredentialManager.deleteOAuthRefreshTokenCache(host, loginInput.getUserName());
    }

    static void deleteOAuthRefreshTokenCache(String host, String user) {
        logger.debug("Removing cached OAuth refresh token from a secure storage for user: {}, host: {}", user, host);
        CredentialManager.getInstance().deleteTemporaryCredential(host, user, CachedCredentialType.OAUTH_REFRESH_TOKEN);
    }

    static String getHostForOAuthCacheKey(SFLoginInput loginInput) throws SFException {
        String oauthTokenRequestUrl = loginInput.getOauthLoginInput().getTokenRequestUrl();
        if (oauthTokenRequestUrl != null) {
            URI parsedUrl = URI.create(oauthTokenRequestUrl);
            return parsedUrl.getHost();
        }
        return loginInput.getHostFromServerUrl();
    }

    synchronized void deleteTemporaryCredential(String host, String user, CachedCredentialType credType) {
        if (this.secureStorageManager == null) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
            return;
        }
        if (StringUtils.isNullOrEmpty((String)user)) {
            logger.debug("Missing username; Cannot delete from credential cache", new Object[0]);
            return;
        }
        try {
            this.secureStorageManager.deleteCredential(host, user, credType.getValue());
        }
        catch (NoClassDefFoundError error) {
            CredentialManager.logMissingJnaJarForSecureLocalStorage();
        }
    }

    private static void logMissingJnaJarForSecureLocalStorage() {
        logger.warn("JNA jar files are needed for Secure Local Storage service. Please follow the Snowflake JDBC instruction for Secure Local Storage feature. Fall back to normal process.", false);
    }

    private static class CredentialManagerHolder {
        private static final CredentialManager INSTANCE = new CredentialManager();

        private CredentialManagerHolder() {
        }
    }
}

