/*
 * Copyright (c) 2012-2016 Snowflake Computing Inc. All right reserved.
 */

package net.snowflake.client.core;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.GZIPOutputStream;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ByteArrayEntity;

/**
 * Incident Event class for triggering and registering Incidents with GS
 * @author jrosen
 */
public class Incident extends Event
{
  private static final Logger logger = Logger.getLogger(Incident.class.getName());

  private static final String SF_PATH_CREATE_INCIDENT = "/incidents/create-incident";

  private static final Random randomGenerator = new Random();

  private final Map<String,Object> incident;

  public Incident(Event.EventType type, Map<String,Object> incident)
  {
    super(type, "");

    this.incident = incident;
  }

  /**
   * Generates the Incident Identifier. This should be reasonably unique within
   * the time instant generated. I.e. in case there happens to be a collision
   * (unlikely), then the id together with the timestamp can be used to
   * disambiguate the two events. (Stolen from GS)
   * @return incident id
   */
  public static String generateIncidentId()
  {
    return Integer.toString(randomGenerator.nextInt(8999999) + 1000000);
  }

  /**
   * Registers the incident with Global Services
   */
  @Override
  public void flush()
  {
    ObjectMapper mapper = new ObjectMapper();
    String json;
    String response;
    HttpClient httpClient;
    URI incidentURI;
    HttpPost postRequest;

    logger.log(Level.FINE, "Flushing incident, type={0}, msg={1}",
        new Object[]{this.getType().getDescription(), this.getMessage()});

    // Get session token and incident info
    String sessionToken =
        (String)incident.get(SFSession.SF_HEADER_TOKEN_TAG);
    String serverUrl =
        (String)incident.get(SFSessionProperty.SERVER_URL.getPropertyKey());
    Map<String,Object> incidentInfo =
        (Map<String,Object>)incident.get(IncidentUtil.INCIDENT_INFO);

    if (sessionToken == null || serverUrl == null)
    {
      logger.log(Level.FINE, "Incident registration failed, sessionToken or "
              + "serverUrl not specified");
      return;
    }

    // Map the incident to a json payload
    try
    {
      json = mapper.writeValueAsString(incidentInfo);
    }
    catch (JsonProcessingException ex)
    {
      logger.log(Level.SEVERE, "Incident registration failed, could not map "
              + "incident report to json string. Exception: {0}", ex.getMessage());
      return;
    }

    // Sanity check...
    Preconditions.checkNotNull(json);

    httpClient = HttpUtil.getHttpClient();

    try
    {
      URIBuilder uriBuilder = new URIBuilder(serverUrl);
      uriBuilder.setPath(SF_PATH_CREATE_INCIDENT);
      incidentURI = uriBuilder.build();
    }
    catch (URISyntaxException ex)
    {
      logger.log(Level.SEVERE, "Incident registration failed, "
              + "URI could not be built. Exception: {0}", ex.getMessage());
      return;
    }

    logger.log(Level.FINE, "Creating new HTTP request, token={0}, URL={1}",
        new Object[]{sessionToken, incidentURI.toString()});

    postRequest = new HttpPost(incidentURI);
    postRequest.setHeader(SFSession.SF_HEADER_AUTHORIZATION,
          SFSession.SF_HEADER_SNOWFLAKE_AUTHTYPE + " "
              + SFSession.SF_HEADER_TOKEN_TAG + "=\""
              + sessionToken + "\"");

    // Compress the payload.
    ByteArrayEntity input = null;
    try
    {
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      GZIPOutputStream gzos = new GZIPOutputStream(baos);
      byte[] bytes = json.getBytes("UTF-8");
      gzos.write(bytes);
      gzos.finish();
      input = new ByteArrayEntity(baos.toByteArray());
      input.setContentType("application/json");
    }
    catch(IOException exc)
    {
      logger.log(Level.WARNING, "Incident registration failed, could not compress"
              + " payload. Exception: {0}", exc.getMessage());
    }

    postRequest.setEntity(input);
    postRequest.addHeader("content-encoding", "gzip");

    response = null;

    try
    {
      response = HttpUtil.executeRequest(postRequest,
                                         httpClient,
                                         1000, 0,
                                         null);
    }
    catch (Exception ex)
    {
      // No much we can do here besides complain.
      logger.log(Level.SEVERE,
          "Incident registration request failed, " +
              "response: {0}, exception: {1}", new Object[]{response,
                                                            ex.getMessage()});
    }
  }
}
