/*
 * Copyright (c) 2012-2016 Snowflake Computing Inc. All right reserved.
 */

package net.snowflake.client.core;

import net.snowflake.client.jdbc.ErrorCode;
import net.snowflake.client.jdbc.SnowflakeSQLException;
import net.snowflake.client.jdbc.SnowflakeUtil;
import com.snowflake.gscommon.core.SFBinary;
import com.snowflake.gscommon.core.SFBinaryFormat;
import com.snowflake.gscommon.core.SFTime;
import com.snowflake.gscommon.core.SFTimestamp;
import com.snowflake.gscommon.core.SnowflakeDateTimeFormat;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.sql.Date;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Base class for query result set and metadata result set
 *
 * @author jhuang
 */
public abstract class SFBaseResultSet
{
  static final Logger logger = Logger.getLogger(SFBaseResultSet.class.getName());

  protected boolean wasNull = false;

  protected Object[] nextRow = null;

  protected SFResultSetMetaData resultSetMetaData = null;

  protected int row = 0;

  protected Map<String, Object> parameters = new HashMap<>();

  protected TimeZone timeZone;

  // Timezone used for TimestampNTZ
  private static TimeZone timeZoneUTC = TimeZone.getTimeZone("UTC");

  // Formatters for different datatypes
  protected SnowflakeDateTimeFormat timestampNTZFormatter;
  protected SnowflakeDateTimeFormat timestampLTZFormatter;
  protected SnowflakeDateTimeFormat timestampTZFormatter;
  protected SnowflakeDateTimeFormat dateFormatter;
  protected SnowflakeDateTimeFormat timeFormatter;
  protected boolean honorClientTZForTimestampNTZ = true;
  protected SFBinaryFormat binaryFormatter;

  protected long resultVersion = 0;

  protected int numberOfBinds = 0;

  // For creating incidents
  protected SFSession session;

  // indicate whether the result set has been closed or not.
  protected boolean isClosed = true;

  public void setSession(SFSession session)
  {
    this.session = session;
  }

  // default implementation
  public boolean next() throws SFException, SnowflakeSQLException
  {
    logger.log(Level.FINER, "public boolean next()");

    return false;
  }

  public void close()
  {
    if (logger.isLoggable(Level.FINER))
      logger.log(Level.FINER, "Time: " + System.currentTimeMillis() +
          " public void close()");

    resultSetMetaData = null;
    isClosed = true;
  }

  public boolean wasNull()
  {
    if (logger.isLoggable(Level.FINER))
      logger.log(Level.FINER, "public boolean wasNull() returning " + wasNull);

    return wasNull;
  }

  public String getString(int columnIndex) throws SFException
  {
    logger.log(Level.FINER, "public String getString(int columnIndex)");

    // Column index starts from 1, not 0.
    Object obj = getObjectInternal(columnIndex);
    if (obj == null)
    {
      return null;
    }

    // print timestamp in string format
    int columnType = resultSetMetaData.getInternalColumnType(columnIndex);
    switch (columnType)
    {
      case Types.BOOLEAN:
        return ResultUtil.getBooleanAsString(
            ResultUtil.getBoolean(obj.toString()));

      case Types.TIMESTAMP:
      case SnowflakeUtil.EXTRA_TYPES_TIMESTAMP_LTZ:
      case SnowflakeUtil.EXTRA_TYPES_TIMESTAMP_TZ:

        SFTimestamp sfTS = getSFTimestamp(columnIndex);
        int columnScale = resultSetMetaData.getScale(columnIndex);

        String timestampStr = ResultUtil.getSFTimestampAsString(
            sfTS, columnType, columnScale, timestampNTZFormatter,
            timestampLTZFormatter, timestampTZFormatter, session);

        if (logger.isLoggable(Level.FINER))
          logger.log(Level.FINER, "Converting timestamp to string from: "
              + obj.toString() + " to: " + timestampStr);

        return timestampStr;

      case Types.DATE:
        Date date = getDate(columnIndex, timeZoneUTC);

        if (dateFormatter == null)
        {
          throw IncidentUtil.
              generateIncidentWithException(session, null, null,
                                            ErrorCode.INTERNAL_ERROR,
                                            "missing date formatter");
        }

        String dateStr = ResultUtil.getDateAsString(date, dateFormatter);

        if (logger.isLoggable(Level.FINER))
          logger.log(Level.FINER, "Converting date to string from: "
              + obj.toString() + " to: " + dateStr);
        return dateStr;

      case Types.TIME:
        SFTime sfTime = getSFTime(columnIndex);

        if (timeFormatter == null)
        {
          throw IncidentUtil
              .generateIncidentWithException(session, null, null,
                                             ErrorCode.INTERNAL_ERROR,
                                             "missing time formatter");
        }

        int scale = resultSetMetaData.getScale(columnIndex);
        String timeStr = ResultUtil.getSFTimeAsString(sfTime, scale, timeFormatter);

        if (logger.isLoggable(Level.FINER))
          logger.log(Level.FINER, "Converting time to string from: "
              + obj.toString() + " to: " + timeStr);
        return timeStr;

      case Types.BINARY:
        if (binaryFormatter == null)
        {
          throw IncidentUtil
              .generateIncidentWithException(session, null, null,
                                             ErrorCode.INTERNAL_ERROR,
                                             "missing binary formatter");
        }

        if (binaryFormatter == SFBinaryFormat.HEX)
        {
          // Shortcut: the values are already passed with hex encoding, so just
          // return the string unchanged rather than constructing an SFBinary.
          return obj.toString();
        }

        SFBinary sfb = new SFBinary(getBytes(columnIndex));
        return binaryFormatter.format(sfb);

      default:
        break;
    }

    return obj.toString();
  }

  public boolean getBoolean(int columnIndex) throws SFException
  {
    logger.log(Level.FINER,
        "public boolean getBoolean(int columnIndex)");

    Object obj = getObjectInternal(columnIndex);
    if (obj == null)
    {
      return false;
    }

    if (obj instanceof Boolean)
    {
      return (Boolean) obj;
    }
    else
    {
      return ResultUtil.getBoolean(obj.toString());
    }
  }

  public short getShort(int columnIndex) throws SFException
  {
    logger.log(Level.FINER, "public short getShort(int columnIndex)");

    // Column index starts from 1, not 0.
    Object obj = getObjectInternal(columnIndex);

    if (obj == null)
      return 0;

    if (obj instanceof String)
    {
      return Short.parseShort((String) obj);
    }
    else
    {
      return ((Number) obj).shortValue();
    }
  }

  public int getInt(int columnIndex) throws SFException
  {
    logger.log(Level.FINER, "public int getInt(int columnIndex)");

    // Column index starts from 1, not 0.
    Object obj = getObjectInternal(columnIndex);

    if (obj == null)
      return 0;

    if (obj instanceof String)
    {
      return Integer.parseInt((String) obj);
    }
    else
    {
      return ((Number) obj).intValue();
    }
  }

  public long getLong(int columnIndex) throws SFException
  {
    logger.log(Level.FINER, "public long getLong(int columnIndex)");

    // Column index starts from 1, not 0.
    Object obj = getObjectInternal(columnIndex);

    if (obj == null)
      return 0;

    try
    {
      if (obj instanceof String)
      {
        return Long.parseLong((String) obj);
      }
      else
      {
        return ((Number) obj).longValue();
      }
    }
    catch (NumberFormatException nfe)
    {
      throw IncidentUtil.
          generateIncidentWithException(session, null, null,
                                        ErrorCode.INTERNAL_ERROR,
                                        "Invalid long: " + obj.toString());
    }
  }

  public float getFloat(int columnIndex) throws SFException
  {
    logger.log(Level.FINER, "public float getFloat(int columnIndex)");

    // Column index starts from 1, not 0.
    Object obj = getObjectInternal(columnIndex);

    if (obj == null)
      return 0;

    if (obj instanceof String)
    {
      return Float.parseFloat((String) obj);
    }
    else
    {
      return ((Number) obj).floatValue();
    }
  }

  public double getDouble(int columnIndex) throws SFException
  {
    logger.log(Level.FINER, "public double getDouble(int columnIndex)");

    // Column index starts from 1, not 0.
    Object obj = getObjectInternal(columnIndex);

    // snow-11974: null for getDouble should return 0
    if (obj == null)
      return 0;

    if (obj instanceof String)
    {
      return Double.parseDouble((String) obj);
    }
    else
    {
      return ((Number) obj).doubleValue();
    }
  }

  public byte[] getBytes(int columnIndex) throws SFException
  {
    logger.log(Level.FINER, "public byte[] getBytes(int columnIndex)");

    // Column index starts from 1, not 0.
    Object obj = getObjectInternal(columnIndex);

    if (obj == null)
      return null;

    try
    {
      return SFBinary.fromHex(obj.toString()).getBytes();
    }
    catch (IllegalArgumentException ex)
    {
      throw new SFException(ErrorCode.INTERNAL_ERROR,
          "Invalid binary value: " + obj.toString());
    }
  }

  public Date getDate(int columnIndex, TimeZone tz) throws SFException
  {
    if (tz == null)
    {
      tz = TimeZone.getDefault();
    }

    logger.log(Level.FINER, "public Date getDate(int columnIndex)");

    // Column index starts from 1, not 0.
    Object obj = getObjectInternal(columnIndex);

    if (obj == null)
      return null;

    return ResultUtil.getDate(obj.toString(), tz, session);
  }

  public Date getDate(int columnIndex) throws SFException
  {
    return getDate(columnIndex, TimeZone.getDefault());
  }

  public Time getTime(int columnIndex) throws SFException
  {
    logger.log(Level.FINER, "public Time getTime(int columnIndex)");

    SFTime sfTime = getSFTime(columnIndex);
    if (sfTime == null)
    {
      return null;
    }

    return new Time(sfTime.getFractionalSeconds(3));
  }

  private SFTime getSFTime(int columnIndex) throws SFException
  {
    Object obj = getObjectInternal(columnIndex);

    if (obj == null)
      return null;

    int scale = resultSetMetaData.getScale(columnIndex);
    return ResultUtil.getSFTime(obj.toString(), scale, session);
  }

  private Timestamp getTimestamp(int columnIndex) throws SFException
  {
    return getTimestamp(columnIndex, TimeZone.getDefault());
  }

  public Timestamp getTimestamp(int columnIndex, TimeZone tz)
      throws SFException
  {
    SFTimestamp sfTS = getSFTimestamp(columnIndex);

    if (sfTS == null)
    {
      return null;
    }

    Timestamp res = sfTS.getTimestamp();

    if (res == null)
    {
      return null;
    }

    // SNOW-14777: for timestamp_ntz, we should treat the time as in client time
    // zone so adjust the timestamp by subtracting the offset of the client
    // timezone
    if (honorClientTZForTimestampNTZ &&
        resultSetMetaData.getInternalColumnType(columnIndex) == Types.TIMESTAMP)
    {
      return sfTS.moveToTimeZone(tz).getTimestamp();
    }

    return res;
  }

  private SFTimestamp getSFTimestamp(int columnIndex) throws SFException
  {
    logger.log(Level.FINER,
               "public Timestamp getTimestamp(int columnIndex)");

    Object obj = getObjectInternal(columnIndex);

    if (obj == null)
      return null;

    return ResultUtil.getSFTimestamp(
        obj.toString(),
        resultSetMetaData.getScale(columnIndex),
        resultSetMetaData.getInternalColumnType(columnIndex),
        resultVersion, timeZone, session);
  }

  public SFResultSetMetaData getMetaData() throws SFException
  {
    logger.log(Level.FINER, "public ResultSetMetaData getMetaData()");

    return resultSetMetaData;
  }

  protected abstract Object getObjectInternal(int columnIndex) throws SFException;

  public Object getObject(int columnIndex) throws SFException
  {
    logger.log(Level.FINER,
               "public Object getObject(int columnIndex)");

    int type = resultSetMetaData.getColumnType(columnIndex);

    Object obj = getObjectInternal(columnIndex);
    if (obj == null)
      return null;

    switch(type)
    {
      case Types.VARCHAR:
      case Types.CHAR:
        return getString(columnIndex);

      case Types.BINARY:
        return getBytes(columnIndex);

      case Types.INTEGER:
        return getInt(columnIndex);

      case Types.DECIMAL:
        return getBigDecimal(columnIndex);

      case Types.DOUBLE:
        return getDouble(columnIndex);

      case Types.TIMESTAMP:
        return getTimestamp(columnIndex);

      case Types.DATE:
        return getDate(columnIndex);

      case Types.TIME:
        return getTime(columnIndex);

      case Types.BOOLEAN:
        return getBoolean(columnIndex);

      default:
        throw IncidentUtil.
            generateIncidentWithException(session, null, null,
                                          ErrorCode.FEATURE_UNSUPPORTED,
                                          "data type: " + type);
    }
  }

  public BigDecimal getBigDecimal(int columnIndex) throws SFException
  {
    logger.log(Level.FINER,
               "public BigDecimal getBigDecimal(int columnIndex)");


    // Column index starts from 1, not 0.
    Object obj = getObjectInternal(columnIndex);

    if (obj == null)
      return null;

    return new BigDecimal(obj.toString());
  }

  public BigDecimal getBigDecimal(int columnIndex, int scale) throws SFException
  {
    logger.log(Level.FINER,
        "public BigDecimal getBigDecimal(int columnIndex)");


    Object obj = getObjectInternal(columnIndex);

    if (obj == null)
      return null;

    BigDecimal value = new BigDecimal(obj.toString());

    value = value.setScale(scale, RoundingMode.HALF_UP);

    return value;
  }

  public int getRow() throws SQLException
  {
    logger.log(Level.FINER, "public int getRow()");

    return row;
  }

  public boolean absolute(int row) throws SFException
  {
    logger.log(Level.FINER, "public boolean absolute(int row)");

    throw new SFException(
            ErrorCode.FEATURE_UNSUPPORTED, "seek to a specific row");
  }

  public boolean relative(int rows) throws SFException
  {
    logger.log(Level.FINER, "public boolean relative(int rows)");

    throw new SFException(
            ErrorCode.FEATURE_UNSUPPORTED, "seek to a row relative to current row");
  }

  public boolean previous() throws SFException
  {
    logger.log(Level.FINER, "public boolean previous()");

    throw new SFException(
            ErrorCode.FEATURE_UNSUPPORTED, "seek to a previous row");
  }

  protected int getNumberOfBinds()
  {
    return numberOfBinds;
  }

  public boolean isFirst()
  {
    logger.log(Level.FINER, "public boolean isFirst()");

    return row == 1;
  }

  public boolean isClosed()
  {
    return isClosed;
  }
}
