/*
 * Copyright (c) 2012-2016 Snowflake Computing Inc. All right reserved.
 */

package net.snowflake.client.core;

import net.snowflake.client.jdbc.ErrorCode;
import net.snowflake.client.jdbc.SnowflakeBaseResultSet;
import net.snowflake.client.jdbc.SnowflakeFixedView;
import net.snowflake.client.jdbc.SnowflakeResultSetMetaData;
import net.snowflake.client.jdbc.SnowflakeSQLException;
import com.snowflake.gscommon.core.SqlState;

import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Fixed view result set. This class iterates through any fixed view
 * implementation and return the objects as rows
 *
 * @author jhuang
 */
public class SFFixedViewResultSet extends SFBaseResultSet
{

  static final Logger logger = Logger.getLogger(
      SFFixedViewResultSet.class.getName());

  private SnowflakeFixedView fixedView;
  private Object[] nextRow = null;

  public SFFixedViewResultSet(SnowflakeFixedView fixedView)
          throws SnowflakeSQLException
  {
    this.fixedView = fixedView;

    try
    {
      resultSetMetaData
          = new SFResultSetMetaData(fixedView.describeColumns(), session,
                                    timestampNTZFormatter,
                                    timestampLTZFormatter,
                                    timestampTZFormatter,
                                    dateFormatter,
                                    timeFormatter);

    }
    catch (Exception ex)
    {
      throw new SnowflakeSQLException(ex, SqlState.INTERNAL_ERROR,
                                      ErrorCode.INTERNAL_ERROR.getMessageCode(),
                                      "Failed to describe fixed view: "
                                      + fixedView.getClass().getName());
    }
  }

  /**
   * Advance to next row
   *
   * @return true if next row exists, false otherwise
   * @throws net.snowflake.client.core.SFException if failed to get next row
   */
  @Override
  public boolean next() throws SFException
  {
    logger.log(Level.FINER, "next called");

    List<Object> nextRowList;
    try
    {
      // call the fixed view next row method
      nextRowList = fixedView.getNextRow();
    }
    catch (Exception ex)
    {
      throw IncidentUtil.
          generateIncidentWithException(session, null, null,
                                        ex, ErrorCode.INTERNAL_ERROR,
                                        "Error getting next row from fixed view");
    }

    if (nextRowList == null)
    {
      logger.log(Level.FINER, "end of result");
      return false;
    }

    row++;

    if (nextRow == null)
    {
      nextRow = new Object[nextRowList.size()];
    }
    nextRow = nextRowList.toArray(nextRow);

    return true;
  }

  @Override
  protected Object getObjectInternal(int columnIndex) throws SFException
  {
    logger.log(Level.FINER,
               "public Object getObjectInternal(int columnIndex)");

    if (nextRow == null)
    {
      throw IncidentUtil.
          generateIncidentWithException(session, null, null,
                                        ErrorCode.ROW_DOES_NOT_EXIST);
    }

    if (columnIndex <= 0 || columnIndex > nextRow.length)
    {
      throw new SFException(ErrorCode.COLUMN_DOES_NOT_EXIST, columnIndex);
    }

    wasNull = nextRow[columnIndex - 1] == null;

    return nextRow[columnIndex - 1];
  }

  @Override
  public void close()
  {
    super.close();
    // free the object so that they can be Garbage collected
    nextRow = null;
    fixedView = null;
  }
}
