/*
 * Copyright (c) 2012-2016 Snowflake Computing Inc. All right reserved.
 */

package net.snowflake.client.jdbc;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import net.snowflake.client.core.SFException;
import net.snowflake.client.core.SFSession;
import net.snowflake.client.core.SFSessionProperty;
import com.snowflake.gscommon.core.LoginInfoDTO;
import com.snowflake.gscommon.core.ResourceBundleManager;
import com.snowflake.gscommon.core.SqlState;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpRequestBase;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.sql.Array;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.NClob;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLClientInfoException;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Savepoint;
import java.sql.Statement;
import java.sql.Struct;
import java.util.*;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Snowflake connection implementation
 *
 * @author jhuang
 */
public class SnowflakeConnectionV1 implements Connection
{

  static final
  Logger logger = Logger.getLogger(SnowflakeConnectionV1.class.getName());

  private static final String JDBC_PROTOCOL_PREFIX = "jdbc:snowflake";

  private static final String NATIVE_PROTOCOL = "http";

  private static final String SSL_NATIVE_PROTOCOL = "https";

  private boolean isClosed = true;

  private String sessionToken;
  private String masterToken;

  private String userName;

  private String password;

  private String accountName;

  private String databaseName;

  private String schemaName;

  private String warehouse;

  protected Level tracingLevel = Level.INFO;

  private String serverUrl;

  private boolean sslOn = true;

  private boolean passcodeInPassword = false;

  private String passcode;

  private String role;

  private String authenticator;

  private List<SQLWarning> sqlWarnings = new LinkedList<SQLWarning>();

  private String newClientForUpdate;

  private Map<String, Object> sessionParameters = new HashMap<>();

  // Injected delay for the purpose of connection timeout testing
  // Any statement execution will sleep for the specified number of milliseconds
  private AtomicInteger _injectedDelay = new AtomicInteger(0);

  private String databaseVersion = null;
  private int databaseMajorVersion = 0;
  private int databaseMinorVersion = 0;

  /**
   * Amount of seconds a user is willing to tolerate for establishing the
   * connection with database. In our case, it means the first login
   * request to get authorization token.
   *
   * A value of 0 means no timeout.
   *
   * Default: to login timeout in driver manager if set or 60 seconds
   */
  private int loginTimeout = (DriverManager.getLoginTimeout() > 0)?
                             DriverManager.getLoginTimeout() : 60;

  /**
   * Amount of milliseconds a user is willing to tolerate for network related
   * issues (e.g. HTTP 503/504) or database transient issues (e.g. GS
   * not responding)
   *
   * A value of 0 means no timeout
   *
   * Default: 300 seconds
   */
  private int networkTimeoutInMilli = 0; // in milliseconds

  /**
   * Amount of seconds a user is willing to tolerate for an individual query.
   * Both network/GS issues and query processing itself can contribute
   * to the amount time spent for a query.
   *
   * A value of 0 means no timeout
   *
   * Default: 0
   */
  private int queryTimeout = 0; // in seconds

  private boolean useProxy = false;

  private AtomicInteger sequenceId = new AtomicInteger(0);

  private Map sessionProperties = new HashMap<String, Object>(1);

  private final static ObjectMapper mapper = new ObjectMapper();

  private static String IMPLEMENTATION_VERSION_TESTING =
      Integer.MAX_VALUE + ".0.0";
  private static Long SVN_REVISION_TESTING = Long.MAX_VALUE;

  static final ResourceBundleManager errorResourceBundleManager =
  ResourceBundleManager.getSingleton(ErrorCode.errorMessageResource);

  boolean internalTesting = false;

  private Properties clientInfo = new Properties();

  // TODO this should be set to Connection.TRANSACTION_READ_COMMITTED
  // TODO There may not be many implications here since the call to
  // TODO setTransactionIsolation doesn't do anything.
  private int transactionIsolation = Connection.TRANSACTION_NONE;

  //--- Simulated failures for testing

  // whether we try to simulate a socket timeout (a default value of 0 means
  // no simulation). The value is in milliseconds
  private int injectSocketTimeout = 0;

  // simulate client pause after initial execute and before first get-result
  // call ( a default value of 0 means no pause). The value is in seconds
  private int injectClientPause = 0;

  //Generate exception while uploading file with a given name
  private String injectFileUploadFailure = null;

  private boolean useV1QueryAPI = false;

  private boolean retryQuery = false;

  private SFSession sfSession;

  /**
   * A connection will establish a session token from snowflake
   *
   * @param url server url used to create snowflake connection
   * @param info properties about the snowflake connection
   * @throws java.sql.SQLException if failed to create a snowflake connection
   *         i.e. username or password not specified
   */
  public SnowflakeConnectionV1(String url,
                               Properties info)
          throws SQLException
  {
    processParameters(url, info);

    // userName and password are expected
    if (userName == null || userName.isEmpty())
    {
      throw new SQLException(
              errorResourceBundleManager.getLocalizedMessage(
                      ErrorCode.MISSING_USERNAME.getMessageCode().toString()));
    }

    if (password == null || userName.isEmpty())
    {
      throw new SQLException(
              errorResourceBundleManager.getLocalizedMessage(
                      ErrorCode.MISSING_PASSWORD.getMessageCode().toString()));
    }

    // replace protocol name
    serverUrl = serverUrl.replace(JDBC_PROTOCOL_PREFIX,
                                  sslOn?SSL_NATIVE_PROTOCOL:NATIVE_PROTOCOL);

    logger.log(Level.FINER, "Connecting to: " + serverUrl + " with " +
                           "userName=" + userName +
                           " accountName=" + accountName +
                           " databaseName=" + databaseName +
                           " schemaName=" + schemaName +
                           " warehouse=" + warehouse +
                           " ssl=" + sslOn);

    // open connection to GS
    sfSession = new SFSession();

    try
    {
      // pass the parameters to sfSession
      initSessionProperties();

      sfSession.open();

      // SNOW-18963: return database version
      databaseVersion = sfSession.getDatabaseVersion();
      databaseMajorVersion = sfSession.getDatabaseMajorVersion();
      databaseMinorVersion = sfSession.getDatabaseMinorVersion();
      newClientForUpdate = sfSession.getNewClientForUpdate();
    }
    catch (SFException ex)
    {
      throw new SnowflakeSQLException(ex.getCause(), ex.getSqlState(),
          ex.getVendorCode(), ex.getParams());
    }
    isClosed = false;
  }

  private void initSessionProperties() throws SFException
  {
    sfSession.addProperty(
        SFSessionProperty.SERVER_URL.getPropertyKey(), serverUrl);
    sfSession.addProperty(SFSessionProperty.USER.getPropertyKey(), userName);
    sfSession.addProperty(SFSessionProperty.PASSWORD.getPropertyKey(), password);
    sfSession.addProperty(
        SFSessionProperty.ACCOUNT.getPropertyKey(), accountName);

    if (databaseName != null)
      sfSession.addProperty(
          SFSessionProperty.DATABASE.getPropertyKey(), databaseName);
    if (schemaName != null)
      sfSession.addProperty(
          SFSessionProperty.SCHEMA.getPropertyKey(), schemaName);
    if (warehouse != null)
      sfSession.addProperty(
          SFSessionProperty.WAREHOUSE.getPropertyKey(), warehouse);
    if (role != null)
      sfSession.addProperty(SFSessionProperty.ROLE.getPropertyKey(), role);
    if (authenticator != null)
      sfSession.addProperty(SFSessionProperty.AUTHENTICATOR.getPropertyKey(),
          authenticator);
    sfSession.addProperty(SFSessionProperty.APP_ID.getPropertyKey(),
        LoginInfoDTO.SF_JDBC_APP_ID);

    if (internalTesting)
    {
      sfSession.addProperty(SFSessionProperty.APP_VERSION.getPropertyKey(),
          SnowflakeConnectionV1.IMPLEMENTATION_VERSION_TESTING);

      sfSession.addProperty(SFSessionProperty.APP_BUILD_ID.getPropertyKey(),
          String.valueOf(SnowflakeConnectionV1.SVN_REVISION_TESTING));
    }
    else
    {
      sfSession.addProperty(SFSessionProperty.APP_VERSION.getPropertyKey(),
          SnowflakeDriver.implementVersion);

      sfSession.addProperty(SFSessionProperty.APP_BUILD_ID.getPropertyKey(),
          String.valueOf(SnowflakeDriver.svnRevision));
    }

    sfSession.addProperty(SFSessionProperty.LOGIN_TIMEOUT.getPropertyKey(),
        loginTimeout);

    sfSession.addProperty(SFSessionProperty.NETWORK_TIMEOUT.getPropertyKey(),
        networkTimeoutInMilli);

    sfSession.addProperty(SFSessionProperty.USE_PROXY.getPropertyKey(),
        useProxy);

    sfSession.addProperty(
        SFSessionProperty.INJECT_SOCKET_TIMEOUT.getPropertyKey(),
        injectSocketTimeout);

    sfSession.addProperty(
        SFSessionProperty.INJECT_CLIENT_PAUSE.getPropertyKey(),
        injectClientPause);

    sfSession.addProperty(
        SFSessionProperty.PASSCODE.getPropertyKey(),
        passcode);

    sfSession.addProperty(
        SFSessionProperty.PASSCODE_IN_PASSWORD.getPropertyKey(),
        passcodeInPassword);

    // Now add the session parameters
    for (String param_name: sessionParameters.keySet())
    {
      Object param_value = sessionParameters.get(param_name);
      sfSession.addProperty(param_name, param_value);
    }
  }

  private void processParameters(String url, Properties info)
  {
    /*
     * Extract accountName, databaseName, schemaName from
     * the URL if it is specified there
     *
     * URL is in the form of:
     * jdbc:snowflake://host:port/?user=v&password=v&account=v&
     * db=v&schema=v&ssl=v&[passcode=v|passcodeInPassword=on]
     */

    serverUrl = url;

    /*
     * Find the query parameter substring if it exists and extract properties
     * out of it
     */
    int queryParamsIndex = url.indexOf("/?");

    if(queryParamsIndex > 0)
    {
      String queryParams = url.substring(queryParamsIndex+2);

      String[] tokens = StringUtils.split(queryParams, "=&");

      // update the server url for REST call to GS
      serverUrl = serverUrl.substring(0, queryParamsIndex);

      logger.log(Level.FINE, "server url: {0}", serverUrl);

      //assert that tokens lenth is even so that there is a value for each param
      if(tokens.length%2 != 0)
      {
        throw new
        IllegalArgumentException("Missing value for some query param");
      }

      for(int paramIdx = 0; paramIdx < tokens.length; paramIdx=paramIdx+2)
      {
        if("user".equalsIgnoreCase(tokens[paramIdx]))
        {
          userName = tokens[paramIdx+1];

          logger.log(Level.FINE, "user name: {0}", userName);

        }
        else if("password".equalsIgnoreCase(tokens[paramIdx]))
        {
          password = tokens[paramIdx+1];
        }
        else if("account".equalsIgnoreCase(tokens[paramIdx]))
        {
          accountName = tokens[paramIdx+1];

          logger.log(Level.FINE, "account: {0}", accountName);

        }
        else if("db".equalsIgnoreCase(tokens[paramIdx]) ||
            "database".equalsIgnoreCase(tokens[paramIdx]))
        {
          databaseName = tokens[paramIdx+1];

          logger.log(Level.FINE, "db: {0}", databaseName);
        }
        else if("schema".equalsIgnoreCase(tokens[paramIdx]))
        {
          schemaName = tokens[paramIdx+1];
          logger.log(Level.FINE, "schema: {0}", schemaName);
        }
        else if("ssl".equalsIgnoreCase(tokens[paramIdx]))
        {
          sslOn = !("off".equalsIgnoreCase(tokens[paramIdx+1]) ||
          "false".equalsIgnoreCase(tokens[paramIdx+1]));

          logger.log(Level.FINE, "ssl: {0}", tokens[paramIdx+1]);

        }
        else if("passcodeInPassword".equalsIgnoreCase(tokens[paramIdx]))
        {
          passcodeInPassword = "on".equalsIgnoreCase(tokens[paramIdx+1]) ||
              "true".equalsIgnoreCase(tokens[paramIdx+1]);

          logger.log(Level.FINE, "passcodeInPassword: {0}", tokens[paramIdx+1]);
        }
        else if("passcode".equalsIgnoreCase(tokens[paramIdx]))
        {
          passcode = tokens[paramIdx+1];
        }
        else if("role".equalsIgnoreCase(tokens[paramIdx]))
        {
          role = tokens[paramIdx+1];
          logger.log(Level.FINE, "role: {0}", role);
        }
        else if("authenticator".equalsIgnoreCase(tokens[paramIdx]))
        {
          authenticator = tokens[paramIdx+1];
        }
        else if("internal".equalsIgnoreCase(tokens[paramIdx]))
        {
          internalTesting = "true".equalsIgnoreCase(tokens[paramIdx+1]);
        }
        else if("warehouse".equalsIgnoreCase(tokens[paramIdx]))
        {
          warehouse = tokens[paramIdx+1];

          logger.log(Level.FINE, "warehouse: {0}", warehouse);
        }
        else if("loginTimeout".equalsIgnoreCase(tokens[paramIdx]))
        {
          loginTimeout = Integer.parseInt(tokens[paramIdx+1]);

          logger.log(Level.FINE, "login timeout: {0}", loginTimeout);
        }
        else if("networkTimeout".equalsIgnoreCase(tokens[paramIdx]))
        {
          networkTimeoutInMilli = Integer.parseInt(tokens[paramIdx+1]) * 1000;

          logger.log(Level.FINE, "network timeout in milli: {0}",
              networkTimeoutInMilli);
        }
        else if("queryTimeout".equalsIgnoreCase(tokens[paramIdx]))
        {
          queryTimeout = Integer.parseInt(tokens[paramIdx+1]);

          logger.log(Level.FINE, "queryTimeout: {0}", queryTimeout);
        }
        else if("useProxy".equalsIgnoreCase(tokens[paramIdx]))
        {
          useProxy = "on".equalsIgnoreCase(tokens[paramIdx+1]) ||
              "true".equalsIgnoreCase(tokens[paramIdx+1]);

          logger.log(Level.FINE, "useProxy: {0}", tokens[paramIdx+1]);
        }
        else if("injectSocketTimeout".equalsIgnoreCase(tokens[paramIdx]))
        {
          injectSocketTimeout = Integer.parseInt(tokens[paramIdx+1]);

          logger.log(Level.FINE, "injectSocketTimeout: {0}",
              injectSocketTimeout);
        }
        else if("injectClientPause".equalsIgnoreCase(tokens[paramIdx]))
        {
          injectClientPause = Integer.parseInt(tokens[paramIdx+1]);

          logger.log(Level.FINE, "injectClientPause: {0}", injectClientPause);
        }
        else if("useV1QueryAPI".equalsIgnoreCase(tokens[paramIdx]))
        {
          if ("on".equalsIgnoreCase(tokens[paramIdx+1]) ||
              "true".equalsIgnoreCase(tokens[paramIdx+1]))
            useV1QueryAPI = true;

          logger.log(Level.FINE, "useV1QueryAPI: {0}", tokens[paramIdx+1]);
        }
        else if("retryQuery".equalsIgnoreCase(tokens[paramIdx]))
        {
          if ("on".equalsIgnoreCase(tokens[paramIdx+1]) ||
              "true".equalsIgnoreCase(tokens[paramIdx+1]))
            retryQuery = true;

          logger.log(Level.FINE, "retryQuery: {0}", tokens[paramIdx+1]);
        }
        else if("tracing".equalsIgnoreCase(tokens[paramIdx]))
        {
          String tracingLevelStr = tokens[paramIdx + 1].toUpperCase();

          logger.log(Level.FINE, "tracing level specified in connection url: {0}",
                                   tracingLevelStr);

          tracingLevel = Level.parse(tracingLevelStr);
          if (tracingLevel != null)
          {
            Logger snowflakeLogger = Logger.getLogger("net.snowflake");
            snowflakeLogger.setLevel(tracingLevel);
          }
        }
        // If the name of the parameter does not match any of the built in
        // names, assume it is a session level parameter
        else
        {
          String param_name = tokens[paramIdx];
          String param_value = tokens[paramIdx + 1];

          logger.log(Level.FINE, "parameter {0} set to {1}",
                     new Object[]{param_name, param_value});
          sessionParameters.put(param_name, param_value);
        }
      }
    }

    // the properties can be overriden
    for (Object key: info.keySet())
    {
      if (key.equals("user"))
      {
        userName = info.getProperty("user");

        logger.log(Level.FINE, "user name property: {0}", userName);
      }
      else if (key.equals("password"))
      {
        password = info.getProperty("password");
      }
      else if (key.equals("account"))
      {
        accountName = info.getProperty("account");

        logger.log(Level.FINE, "account name property: {0}", accountName);
      }
      else if (key.equals("db"))
      {
        databaseName = info.getProperty("db");

        logger.log(Level.FINE, "database name property: {0}", databaseName);
      }
      else if (key.equals("database"))
      {
        databaseName = info.getProperty("database");

        logger.log(Level.FINE, "database name property: {0}", databaseName);
      }
      else if (key.equals("schema"))
      {
        schemaName = info.getProperty("schema");

        logger.log(Level.FINE, "schema name property: {0}", schemaName);
      }
      else if (key.equals("warehouse"))
      {
        warehouse = info.getProperty("warehouse");

        logger.log(Level.FINE, "warehouse property: {0}", warehouse);
      }
      else if (key.equals("role"))
      {
        role = info.getProperty("role");

        logger.log(Level.FINE, "role property: {0}", role);
      }
      else if (key.equals("authenticator"))
      {
        authenticator = info.getProperty("authenticator");

        logger.log(Level.FINE, "authenticator property: {0}", authenticator);
      }
      else if (key.equals("ssl"))
      {
        sslOn = !("off".equalsIgnoreCase(info.getProperty("ssl")) ||
          "false".equalsIgnoreCase(info.getProperty("ssl")));

        logger.log(Level.FINE, "ssl property: {0}", info.getProperty("ssl"));
      }
      else if (key.equals("passcodeInPassword"))
      {
        passcodeInPassword =
          "on".equalsIgnoreCase(info.getProperty("passcodeInPassword")) ||
              "true".equalsIgnoreCase(info.getProperty("passcodeInPassword"));
      }
      else if (key.equals("passcode"))
      {
        passcode = info.getProperty("passcode");
      }
      else if (key.equals("internal"))
      {
        internalTesting = "true".equalsIgnoreCase(info.getProperty("internal"));
      }
      else if (key.equals("loginTimeout"))
      {
        loginTimeout = Integer.parseInt(info.getProperty("loginTimeout"));
      }
      else if (key.equals("netowrkTimeout"))
      {
        networkTimeoutInMilli =
        Integer.parseInt(info.getProperty("networkTimeout")) * 1000;
      }
      else if (key.equals("queryTimeout"))
      {
        queryTimeout = Integer.parseInt(info.getProperty("queryTimeout"));
      }
      else if (key.equals("injectSocketTimeout"))
      {
        injectSocketTimeout =
          Integer.parseInt(info.getProperty("injectSocketTimeout"));
      }
      else if (key.equals("injectClientPause"))
      {
        injectClientPause =
          Integer.parseInt(info.getProperty("injectClientPause"));
      }
      else if (key.equals("useV1QueryAPI"))
      {
        String val = info.getProperty("useV1QueryAPI");
        if ("on".equalsIgnoreCase(val) ||
            "true".equalsIgnoreCase(val))
          useV1QueryAPI = true;

        logger.log(Level.FINE, "useV1QueryAPI property: {0}", val);
      }
      else if (key.equals("retryQuery"))
      {
        String val = info.getProperty("retryQuery");
        if ("on".equalsIgnoreCase(val) ||
            "true".equalsIgnoreCase(val))
          retryQuery = true;

        logger.log(Level.FINE, "retryQuery property: {0}", val);
      }
      else if (key.equals("tracing"))
      {
        tracingLevel = Level.parse(info.getProperty("tracing").toUpperCase());
        if (tracingLevel != null)
        {
          Logger snowflakeLogger = Logger.getLogger("net.snowflake");
          snowflakeLogger.setLevel(tracingLevel);
        }

        logger.log(Level.FINE, "tracingLevel property: {0}",
            info.getProperty("tracing"));
      }
      // If the key does not match any of the built in values, assume it's a
      // session level parameter
      else
      {
        String param_name = key.toString();
        Object param_value = info.get(key);
        sessionParameters.put(param_name, param_value);
        logger.log(Level.FINE, "parameter {0} set to {1}",
                   new Object[]{param_name, param_value.toString()});
      }
    }

    // initialize account name from host if necessary
    if (accountName == null && serverUrl != null &&
        serverUrl.indexOf(".") > 0 &&
        serverUrl.indexOf("://") > 0)
    {
      accountName = serverUrl.substring(serverUrl.indexOf("://")+3,
          serverUrl.indexOf("."));

      logger.log(Level.FINE, "set account name to {0}", accountName);
    }
  }

  /**
   * Execute a statement where the result isn't needed, and the statement is
   * closed before this method returns
   * @param stmtText text of the statement
   * @throws java.sql.SQLException exception thrown it the statement fails to execute
   */
  private void executeImmediate(String stmtText) throws SQLException
  {
    // execute the statement and auto-close it as well
    try (final Statement statement = this.createStatement();)
    {
      statement.execute(stmtText);
    }
  }

  public String getNewClientForUpdate()
  {
    return newClientForUpdate;
  }

  public void setNewClientForUpdate(String newClientForUpdate)
  {
    this.newClientForUpdate = newClientForUpdate;
  }


  public int getAndIncrementSequenceId()
  {
    return sequenceId.getAndIncrement();
  }

  /**
   * get session token
   *
   * @return session token
   */
  public String getSessionToken()
  {
    return sessionToken;
  }

  /**
   * get server url
   *
   * @return server url
   */
  protected String getServerUrl()
  {
    return serverUrl;
  }

  /**
   * Create a statement
   *
   * @return statement
   * @throws java.sql.SQLException if failed to create a snowflake statement
   */
  @Override
  public Statement createStatement() throws SQLException
  {
    Statement statement = new SnowflakeStatementV1(this);
    statement.setQueryTimeout(queryTimeout);
    return statement;
  }

  /**
   * Close the connection
   *
   * @throws java.sql.SQLException failed to close the connection
   */
  @Override
  public void close() throws SQLException
  {
    logger.log(Level.FINER, " public void close() throws SQLException");

    if (isClosed)
    {
      return;
    }

    try
    {
      if (sfSession != null)
      {
        sfSession.close();
        sfSession = null;
      }
      isClosed = true;
    }
    catch (SFException ex)
    {
      throw new SnowflakeSQLException(ex.getCause(),
          ex.getSqlState(), ex.getVendorCode(), ex.getParams());
    }
  }

  @Override
  public boolean isClosed() throws SQLException
  {
    logger.log(Level.FINER, " public boolean isClosed() throws SQLException");

    return isClosed;
  }

  /**
   * Return the database metadata
   *
   * @return Database metadata
   * @throws java.sql.SQLException f
   */
  @Override
  public DatabaseMetaData getMetaData() throws SQLException
  {
    logger.log(Level.FINER,
               " public DatabaseMetaData getMetaData() throws SQLException");

    return new SnowflakeDatabaseMetaData(this);
  }

  @Override
  public CallableStatement prepareCall(String sql) throws SQLException
  {
    logger.log(Level.FINER,
               " public CallableStatement prepareCall(String sql) throws "
                       + "SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public CallableStatement prepareCall(String sql, int resultSetType,
                                       int resultSetConcurrency)
          throws SQLException
  {
    logger.log(Level.FINER,
               " public CallableStatement prepareCall(String sql, int "
                       + "resultSetType,");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public CallableStatement prepareCall(String sql, int resultSetType,
                                       int resultSetConcurrency,
                                       int resultSetHoldability)
          throws SQLException
  {
    logger.log(Level.FINER,
               " public CallableStatement prepareCall(String sql, int "
                       + "resultSetType,");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public String nativeSQL(String sql) throws SQLException
  {
    logger.log(Level.FINER,
               " public String nativeSQL(String sql) throws SQLException");

    return sql;
  }

  @Override
  public void setAutoCommit(boolean isAutoCommit) throws SQLException
  {
    logger.log(Level.FINER,
               " public void setAutoCommit(boolean isAutoCommit) throws "
                       + "SQLException");

    try
    {
      this.executeImmediate("alter session set autocommit=" +
                            Boolean.toString(isAutoCommit));
    }
    catch (SnowflakeSQLException sse)
    {
      // check whether this is the autocommit api unsupported exception
      if (sse.getSQLState().equals(SqlState.FEATURE_NOT_SUPPORTED))
      {
        // autocommit api support has not yet been enabled in this session/account
        // do nothing for backward compatibility
        logger.log(Level.FINER, "Autocommit API is not supported for this " +
            "connection.");
      }
      else
      {
        // this is a different exception, rethrow it
        throw sse;
      }
    }
  }

  /**
   * Look up the GS metadata using sql command,
   * provide a list of column names
   * and return these column values in the row one of result set
   * @param querySQL,
   * @param columnNames
   * @return
   * @throws SQLException
   */
  private List<String> queryGSMetaData(String querySQL, List<String> columnNames) throws SQLException
  {
    // try with auto-closing statement resource
    try (Statement statement = this.createStatement())
    {
      statement.execute(querySQL);

      // handle the case where the result set is not what is expected
      // try with auto-closing resultset resource
      try (ResultSet rs = statement.getResultSet())
      {
        if (rs != null && rs.next())
        {
          List<String> columnValues = new ArrayList<>();
          for (String columnName : columnNames)
          {
            columnValues.add(rs.getString(columnName));
          }
          return columnValues;
        }
        else
        {
          // returned no results or an error
          throw new SQLException(
              errorResourceBundleManager.getLocalizedMessage(
                  ErrorCode.BAD_RESPONSE.getMessageCode().toString()));
        }
      }
    }
  }

  @Override
  public boolean getAutoCommit() throws SQLException
  {
    logger.log(Level.FINER,
               " public boolean getAutoCommit() throws SQLException");

    return sfSession.getAutoCommit();
  }

  @Override
  public void commit() throws SQLException
  {
    logger.log(Level.FINER, " public void commit() throws SQLException");

    // commit
    this.executeImmediate("commit");
  }

  @Override
  public void rollback() throws SQLException
  {
    logger.log(Level.FINER, " public void rollback() throws SQLException");

    // rollback
    this.executeImmediate("rollback");
  }

  @Override
  public void rollback(Savepoint savepoint) throws SQLException
  {
    logger.log(Level.FINER,
               " public void rollback(Savepoint savepoint) throws "
                       + "SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void setReadOnly(boolean readOnly) throws SQLException
  {
      logger.log(Level.FINER,
                 " public void setReadOnly(boolean readOnly) throws "
              + "SQLException");

    if (readOnly)
    {
      throw new SQLFeatureNotSupportedException();
    }
  }

  @Override
  public boolean isReadOnly() throws SQLException
  {
    logger.log(Level.FINER, " public boolean isReadOnly() throws SQLException");

    return false;
  }

  @Override
  public void setCatalog(String catalog) throws SQLException
  {
      logger.log(Level.FINER,
                 " public void setCatalog(String catalog) throws SQLException");

    // change database name
    databaseName = catalog;

    // switch db by running "use db"
    this.executeImmediate("use database \"" + databaseName + "\"");
  }

  @Override
  public String getCatalog() throws SQLException
  {
    logger.log(Level.FINER, " public String getCatalog() throws SQLException");

    String querySQL = "select current_database() as db";
    List<String> queryResults = queryGSMetaData(querySQL, Arrays.asList("DB"));
    return queryResults.get(0);
  }

  @Override
  public void setTransactionIsolation(int level) throws SQLException
  {
    logger.log(Level.FINER,
        " public void setTransactionIsolation(int level) "
            + "throws SQLException. level = {0}", level);
    if (level == Connection.TRANSACTION_NONE
        || level == Connection.TRANSACTION_READ_COMMITTED)
    {
      this.transactionIsolation = level;
    }
    else
    {
      throw new SQLFeatureNotSupportedException(
          "Transaction Isolation " + Integer.toString(level) + " not supported.",
          ErrorCode.FEATURE_UNSUPPORTED.getSqlState(),
          ErrorCode.FEATURE_UNSUPPORTED.getMessageCode());
    }
  }

  @Override
  public int getTransactionIsolation() throws SQLException
  {
      logger.log(Level.FINER,
                 " public int getTransactionIsolation() throws SQLException");

    return this.transactionIsolation;
  }

  @Override
  public SQLWarning getWarnings() throws SQLException
  {
    logger.log(Level.FINER,
               " public SQLWarning getWarnings() throws SQLException");

    if (sqlWarnings != null && !sqlWarnings.isEmpty())
      return sqlWarnings.get(0);
    else
      return null;
  }

  @Override
  public void clearWarnings() throws SQLException
  {
    logger.log(Level.FINER, " public void clearWarnings() throws SQLException");
    if (sqlWarnings != null)
      sqlWarnings.clear();
  }

  @Override
  public Statement createStatement(int resultSetType, int resultSetConcurrency)
          throws SQLException
  {
      logger.log(Level.FINER,
                 " public Statement createStatement(int resultSetType, "
              + "int resultSetConcurrency)");

      logger.log(Level.FINER, "resultSetType=" + resultSetType +
                             "; resultSetConcurrency=" + resultSetConcurrency);

    if (resultSetType != ResultSet.TYPE_FORWARD_ONLY
        || resultSetConcurrency != ResultSet.CONCUR_READ_ONLY)
    {
      throw new SQLFeatureNotSupportedException();
    }
    return createStatement();
  }

  @Override
  public Statement createStatement(int resultSetType, int resultSetConcurrency,
                                   int resultSetHoldability)
          throws SQLException
  {
      logger.log(Level.FINER,
                 " public Statement createStatement(int resultSetType, "
              + "int resultSetConcurrency,");

    if (resultSetHoldability != ResultSet.CLOSE_CURSORS_AT_COMMIT)
    {
      throw new SQLFeatureNotSupportedException();
    }
    return createStatement(resultSetType, resultSetConcurrency);
  }

  @Override
  public PreparedStatement prepareStatement(String sql) throws SQLException
  {
    logger.log(Level.FINER,
               " public PreparedStatement prepareStatement(String sql) "
                       + "throws SQLException");

    return new SnowflakePreparedStatementV1(this, sql);
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys)
          throws SQLException
  {
      logger.log(Level.FINER,
                 " public PreparedStatement prepareStatement(String sql, "
              + "int autoGeneratedKeys)");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int[] columnIndexes)
          throws SQLException
  {
      logger.log(Level.FINER,
                 " public PreparedStatement prepareStatement(String sql, "
              + "int[] columnIndexes)");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public PreparedStatement prepareStatement(String sql, String[] columnNames)
          throws SQLException
  {
      logger.log(Level.FINER,
                 " public PreparedStatement prepareStatement(String sql, "
              + "String[] columnNames)");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int resultSetType,
                                            int resultSetConcurrency)
          throws SQLException
  {
      logger.log(Level.FINER,
                 " public PreparedStatement prepareStatement(String sql, "
              + "int resultSetType,");

    if (resultSetType != ResultSet.TYPE_FORWARD_ONLY
        || resultSetConcurrency != ResultSet.CONCUR_READ_ONLY)
    {
      logger.log(Level.SEVERE,
                 "result set type ({0}) or result set concurrency ({1}) "
                         + "not supported",
                 new Object[]{resultSetType, resultSetConcurrency});

      throw new SQLFeatureNotSupportedException();
    }
    return prepareStatement(sql);
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int resultSetType,
                                            int resultSetConcurrency,
                                            int resultSetHoldability)
          throws SQLException
  {
      logger.log(Level.FINER,
                 " public PreparedStatement prepareStatement(String sql, "
              + "int resultSetType,");

    if (resultSetHoldability != ResultSet.CLOSE_CURSORS_AT_COMMIT)
    {
      throw new SQLFeatureNotSupportedException();
    }
    return prepareStatement(sql, resultSetType, resultSetConcurrency);
  }

  @Override
  public Map<String, Class<?>> getTypeMap() throws SQLException
  {
      logger.log(Level.FINER,
                 " public Map<String, Class<?>> getTypeMap() throws "
              + "SQLException");

    return Collections.emptyMap();
  }

  @Override
  public void setTypeMap(Map<String, Class<?>> map) throws SQLException
  {
      logger.log(Level.FINER,
                 " public void setTypeMap(Map<String, Class<?>> map) "
              + "throws SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void setHoldability(int holdability) throws SQLException
  {
      logger.log(Level.FINER,
                 " public void setHoldability(int holdability) throws "
              + "SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getHoldability() throws SQLException
  {
      logger.log(Level.FINER, " public int getHoldability() throws SQLException");

    return ResultSet.CLOSE_CURSORS_AT_COMMIT;
  }

  @Override
  public Savepoint setSavepoint() throws SQLException
  {
      logger.log(Level.FINER,
                 " public Savepoint setSavepoint() throws SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public Savepoint setSavepoint(String name) throws SQLException
  {
      logger.log(Level.FINER,
                 " public Savepoint setSavepoint(String name) throws "
              + "SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void releaseSavepoint(Savepoint savepoint) throws SQLException
  {
      logger.log(Level.FINER,
                 " public void releaseSavepoint(Savepoint savepoint) throws "
              + "SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public Blob createBlob() throws SQLException
  {
      logger.log(Level.FINER, " public Blob createBlob() throws SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public Clob createClob() throws SQLException
  {
      logger.log(Level.FINER, " public Clob createClob() throws SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public NClob createNClob() throws SQLException
  {
      logger.log(Level.FINER, " public NClob createNClob() throws SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public SQLXML createSQLXML() throws SQLException
  {
      logger.log(Level.FINER,
                   " public SQLXML createSQLXML() throws SQLException");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean isValid(int timeout) throws SQLException
  {
      logger.log(Level.FINER,
                 " public boolean isValid(int timeout) throws SQLException");

    // TODO: run query here or ping
    return !isClosed;
  }

  @Override
  public void setClientInfo(Properties properties)
          throws SQLClientInfoException
  {
      logger.log(Level.FINER,
                   " public void setClientInfo(Properties properties)");

    if (this.clientInfo == null)
      this.clientInfo = new Properties();

    // make a copy, don't point to the properties directly since we don't
    // own it.
    this.clientInfo.clear();
    this.clientInfo.putAll(properties);

    if (sfSession != null)
      sfSession.setClientInfo(properties);
  }

  @Override
  public void setClientInfo(String name, String value)
          throws SQLClientInfoException
  {
      logger.fine(" public void setClientInfo(String name, String value)");

    if (this.clientInfo == null)
      this.clientInfo = new Properties();

    this.clientInfo.setProperty(name, value);

    if (sfSession != null)
      sfSession.setClientInfo(name, value);
  }

  @Override
  public Properties getClientInfo() throws SQLException
  {
    logger.log(Level.FINER,
        " public Properties getClientInfo() throws SQLException");

    if (sfSession != null)
      return sfSession.getClientInfo();

    if (this.clientInfo != null)
    {
      // defensive copy to avoid client from changing the properties
      // directly w/o going through the API
      Properties copy = new Properties();
      copy.putAll(this.clientInfo);

      return copy;
    }
    else
      return null;
  }

  @Override
  public String getClientInfo(String name)
  {
    if (sfSession != null)
      return sfSession.getClientInfo(name);

    logger.log(Level.FINER, " public String getClientInfo(String name)");

    if (this.clientInfo != null)
      return this.clientInfo.getProperty(name);
    else
      return null;
  }

  @Override
  public Array createArrayOf(String typeName, Object[] elements)
          throws SQLException
  {
      logger.log(Level.FINER,
                 " public Array createArrayOf(String typeName, Object[] "
              + "elements)");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public Struct createStruct(String typeName, Object[] attributes)
          throws SQLException
  {
      logger.log(Level.FINER,
                 " public Struct createStruct(String typeName, Object[] "
              + "attributes)");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void setSchema(String schema) throws SQLException
  {
      logger.log(Level.FINER,
                 " public void setSchema(String schema) throws SQLException");

    schemaName = schema;

    // switch schema by running "use db.schema"
    if (databaseName == null)
    {
      this.executeImmediate("use schema \"" + schemaName + "\"");
    }
    else
    {
      this.executeImmediate("use schema \"" + databaseName + "\".\"" + schemaName + "\"");
    }
  }

  @Override
  public String getSchema() throws SQLException
  {
    logger.log(Level.FINER, " public String getSchema() throws SQLException");

    String querySQL = "select current_schema() as schema";
    List<String> queryResults = queryGSMetaData(querySQL, Arrays.asList("SCHEMA"));
    return queryResults.get(0);
  }

  @Override
  public void abort(Executor executor) throws SQLException
  {
      logger.log(Level.FINER,
                 " public void abort(Executor executor) throws SQLException");

    close();
  }

  @Override
  public void setNetworkTimeout(Executor executor, int milliseconds)
          throws SQLException
  {
      logger.log(Level.FINER,
                 " public void setNetworkTimeout(Executor executor, int "
              + "milliseconds)");

    networkTimeoutInMilli = milliseconds;
  }

  @Override
  public int getNetworkTimeout() throws SQLException
  {
      logger.log(Level.FINER,
                 " public int getNetworkTimeout() throws SQLException");

    return networkTimeoutInMilli;
  }

  @Override
  public boolean isWrapperFor(Class<?> iface) throws SQLException
  {
      logger.log(Level.FINER,
                 " public boolean isWrapperFor(Class<?> iface) throws "
              + "SQLException");

    return iface.isInstance(this);
  }

  @SuppressWarnings("unchecked")
  @Override
  public <T> T unwrap(Class<T> iface) throws SQLException
  {
      logger.log(Level.FINER,
                 " public <T> T unwrap(Class<T> iface) throws SQLException");


    if (!iface.isInstance(this))
    {
      throw new RuntimeException(this.getClass().getName()
                                 + " not unwrappable from " + iface.getName());
    }
    return (T) this;
  }

  void setSFSessionProperty(String propertyName, boolean b)
  {
    this.sessionProperties.put(propertyName, b);
  }

  public Object getSFSessionProperty(String propertyName)
  {
    return this.sessionProperties.get(propertyName);
  }

  public static int getMajorVersion()
  {
    return SnowflakeDriver.majorVersion;
  }

  public static int getMinorVersion()
  {
    return SnowflakeDriver.minorVersion;
  }

  public int getDatabaseMajorVersion()
  {
    return databaseMajorVersion;
  }

  public int getDatabaseMinorVersion()
  {
    return databaseMinorVersion;
  }

  public String getDatabaseVersion()
  {
    return databaseVersion;
  }

  public String getUserName()
  {
    return userName;
  }

  public void setUserName(String userName)
  {
    this.userName = userName;
  }

  public String getAccountName()
  {
    return accountName;
  }

  public void setAccountName(String accountName)
  {
    this.accountName = accountName;
  }

  public String getDatabaseName()
  {
    return databaseName;
  }

  public void setDatabaseName(String databaseName)
  {
    this.databaseName = databaseName;
  }

  public String getSchemaName()
  {
    return schemaName;
  }

  public void setSchemaName(String schemaName)
  {
    this.schemaName = schemaName;
  }

  public String getURL()
  {
    return serverUrl;
  }

  public int getQueryTimeout()
  {
    return queryTimeout;
  }

  /**
   * Method to put data from a stream at a stage location. The data will be
   * uploaded as one file. No splitting is done in this method.
   *
   * Stream size must match the total size of data in the input stream unless
   * compressData parameter is set to true.
   *
   * caller is responsible for passing the correct size for the data in the
   * stream and releasing the inputStream after the method is called.
   *
   * @param stageName
   *   stage name: e.g. ~ or table name or stage name
   * @param destPrefix
   *   path prefix under which the data should be uploaded on the stage
   * @param inputStream
   *   input stream from which the data will be uploaded
   * @param destFileName
   *   destination file name to use
   * @param streamSize
   *   data size in the stream
   * @throws java.sql.SQLException failed to put data from a stream at stage
   */
  public void uploadStream(String stageName,
                           String destPrefix,
                           InputStream inputStream,
                           String destFileName,
                           long streamSize)
      throws SQLException
  {
    uploadStreamInternal(stageName, destPrefix, inputStream,
        destFileName, streamSize, false);
  }

  /**
   * Method to compress data from a stream and upload it at a stage location.
   * The data will be uploaded as one file. No splitting is done in this method.
   *
   * caller is responsible for releasing the inputStream after the method is
   * called.
   *
   * @param stageName
   *   stage name: e.g. ~ or table name or stage name
   * @param destPrefix
   *   path prefix under which the data should be uploaded on the stage
   * @param inputStream
   *   input stream from which the data will be uploaded
   * @param destFileName
   *   destination file name to use
   * @throws java.sql.SQLException failed to compress and put data from a stream at stage
   */
  public void compressAndUploadStream(String stageName,
                                     String destPrefix,
                                     InputStream inputStream,
                                     String destFileName)
      throws SQLException
  {
    uploadStreamInternal(stageName, destPrefix, inputStream,
        destFileName, 0, true);
  }

  /**
   * Method to put data from a stream at a stage location. The data will be
   * uploaded as one file. No splitting is done in this method.
   *
   * Stream size must match the total size of data in the input stream unless
   * compressData parameter is set to true.
   *
   * caller is responsible for passing the correct size for the data in the
   * stream and releasing the inputStream after the method is called.
   *
   * @param stageName
   *   stage name: e.g. ~ or table name or stage name
   * @param destPrefix
   *   path prefix under which the data should be uploaded on the stage
   * @param inputStream
   *   input stream from which the data will be uploaded
   * @param destFileName
   *   destination file name to use
   * @param streamSize
   *   data size in the stream
   * @param compressData
   *   whether compression is requested fore uploading data
   * @throws java.sql.SQLException
   */
  private void uploadStreamInternal(String stageName,
                           String destPrefix,
                           InputStream inputStream,
                           String destFileName,
                           long streamSize,
                           boolean compressData)
      throws SQLException
  {
    logger.log(Level.FINER, "upload data from stream: stageName={0}" +
        ", destPrefix={1}, destFileName={2}",
        new Object[]{stageName, destPrefix, destFileName});

    if (stageName == null)
      throw new SnowflakeSQLException(SqlState.INTERNAL_ERROR,
          ErrorCode.INTERNAL_ERROR.getMessageCode(),
          "stage name is null");

    if (destFileName == null)
      throw new SnowflakeSQLException(SqlState.INTERNAL_ERROR,
          ErrorCode.INTERNAL_ERROR.getMessageCode(),
          "stage name is null");

    SnowflakeStatementV1 stmt = new SnowflakeStatementV1(this);

    StringBuilder putCommand = new StringBuilder();

    // use a placeholder for source file
    putCommand.append("put file:///tmp/placeholder ");

    // add stage name
    if (!stageName.startsWith("@"))
      putCommand.append("@");
    putCommand.append(stageName);

    // add dest prefix
    if (destPrefix != null)
    {
      if (!destPrefix.startsWith("/"))
        putCommand.append("/");
      putCommand.append(destPrefix);
    }

    SnowflakeFileTransferAgent transferAgent = null;
    transferAgent = new SnowflakeFileTransferAgent(putCommand.toString(),
        sfSession, stmt.getSfStatement());

    transferAgent.setSourceStream(inputStream);
    transferAgent.setDestFileNameForStreamSource(destFileName);
    transferAgent.setSourceStreamSize(streamSize);
    transferAgent.setCompressSourceFromStream(compressData);
    transferAgent.setOverwrite(true);
    transferAgent.execute();

    stmt.close();
  }

  public void setInjectedDelay(int delay)
  {
    if (sfSession != null)
      sfSession.setInjectedDelay(delay);

    this._injectedDelay.set(delay);
  }

  void injectedDelay()
  {

    int d = _injectedDelay.get();

    if (d != 0)
    {
      _injectedDelay.set(0);
      try
      {
        logger.log(Level.FINEST, "delayed for {0}", d);

        Thread.sleep(d);
      }
      catch (InterruptedException ex)
      {
      }
    }
  }

  public int getInjectSocketTimeout()
  {
    return injectSocketTimeout;
  }

  public void setInjectSocketTimeout(int injectSocketTimeout)
  {
    if (sfSession != null)
      sfSession.setInjectSocketTimeout(injectSocketTimeout);

    this.injectSocketTimeout = injectSocketTimeout;
  }

  public boolean isUseV1QueryAPI()
  {
    return useV1QueryAPI;
  }

  public void setUseV1QueryAPI(boolean useV1QueryAPI)
  {
    this.useV1QueryAPI = useV1QueryAPI;
  }

  public boolean isRetryQuery()
  {
    return retryQuery;
  }

  public void setRetryQuery(boolean retryQuery)
  {
    this.retryQuery = retryQuery;
  }

  public int getInjectClientPause()
  {
    return injectClientPause;
  }

  public void setInjectClientPause(int injectClientPause)
  {
    if (sfSession != null)
      sfSession.setInjectClientPause(injectClientPause);

    this.injectClientPause = injectClientPause;
  }

  public void setInjectFileUploadFailure(String fileToFail)
  {
    if (sfSession != null)
      sfSession.setInjectFileUploadFailure(fileToFail);

    this.injectFileUploadFailure = fileToFail;
  }


  public String getInjectFileUploadFailure()
  {
    return this.injectFileUploadFailure;
  }

  public SFSession getSfSession()
  {
    return sfSession;
  }
}
