/*
 * Copyright (c) 2012-2016 Snowflake Computing Inc. All right reserved.
 */

package net.snowflake.client.jdbc;

import net.snowflake.client.core.SFLogger;
import net.snowflake.client.core.EventHandler;
import net.snowflake.client.core.EventUtil;
import com.snowflake.gscommon.core.LoginInfoDTO;
import com.snowflake.gscommon.core.SqlState;

import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.DriverPropertyInfo;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.Properties;
import java.util.jar.Attributes;
import java.util.jar.JarInputStream;
import java.util.jar.Manifest;
import java.util.logging.FileHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;
import java.util.regex.Pattern;

/**
 * JDBC Driver implementation of Snowflake for production.
 * To use this driver, specify the following URL:
 * jdbc:snowflake://host:port
 *
 * @author jhuang
 */
public class SnowflakeDriver implements Driver
{
  protected static Handler fileHandler = null;

  static final
  Logger logger = Logger.getLogger(SnowflakeDriver.class.getName());

  private static final String JDBC_PROTOCOL = "jdbc:snowflake://";

  // pattern for jdbc:snowflake://[host[:port]]/?q1=v1&q2=v2...
  private static final String JDBC_PROTOCOL_REGEX =
  "jdbc:snowflake://([a-zA-Z_\\-0-9\\.]+(:\\d+)?)?"
  + "(/(\\?\\w+=\\w+)?(\\&\\w+=\\w+)*)?";

  public static SnowflakeDriver INSTANCE = null;

  private static final
  DriverPropertyInfo[] EMPTY_INFO = new DriverPropertyInfo[0];

  public static String implementVersion = null;
  public static Long svnRevision = null;

  public static int majorVersion = 0;
  public static int minorVersion = 0;
  public static long changeVersion = 0;

  protected static boolean disableIncidents = false;

  static
  {
    try
    {
      DriverManager.registerDriver(INSTANCE = new SnowflakeDriver());
    }
    catch (SQLException ex)
    {
      throw new IllegalStateException("Unable to register "
              + SnowflakeDriver.class.getName(), ex);
    }

    try
    {
      String defaultLogSizeVal = System.getProperty("snowflake.jdbc.log.size");
      String defaultLogCountVal = System.getProperty("snowflake.jdbc.log.count");

      // default log size to 1 GB
      int logSize = 1000000000;

      // default number of log files to rotate to 2
      int logCount = 2;

      if (defaultLogSizeVal != null)
      {
        try
        {
          logSize = Integer.parseInt(defaultLogSizeVal);
        }
        catch(Exception ex)
        {
          ;
        }
      }

      if (defaultLogCountVal != null)
      {
        try
        {
          logCount = Integer.parseInt(defaultLogCountVal);
        }
        catch(Exception ex)
        {
          ;
        }
      }

      fileHandler = new FileHandler("%t/snowflake_jdbc%u.log",
          logSize, logCount, true);

      EventUtil.initEventHandlerInstance(1000, 10000);
      EventHandler eventHandler = EventUtil.getEventHandlerInstance();

      String defaultLevelVal = System.getProperty("snowflake.jdbc.log.level");

      Level defaultLevel = Level.WARNING;

      if (defaultLevelVal != null)
      {
        defaultLevel = Level.parse(defaultLevelVal.toUpperCase());

        if (defaultLevel == null)
          defaultLevel = Level.WARNING;
      }

      fileHandler.setLevel(Level.ALL);
      fileHandler.setFormatter(new SFLogger());

      eventHandler.setLevel(Level.ALL);
      eventHandler.setFormatter(new SimpleFormatter());

      // set default level and add handler for snowflake logger
      Logger snowflakeLogger = Logger.getLogger("net.snowflake");
      snowflakeLogger.setLevel(defaultLevel);
      snowflakeLogger.addHandler(SnowflakeDriver.fileHandler);
      snowflakeLogger.addHandler(eventHandler);
             
      logger.log(Level.FINE, "registered driver");
    }
    catch (Exception ex)
    {
      System.err.println("Unable to set up logging");
      ex.printStackTrace();
    }

    /*
     * Get the manifest properties here.
     */

    initializeClientVersionFromManifest();
  }

  static private void initializeClientVersionFromManifest()
  {
    /*
     * Get JDBC version numbers from version.properties in GSCommon
     */
    try
    {
      implementVersion = LoginInfoDTO.getLatestJDBCAppVersion();

      logger.log(Level.FINE, "implement version: {0}", implementVersion);

      // parse implementation version major.minor.change
      if (implementVersion != null)
      {
        String[] versionBreakdown = implementVersion.split("\\.");

        if (versionBreakdown != null && versionBreakdown.length == 3)
        {
          majorVersion = Integer.parseInt(versionBreakdown[0]);
          minorVersion = Integer.parseInt(versionBreakdown[1]);
          changeVersion = Long.parseLong(versionBreakdown[2]);
        }
        else
          throw new SnowflakeSQLException(SqlState.INTERNAL_ERROR,
              ErrorCode.INTERNAL_ERROR.getMessageCode(),
              "Invalid implementation version: " + implementVersion);
      }
      else
        throw new SnowflakeSQLException(SqlState.INTERNAL_ERROR,
            ErrorCode.INTERNAL_ERROR.getMessageCode(),
            "Null implementation version");

      logger.log(Level.FINE,
          "implementation_version = " + implementVersion);
      logger.log(Level.FINE, "major version = " + majorVersion);
      logger.log(Level.FINE, "minor version = " + minorVersion);
      logger.log(Level.FINE, "change version = " + changeVersion);
    }
    catch (Exception ex)
    {
      logger.log(Level.SEVERE, "Exception encountered when retrieving client "
          + "version attributes: "
          + ex.getMessage());
    }

    /*
     * Get the svn revision here.
     */
    try
    {
      if (SnowflakeConnectionV1.class.getProtectionDomain() == null ||
          SnowflakeConnectionV1.class.getProtectionDomain().getCodeSource()
          == null ||
          SnowflakeConnectionV1.class.getProtectionDomain().getCodeSource().
                  getLocation() == null)
      {
        logger.log(Level.FINE, "Couldn't get code source location");
        return;
      }

      URI jarURI =
              SnowflakeConnectionV1.class.getProtectionDomain().getCodeSource().
                      getLocation().toURI();

      logger.log(Level.FINE, "jar uri: " + jarURI.getPath() +
                            ", to_url: " + jarURI.toURL());

      // if code source is not from a jar, there will be no manifest, so skip
      // the svn revision setup
      if (jarURI == null || jarURI.getPath() == null ||
          !jarURI.getPath().endsWith(".jar"))
      {
        logger.log(Level.FINE, "couldn't determine code source");
        return;
      }

      URL jarURL = jarURI.toURL();

      if (jarURL == null)
      {
        logger.log(Level.FINE, "null jar URL");
        return;
      }

      InputStream is = jarURL.openStream();

      if (is == null)
      {
        logger.log(Level.FINE, "Can not open snowflake_jdbc.jar: " +
                               jarURI.getPath());
        return;
      }

      JarInputStream jarStream = new JarInputStream(is);
      Manifest mf = jarStream.getManifest();

      if (mf == null)
      {
        logger.log(Level.FINE, "Manifest not found from snowflake_jdbc.jar");
        return;
      }

      Attributes mainAttribs = mf.getMainAttributes();

      if (mainAttribs == null)
      {
        logger.log(Level.FINE,
                  "mainAttribs null from the manifest in snowflake_jdbc.jar");
        return;
      }

      if (mainAttribs.getValue("SVN-Revision") == null)
      {
        logger.log(Level.FINE, "SVN-Revision is null");
        return;
      }

      svnRevision = Long.valueOf(mainAttribs.getValue("SVN-Revision"));

      logger.log(Level.FINE, "svn_revision = " + svnRevision);
    }
    catch (Throwable ex)
    {
      logger.log(Level.FINE, "Exception encountered when retrieving client "
                               + "svn revision attribute from manifest: "
                               + ex.getMessage());
    }
  }

  /**
   * Checks whether a given url is in a valid format.
   *
   * The current uri format is: jdbc:snowflake://[host[:port]]
   *
   * jdbc:snowflake:// - run in embedded mode jdbc:snowflake://localhost -
   * connect to localhost default port (8080)
   *
   * jdbc:snowflake://localhost:8080- connect to localhost port 8080
   *
   * @param url
   *   url of the database including host and port
   * @return
   *   true if the url is valid
   * @throws SQLException if failed to accept url
   */
  @Override
  public boolean acceptsURL(String url) throws SQLException
  {
    if (url == null)
      return false;

    return url.indexOf("/?") > 0?
           Pattern.matches(JDBC_PROTOCOL_REGEX,
                           url.substring(0, url.indexOf("/?"))):
           Pattern.matches(JDBC_PROTOCOL_REGEX, url);
  }

  /**
   * Connect method
   *
   * @param url
   *   jdbc url
   * @param info
   *   addition info for passing database/schema names
   * @return
   *   connection
   * @throws SQLException if failed to create a snowflake connection
   */
  @Override
  public Connection connect(String url, Properties info) throws SQLException
  {
    if (acceptsURL(url))
    {
      return new SnowflakeConnectionV1(url, info);
    }
    else
      return null;
  }

  @Override
  public int getMajorVersion()
  {
    return majorVersion;
  }

  @Override
  public int getMinorVersion()
  {
    return minorVersion;
  }

  @Override
  public DriverPropertyInfo[] getPropertyInfo(String url, Properties info)
          throws SQLException
  {
    return EMPTY_INFO;
  }

  @Override
  public boolean jdbcCompliant()
  {
    return false;
  }

  //@Override
  public Logger getParentLogger() throws SQLFeatureNotSupportedException
  {
    return null;
  }

  public static boolean isDisableIncidents()
  {
    return disableIncidents;
  }

  public static void setDisableIncidents(
      boolean throttleIncidents)
  {
    if (throttleIncidents)
      logger.finest("setting throttle incidents");

    SnowflakeDriver.disableIncidents =
        throttleIncidents;
  }  
}
