/*
 * Copyright (c) 2012-2016 Snowflake Computing Inc. All right reserved.
 */

package net.snowflake.client.jdbc;

import com.snowflake.gscommon.core.SqlState;
import java.math.BigDecimal;
import java.sql.Date;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.List;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Snowflake ResultSetMetaData
 *
 * @author jhuang
 */
public class SnowflakeResultSetMetaData implements ResultSetMetaData
{

  static final
  Logger logger = Logger.getLogger(SnowflakeResultSetMetaData.class.getName());

  private int columnCount = 0;

  private List<String> columnNames;

  private List<String> columnTypeNames;

  private List<Integer> columnTypes;

  private List<Integer> precisions;

  private List<Integer> scales;

  private String queryId;

  private Map<String, Integer> columnNamePositionMap =
  new HashMap<>();

  public SnowflakeResultSetMetaData() {}

  public SnowflakeResultSetMetaData(int columnCount,
                                    List<String> columnNames,
                                    List<String> columnTypeNames,
                                    List<Integer> columnTypes)
          throws SnowflakeSQLException
  {
    this.columnCount = columnCount;
    this.columnNames = columnNames;
    this.columnTypeNames = columnTypeNames;
    this.columnTypes = columnTypes;
  }

  public SnowflakeResultSetMetaData(List<SnowflakeColumnMetadata>
                                    columnMetadata)
          throws SnowflakeSQLException
  {
    this(columnMetadata, "none");
  }

  public SnowflakeResultSetMetaData(List<SnowflakeColumnMetadata> columnMetadata,
                                    String queryId)
          throws SnowflakeSQLException
  {
    this.columnCount = columnMetadata.size();
    this.queryId = queryId;

    this.columnNames = new ArrayList<String>(this.columnCount);
    this.columnTypeNames = new ArrayList<String>(this.columnCount);
    this.columnTypes = new ArrayList<Integer>(this.columnCount);
    this.precisions = new ArrayList<Integer>(this.columnCount);
    this.scales = new ArrayList<Integer>(this.columnCount);

    for(int colIdx = 0; colIdx < columnCount; colIdx ++)
    {
      columnNames.add(columnMetadata.get(colIdx).getName());
      columnTypeNames.add(columnMetadata.get(colIdx).getTypeName());
      precisions.add(columnMetadata.get(colIdx).getPrecision());
      columnTypes.add(columnMetadata.get(colIdx).getType());
      scales.add(columnMetadata.get(colIdx).getScale());
    }
  }

  /**
   * @return query id
   */
  public String getQueryId()
  {
    return queryId;
  }

  /**
   * @return list of column names
   */
  public List<String> getColumnNames()
  {
    return columnNames;
  }

  /**
   * @param columnName column name
   * @return index of the column
   */
  public int getColumnIndex(String columnName)
  {
    if (columnNamePositionMap.get(columnName) != null)
    {
      return columnNamePositionMap.get(columnName);
    }
    else
    {
      int columnIndex = columnNames.indexOf(columnName);
      columnNamePositionMap.put(columnName, columnIndex);
      return columnIndex;
    }
  }

  /**
   * @return column count
   * @throws SQLException if failed to get column count
   */
  @Override
  public int getColumnCount() throws SQLException
  {
    logger.log(Level.FINER, 
	       "public int getColumnCount(), columnCount= {0}",
	       columnCount);

    return columnCount;
  }

  @Override
  public boolean isAutoIncrement(int column) throws SQLException
  {
      logger.log(Level.FINER, "public boolean isAutoIncrement(int column)");

    return false;
  }

  @Override
  public boolean isCaseSensitive(int column) throws SQLException
  {
      logger.log(Level.FINER, "public boolean isCaseSensitive(int column)");

    return false;
  }

  @Override
  public boolean isSearchable(int column) throws SQLException
  {
      logger.log(Level.FINER, "public boolean isSearchable(int column)");

    return true;
  }

  @Override
  public boolean isCurrency(int column) throws SQLException
  {
      logger.log(Level.FINER, "public boolean isCurrency(int column)");

    return false;
  }

  @Override
  public int isNullable(int column) throws SQLException
  {
      logger.log(Level.FINER, "public int isNullable(int column)");

    return columnNullableUnknown;
  }

  @Override
  public boolean isSigned(int column) throws SQLException
  {
    logger.log(Level.FINER, "public boolean isSigned(int column)");

    if (columnTypes.get(column-1) == Types.INTEGER ||
        columnTypes.get(column-1) == Types.DECIMAL ||
        columnTypes.get(column-1) == Types.DOUBLE)
      return true;
    else
      return false;
  }

  @Override
  public int getColumnDisplaySize(int column) throws SQLException
  {
      logger.log(Level.FINER, "public int getColumnDisplaySize(int column)");

    return 25;
  }

  @Override
  public String getColumnLabel(int column) throws SQLException
  {
    logger.log(Level.FINER, "public String getColumnLabel(int column)");

    if(columnNames != null)
      return columnNames.get(column-1);
    else
      return "C" + Integer.toString(column-1);
  }

  @Override
  public String getColumnName(int column) throws SQLException
  {
      logger.log(Level.FINER, "public String getColumnName(int column)");

    if(columnNames != null)
      return columnNames.get(column-1);
    else
      return "C" + Integer.toString(column-1);
  }

  @Override
  public String getSchemaName(int column) throws SQLException
  {
      logger.log(Level.FINER, "public String getSchemaName(int column)");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getPrecision(int column) throws SQLException
  {
      logger.log(Level.FINER, "public int getPrecision(int column)");

    if (precisions != null && precisions.size() >= column)
    {
      return precisions.get(column-1);
    }
    else
    {
      // TODO: fix this later to use different defaults for number or timestamp
      return 9;
    }
  }

  @Override
  public int getScale(int column) throws SQLException
  {
      logger.log(Level.FINER, "public int getScale(int column)");

    if (scales != null && scales.size() >= column)
    {
      return scales.get(column-1);
    }
    else
    {
      // TODO: fix this later to use different defaults for number or timestamp
      return 9;
    }
  }

  @Override
  public String getTableName(int column) throws SQLException
  {
      logger.log(Level.FINER, "public String getTableName(int column)");

    return "T";
  }

  @Override
  public String getCatalogName(int column) throws SQLException
  {
      logger.log(Level.FINER, "public String getCatalogName(int column)");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getColumnType(int column) throws SQLException
  {
      logger.log(Level.FINER, "public int getColumnType(int column)");

    int internalColumnType = getInternalColumnType(column);

    int externalColumnType = internalColumnType;

    if (internalColumnType == SnowflakeUtil.EXTRA_TYPES_TIMESTAMP_LTZ ||
        internalColumnType == SnowflakeUtil.EXTRA_TYPES_TIMESTAMP_TZ)
      externalColumnType = Types.TIMESTAMP;

    logger.log(Level.FINER, 
	       "column type = {0}", 
	       externalColumnType);

    return externalColumnType;
  }

  public int getInternalColumnType(int column) throws SQLException
  {
      logger.log(Level.FINER, "public int getInternalColumnType(int column)");

    int columnIdx = column-1;
    if (column > columnTypes.size())
    {
      throw new SQLException("Invalid column index: " + column);
    }

    if(columnTypes.get(columnIdx) == null)
    {
      throw new SnowflakeSQLException(SqlState.INTERNAL_ERROR,
              ErrorCode.INTERNAL_ERROR.getMessageCode(),
              "Missing column type for column " + column);
    }

    logger.log(Level.FINER, 
	       "column type = {0}",
	       columnTypes.get(column-1));

    return columnTypes.get(columnIdx);
  }

  @Override
  public String getColumnTypeName(int column) throws SQLException
  {
      logger.log(Level.FINER, "public String getColumnTypeName(int column)");

    if (column > columnTypeNames.size())
    {
      throw new SQLException("Invalid column index: " + column);
    }

    if(columnTypeNames == null || columnTypeNames.get(column-1) == null)
    {
      throw new SnowflakeSQLException(SqlState.INTERNAL_ERROR,
              ErrorCode.INTERNAL_ERROR.getMessageCode(),
              "Missing column type for column " + column);
    }

    return columnTypeNames.get(column-1);
  }

  @Override
  public boolean isReadOnly(int column) throws SQLException
  {
      logger.log(Level.FINER, "public boolean isReadOnly(int column)");

    return true;
  }

  @Override
  public boolean isWritable(int column) throws SQLException
  {
      logger.log(Level.FINER, "public boolean isWritable(int column)");

    return false;
  }

  @Override
  public boolean isDefinitelyWritable(int column) throws SQLException
  {
      logger.log(Level.FINER, "public boolean isDefinitelyWritable(int column)");

    return false;
  }

  @Override
  public String getColumnClassName(int column) throws SQLException
  {
    logger.log(Level.FINER, "public String getColumnClassName(int column)");

    int type = this.getColumnType(column);

    switch(type)
    {
      case Types.VARCHAR:
      case Types.CHAR:
      case Types.BINARY:
        return String.class.getName();

      case Types.INTEGER:
        return Integer.class.getName();

      case Types.DECIMAL:
        return BigDecimal.class.getName();

      case Types.DOUBLE:
        return Double.class.getName();

      case Types.TIMESTAMP:
        return Timestamp.class.getName();

      case Types.DATE:
        return Date.class.getName();

      case Types.TIME:
        return Time.class.getName();

      case Types.BOOLEAN:
        return Boolean.class.getName();

      default:
        throw new SQLFeatureNotSupportedException();
    }
  }

  @Override
  public <T> T unwrap(Class<T> iface) throws SQLException
  {
      logger.log(Level.FINER, "public <T> T unwrap(Class<T> iface)");

    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean isWrapperFor(Class<?> iface) throws SQLException
  {
      logger.log(Level.FINER, "public boolean isWrapperFor(Class<?> iface)");

    throw new SQLFeatureNotSupportedException();
  }
}
