/*
 * Copyright (c) 2012-2016 Snowflake Computing Inc. All right reserved.
 */

package net.snowflake.client.jdbc;

import com.snowflake.gscommon.core.ResourceBundleManager;
import java.sql.SQLException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author jhuang
 */
public class SnowflakeSQLException extends SQLException
{
  static final Logger logger =
                       Logger.getLogger(SnowflakeSQLException.class.getName());

  static final ResourceBundleManager errorResourceBundleManager =
  ResourceBundleManager.getSingleton(ErrorCode.errorMessageResource);

  private String queryId = "unknown";

  /**
   * This constructor should only be used for error from
   * Global service. Since Global service has already built the error message,
   * we use it as is. For any errors local to JDBC driver, we should use one
   * of the constructors below to build the error message.
   * @param queryId query id
   * @param reason reason for which exception is created
   * @param sqlState sql state
   * @param vendorCode vendor code
   */
  public SnowflakeSQLException(String queryId,
                               String reason,
                               String sqlState,
                               int vendorCode)
  {
    super(reason, sqlState, vendorCode);

    this.queryId = queryId;

    // log user error from GS at fine level
    logger.log(Level.FINER, "Snowflake exception: " + reason +
                             ", sqlState:" + sqlState +
                             ", vendorCode:" + vendorCode +
                             ", queryId: " + queryId);

  }

  public SnowflakeSQLException(String sqlState, int vendorCode)
  {
    super(errorResourceBundleManager.getLocalizedMessage(
            String.valueOf(vendorCode)), sqlState, vendorCode);

    logger.log(Level.FINER, "Snowflake exception: " +
                              errorResourceBundleManager.getLocalizedMessage(
                              String.valueOf(vendorCode))
                              + ", sqlState:" + sqlState +
                              ", vendorCode:" + vendorCode);
  }

  public SnowflakeSQLException(String sqlState, int vendorCode, Object... params)
  {
    super(errorResourceBundleManager.getLocalizedMessage(
            String.valueOf(vendorCode), params), sqlState, vendorCode);

    logger.log(Level.FINER, "Snowflake exception: " +
                           errorResourceBundleManager.getLocalizedMessage(
                                   String.valueOf(vendorCode), params)
                           + ", sqlState:" + sqlState +
                           ", vendorCode:" + vendorCode);

  }

  public SnowflakeSQLException(Throwable ex, String sqlState, int vendorCode)
  {
    super(errorResourceBundleManager.getLocalizedMessage(
            String.valueOf(vendorCode)), sqlState, vendorCode, ex);

    logger.log(Level.FINER, "Snowflake exception: " +
                              errorResourceBundleManager.getLocalizedMessage(
                              String.valueOf(vendorCode)), ex);
  }

  public SnowflakeSQLException(Throwable ex,
                               String sqlState,
                               int vendorCode,
                               Object... params)
  {
    super(errorResourceBundleManager.getLocalizedMessage(
            String.valueOf(vendorCode), params), sqlState, vendorCode, ex);

    logger.log(Level.FINER, "Snowflake exception: " +
                           errorResourceBundleManager.getLocalizedMessage(
                                   String.valueOf(vendorCode), params), ex);
  }

  public String getQueryId()
  {
    return queryId;
  }
}
