/*
 * Copyright (c) 2012-2016 Snowflake Computing Inc. All right reserved.
 */

package net.snowflake.client.loader;

import net.snowflake.client.jdbc.SnowflakeConnectionV1;
import net.snowflake.client.jdbc.SnowflakeFileTransferAgent;
import net.snowflake.client.loader.Loader.ConnectionError;
import java.io.File;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Class responsible for uploading a single data file.
 */
public class FileUploader implements Runnable
{
  private static final Logger LOGGER = Logger.getLogger(
          PutQueue.class.getName());
  private final Thread _thread;
  private final StreamLoader _loader;
  private final String _stage;
  private final File _file;

  public FileUploader(StreamLoader loader, String stage, File file)
  {
    LOGGER.log(Level.FINER, String.format(""));
    _loader = loader;
    _thread = new Thread(this);
    _thread.setName("FileUploaderThread");
    _stage = stage;
    _file = file;
  }

  public synchronized void upload() {
    // throttle up will wait if too many files are uploading
    LOGGER.log(Level.FINER, String.format(""));
    _loader.throttleUp();
     _thread.start();
  }

  @Override
  public void run()
  {

    try
    {

      for (int attempt = 0; attempt <= 6; attempt++)
      {

        if (attempt == 6)
        {
          _loader.abort(new ConnectionError(
                  "File could not be uploaded to remote stage: "
                  + _file.getCanonicalPath()));
          break;
        }

        if (attempt > 0)
        {
          LOGGER.log(Level.INFO,
                   String.format("Will retry PUT after %s seconds",
                                 Math.pow(2, attempt)));
          Thread.sleep(1000 * ((int)Math.pow(2, attempt)));
        }

        // In test mode force fail first file
        if (_loader._testMode)
        {
          if (attempt < 2)
          {
            ((SnowflakeConnectionV1) _loader.getPutConnection())
                .setInjectFileUploadFailure(_file.getName());
          }
          else
          {
            // so that retry now succeeds.
            ((SnowflakeConnectionV1) _loader.getPutConnection())
                .setInjectFileUploadFailure(null);
          }
        }

        // Upload local files to a remote stage

        // No double quote is added _loader.getRemoteStage(), since
        // it is mostly likely to be "~". If not, we may need to double quote
        // them.
        String remoteStage = "@" + _loader.getRemoteStage()
                             + "/" + remoteSeparator(_stage);


        String putStatement = "PUT "
                              + (attempt > 0 ? "/* retry:"+attempt+" */ " : "")
                              + "file://" + _file.getCanonicalPath()
                              + " " + remoteStage
                              + " parallel=4"
                              + " auto_compress=false source_compression=gzip";

        Statement statement = _loader.getPutConnection().createStatement();
        try
        {
          LOGGER.log(Level.FINER, String.format(
                  "Put Statement: %s", putStatement));
          statement.execute(putStatement);
          ResultSet putResult = statement.getResultSet();

          putResult.next();

          String file = localSeparator(
                  putResult.getString(
                          SnowflakeFileTransferAgent.UploadColumns.source.name()));
          String status = putResult.getString(
                  SnowflakeFileTransferAgent.UploadColumns.status.name());
          String message = putResult.getString(
                  SnowflakeFileTransferAgent.UploadColumns.message.name());

          if (status != null && status.equals(
                  SnowflakeFileTransferAgent.ResultStatus.UPLOADED.name()))
          {
            _file.delete();
            break;
            // we are done here.
          }
          else
          {
            // The log level should be WARNING for a single upload failure.
            Level logLevel = 
              message.startsWith("Simulated upload failure") ? 
              Level.INFO : Level.WARNING;
            LOGGER.log(logLevel, String.format(
                    "Failed to upload a file:"
                            + " status=[%s],"
                            + " filename=[%s],"
                            + " message=[%s]",
                    status, file, message));
          }
        }
        catch (Throwable t)
        {
          // The log level for unknown error is set to SEVERE
          LOGGER.log(Level.SEVERE, String.format(
                  "Failed to PUT on attempt: attempt=[%s], "
                  +"Message=[%s]", attempt, t.getMessage()), t.getCause());
        }
      }
    }
    catch (Throwable t)
    {
      LOGGER.log(Level.SEVERE, "PUT exception", t);
      _loader.abort(new Loader.ConnectionError(t.getMessage(), t.getCause()));
    }
    finally
    {
      _loader.throttleDown();
    }
  }

  public void join()
  {
    LOGGER.log(Level.FINER, String.format(""));
    try
    {
      _thread.join(0);
    }
    catch (InterruptedException ex)
    {
      LOGGER.log(Level.SEVERE, null, ex);
    }
  }


  /**
   * convert any back slashes to forward slashes if necessary when converting
   * a local filename to a one suitable for S3
   * 
   * @param fname
   * @return A fname string for S3
   */
  private String remoteSeparator(String fname)
  {
    if (File.separatorChar == '\\')
      return fname.replace("\\", "/");
    else
      return fname;
  }

  /**
   * convert any forward slashes to back slashes if necessary when converting
   * a S3 file name to a local file name
   * @param fname
   * @return A fname string for the local FS (Windows/other Unix like OS)
   */
  private String localSeparator(String fname)
  {
    if (File.separatorChar == '\\')
      return fname.replace("/", "\\");
    else
      return fname;
  }
}

