/*
 * Copyright (c) 2012-2016 Snowflake Computing Inc. All right reserved.
 */

package net.snowflake.client.loader;

import net.snowflake.client.loader.Loader.DataError;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *  This class is responsible for processing a collection of uploaded data files
 *  represented by BufferStage class
 */
public class ProcessQueue implements Runnable
{
  private static final Logger LOGGER = Logger.getLogger(
          ProcessQueue.class.getName());

  private final Thread _thread;

  private final StreamLoader _loader;

  private final boolean _preserveTemp
                        = System.getProperty("snowflake.connector.preserveTemp") != null
                          && System.getProperty(
                  "snowflake.connector.preserveTemp").equalsIgnoreCase("true");

  public ProcessQueue(StreamLoader loader)
  {
    LOGGER.log(Level.FINER, String.format(""));
    
    _loader = loader;
    _thread = new Thread(this);
    _thread.setName("ProcessQueueThread");
    _thread.start();
  }

  @Override
  public void run()
  {

    while (true)
    {

      BufferStage stage = null;

      Connection conn = _loader.getProcessConnection();

      try
      {

        stage = _loader.takeProcess();

        if (stage.getRowCount() == 0)
        {
          // Nothing was written to that stage
          if (stage.isTerminate())
          {
            break;
          }
          else
          {
            continue;
          }
        }

        // Place where the files are.
        // No double quote is added _loader.getRemoteStage(), since
        // it is mostly likely to be "~". If not, we may need to double quote
        // them.
        String remoteStage = "@" + _loader.getRemoteStage() + "/" + stage
                .getRemoteLocation();

        // process uploaded files
        // Loader.abort() and finish() are also synchronized on this
        synchronized (_loader)
        {

          if (_loader.isAborted())
          {

            LOGGER.log(Level.FINER, String.format(
                    "Aborted. RM: %s", remoteStage));
            conn.createStatement().execute("RM " + remoteStage);

            if (stage.isTerminate())
            {
              break;
            }
            else
            {
              continue;
            }
            // Do not do anything to this stage.
            // Everything was rolled back upon abort() call
          }
          // Create a temporary table to hold all uploaded data

          int loaded = 0;
          int parsed = 0;
          int errorCount = 0;
          String lastErrorRow = "";
          String loadStatement;

          // Create temp table to load data (may has a subset of columns)
          LOGGER.log(Level.INFO,
                     String.format("Creating Temporary Table: name=[%s]",
                                   stage.getId()));
          conn.createStatement()
                  .execute("CREATE TEMPORARY TABLE \""
                                + stage.getId() + "\" AS SELECT "
                                + _loader.getColumnsAsString()
                                + " FROM "
                                + _loader.getFullTableName()
                                + " WHERE FALSE");

          // Load data there
          LOGGER.log(Level.INFO,
                     String.format("Copying data in the stage to table:"
                             + " stage=[%s],"
                             + " name=[%s]", remoteStage, stage.getId()));
          ResultSet rs = conn.createStatement()
                  .executeQuery("COPY INTO \""
                                 + stage.getId()
                                 + "\" FROM " + remoteStage
                                + " on_error='continue'"
                                + " file_format=("
                                + "field_optionally_enclosed_by='\"')");

          while(rs.next()) {
            // TODO: hard coded index is not robust
            // Get the number of rows actually loaded
            loaded += rs.getInt(4);
            // Get the number of rows parsed
            parsed += rs.getInt(3);
          }

          int errorRecordCount = parsed - loaded;
          LOGGER.log(Level.INFO,
                     String.format(
                             "errorRecordCount=[%s],"
                                     + " parsed=[%s],"
                                     + " loaded=[%s]",
                             errorRecordCount, parsed, loaded));

          LoadResultListener listener = _loader.getListener();
          listener.addErrorRecordCount(errorRecordCount);

          if (loaded == stage.getRowCount())
          {
            // sucessfully loaded everything
            LOGGER.log(Level.INFO, "COPY command successfully finished");
            listener.addErrorCount(0);
          }
          else
          {
            LOGGER.log(Level.INFO, "Found errors in COPY command");
            if (listener.needErrors())
            {
              ResultSet errorsSet = conn.createStatement()
                      .executeQuery(
                              "COPY INTO \""
                              + stage.getId()
                              + "\" FROM " + remoteStage
                              + " validation_mode='return_all_errors'"
                              + " file_format=("
                              + "field_optionally_enclosed_by='\"')");

              DataError dataError = null;

              while (errorsSet.next())
              {
                errorCount++;
                String rn = errorsSet.getString(
                        LoadingError.ErrorProperty.ROW_NUMBER
                        .name());
                if (rn != null && !lastErrorRow.equals(rn))
                {
                  // de-duping records with multiple errors
                  lastErrorRow = rn;
                }
                LoadingError loadError = new LoadingError(errorsSet, stage,
                                                   _loader);

                listener.addError(loadError);
                if (dataError == null)
                {
                  dataError = loadError.getException();
                }
              }
              LOGGER.log(Level.INFO,
                         String.format("errorCount: %s", errorCount));

              listener.addErrorCount(errorCount);
              if (listener.throwOnError())
              {
                _loader.abort(dataError);

                LOGGER.log(Level.FINER, String.format(
                        "RM: %s", remoteStage));
                conn.createStatement().execute("RM " + remoteStage);

                if (stage.isTerminate())
                {
                  break;
                }
                else
                {
                  continue;
                }
              }
            }
          }

          stage.setState(BufferStage.State.VALIDATED);

          // Generate set and values statement
          StringBuilder setStatement = null;
          StringBuilder valueStatement = null;
          if (stage.getOp() != Operation.INSERT
              && stage.getOp() != Operation.DELETE)
          {

            setStatement = new StringBuilder(" ");
            valueStatement = new StringBuilder("(");

            for (int c = 0; c < _loader.getColumns().size(); ++c)
            {
              String column = _loader.getColumns().get(c);
              if (c > 0)
              {
                setStatement.append(", ");
                valueStatement.append(" , ");
              }
              setStatement.append("T.\"")
                      .append(column)
                      .append("\"=")
                      .append("S.\"")
                      .append(column)
                      .append("\"");
              valueStatement.append("S.\"").append(column).append("\"");
            }
            valueStatement.append(")");
          }


          // generate statement for processing
          switch (stage.getOp())
          {
            case INSERT:
            {
              loadStatement = "INSERT INTO "
                              + _loader.getFullTableName()
                              + "(" + _loader.getColumnsAsString() + ")"
                              + " SELECT * FROM \"" + stage.getId() + "\"";
              break;
            }
            case DELETE:
            {
              loadStatement = "DELETE FROM " + _loader.getFullTableName()
                              + " T USING \""
                              + stage.getId() + "\" AS S WHERE "
                              + getOn(_loader.getKeys(), "T", "S");
              break;
            }
            case MODIFY:
            {
              loadStatement = "MERGE INTO " + _loader.getFullTableName()
                              + " T USING \""
                              + stage.getId() + "\" AS S ON "
                              + getOn(_loader.getKeys(), "T", "S")
                              + " WHEN MATCHED THEN UPDATE SET " + setStatement;
              break;
            }
            case UPSERT:
            {

              loadStatement = "MERGE INTO " + _loader.getFullTableName()
                              + " T USING \""
                              + stage.getId() + "\" AS S ON "
                              + getOn(_loader.getKeys(), "T", "S")
                              + " WHEN MATCHED THEN UPDATE SET " + setStatement
                              + " WHEN NOT MATCHED THEN INSERT("
                              + _loader.getColumnsAsString() + ") VALUES"
                              + valueStatement;
              break;
            }
            default:
              loadStatement = "";
          }

          LOGGER.log(Level.FINER, String.format(
                  "Load Statement: %s", loadStatement));
          Statement s = conn.createStatement();
          s.execute(loadStatement);
          ResultSet prs = s.getResultSet();
          prs.next();

          stage.setState(BufferStage.State.PROCESSED);

          switch (stage.getOp())
          {
            case INSERT:
            {
              _loader.getListener().addProcessedRecordCount(
                      stage.getOp(), stage.getRowCount());

              _loader.getListener().addOperationRecordCount(
                      Operation.INSERT, prs.getInt(1));
              break;
            }
            case DELETE:
            {
              // the number of successful DELETE is the number 
              // of processed rows and not the number of given
              // rows.
              _loader.getListener().addProcessedRecordCount(
                      stage.getOp(), prs.getInt(1));

              _loader.getListener().addOperationRecordCount(
                      Operation.DELETE, prs.getInt(1));
              break;
            }
            case MODIFY:
            {
              // the number of successful UPDATE
              _loader.getListener().addProcessedRecordCount(
                      stage.getOp(), prs.getInt(1));

              _loader.getListener().addOperationRecordCount(
                      Operation.MODIFY, prs.getInt(1));
              break;
            }
            case UPSERT:
            {
              _loader.getListener().addProcessedRecordCount(
                      stage.getOp(), stage.getRowCount());

              _loader.getListener().addOperationRecordCount(
                      Operation.UPSERT, prs.getInt(1) + prs.getInt(2));
              break;
            }

          }

          if (!_preserveTemp)
          {
            conn.createStatement().execute(
                    "DROP TABLE \"" + stage.getId() + "\"");
          }

          conn.createStatement().execute("RM " + remoteStage);

          if (stage.isTerminate())
          {
            break;
          }

        }
      }
      catch (InterruptedException ex)
      {
        LOGGER.log(Level.SEVERE, "Interrupted", ex);
        break;
      }
      catch (SQLException ex)
      {
        LOGGER.log(Level.SEVERE, ex.getMessage(), ex.getCause());
        _loader.abort(
                new Loader.ConnectionError(ex.getMessage(),ex.getCause()));

        if (stage == null || stage.isTerminate())
        {
          break;
        }
      }
      catch (Exception ex)
      {
        LOGGER.log(Level.SEVERE, null, ex);

        _loader.abort(new Loader.ConnectionError(
                ex.getMessage(),ex.getCause()));

        if (stage == null || stage.isTerminate())
        {
          break;
        }
      }
      finally
      {
        // nop
      }
    }
  }

  private String getOn(List<String> keys, String L, String R)
  {
    // L and R don't need to be quoted.
    StringBuilder sb = keys.size() > 1 ? 
                       new StringBuilder(64) : new StringBuilder();
    for (int i = 0; i < keys.size(); i++)
    {
      if (i > 0)
      {
        sb.append("AND ");
      }
      sb.append(L);
      sb.append(".\"");
      sb.append(keys.get(i));
      sb.append("\" = ");
      sb.append(R);
      sb.append(".\"");
      sb.append(keys.get(i));
      sb.append("\" ");
    }
    return sb.toString();
  }

  public void join()
  {
    LOGGER.log(Level.FINER, String.format(""));
    try
    {
      _thread.join(0);
    }
    catch (InterruptedException ex)
    {
      LOGGER.log(Level.SEVERE, null, ex);
    }
  }

}
