/*
 * Copyright (c) 2012-2016 Snowflake Computing Inc. All right reserved.
 */

package net.snowflake.client.loader;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *  Queue that sequentially finalizes BufferStage uploads and schedules them for 
 *  processing in ProcessQueue.
 */
public class PutQueue implements Runnable
{
  private static final Logger LOGGER = Logger.getLogger(
          PutQueue.class.getName());

  private final Thread _thread;

  private final StreamLoader _loader;

  public PutQueue(StreamLoader loader)
  {
    LOGGER.log(Level.FINER, "");
    _loader = loader;
    _thread = new Thread(this);
    _thread.setName("PutQueueThread");
    _thread.start();
  }

  @Override
  public void run()
  {

    while (true)
    {

      BufferStage stage = null;

      try
      {

        stage = _loader.takePut();

        if (stage.getRowCount() == 0)
        {
          // Nothing was written to that stage
          if (stage.isTerminate())
          {
            _loader.queueProcess(stage);
            stage.completeUploading();
            break;
          }
          else
          {
            continue;
          }
        }

        // Uploads the stage
        stage.completeUploading();
        
        // Schedules it for processing
        _loader.queueProcess(stage);

        if (stage.isTerminate())
        {
          break;
        }

      }
      catch (InterruptedException | IOException ex)
      {
        LOGGER.log(Level.SEVERE, null, ex);
        break;
      }
      finally
      {

      }
    }
  }

  public void join()
  {
    try
    {
      _thread.join(0);
    }
    catch (InterruptedException ex)
    {
      LOGGER.log(Level.SEVERE, null, ex);
    }
  }

}
