/* ==================================================================
 * BasicFilterResults.java - 7/02/2020 9:16:50 am
 * 
 * Copyright 2020 SolarNetwork.net Dev Team
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of 
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
 * 02111-1307 USA
 * ==================================================================
 */

package net.solarnetwork.dao;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Set;
import java.util.stream.StreamSupport;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import net.solarnetwork.domain.Identity;

/**
 * Basic implementation of {@link FilterResults}.
 * 
 * @author matt
 * @version 1.2
 * @since 1.59
 */
@JsonPropertyOrder({ "totalResults", "startingOffset", "returnedResultCount", "results" })
public class BasicFilterResults<M extends Identity<K>, K> implements FilterResults<M, K> {

	private final Iterable<M> results;
	private final Long totalResults;
	private final int startingOffset;
	private final int returnedResultCount;

	/**
	 * Constructor.
	 * 
	 * @param results
	 *        the results iterable
	 * @param totalResults
	 *        the total available results, or {@literal null}
	 * @param startingOffset
	 *        the starting offset
	 * @param returnedResultCount
	 *        the count of objects in {@code results}
	 */
	public BasicFilterResults(Iterable<M> results, Long totalResults, int startingOffset,
			int returnedResultCount) {
		super();
		this.results = results;
		this.totalResults = totalResults;
		this.startingOffset = startingOffset;
		this.returnedResultCount = returnedResultCount;
	}

	/**
	 * Constructor.
	 * 
	 * <p>
	 * This total results count will be set to {@literal null}, the starting
	 * offset to {@literal 0}, and the returned result count will be derived
	 * from the number of items in {@code results}.
	 * </p>
	 * 
	 * @param results
	 *        the results iterable
	 */
	public BasicFilterResults(Iterable<M> results) {
		this(results, null, 0, iterableCount(results));
	}

	/**
	 * Create a {@link FilterResults} instance.
	 * 
	 * @param <M>
	 *        the result type
	 * @param <K>
	 *        the result key type
	 * @param data
	 *        the result data
	 * @param criteria
	 *        the criteria used to produce the results
	 * @param totalResults
	 *        the total result count, if available
	 * @param returnedResults
	 *        the returned results count
	 * @return the results instance
	 * @since 1.1
	 */
	public static <M extends Identity<K>, K> FilterResults<M, K> filterResults(Iterable<M> data,
			PaginationCriteria criteria, Long totalResults, int returnedResults) {
		int offset = 0;
		if ( criteria != null && criteria.getMax() != null ) {
			offset = criteria.getOffset() != null ? criteria.getOffset() : 0;
		}
		return new BasicFilterResults<>(data, totalResults, offset, returnedResults);
	}

	private static int iterableCount(Iterable<?> iterable) {
		if ( iterable instanceof Collection<?> ) {
			return ((Collection<?>) iterable).size();
		}
		return (int) StreamSupport.stream(iterable.spliterator(), false).count();
	}

	@Override
	public Iterator<M> iterator() {
		if ( results == null ) {
			Set<M> emptyResult = Collections.emptySet();
			return emptyResult.iterator();
		}
		return results.iterator();
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("BasicFilterResults{");
		if ( totalResults != null ) {
			builder.append("totalCount=");
			builder.append(totalResults);
			builder.append(", ");
		}
		builder.append("offset=");
		builder.append(startingOffset);
		builder.append(", count=");
		builder.append(returnedResultCount);
		builder.append("}");
		return builder.toString();
	}

	@Override
	public Iterable<M> getResults() {
		return results;
	}

	@Override
	public Long getTotalResults() {
		return totalResults;
	}

	@Override
	public int getStartingOffset() {
		return startingOffset;
	}

	@Override
	public int getReturnedResultCount() {
		return returnedResultCount;
	}

}
