/* ==================================================================
 * BasicObjectDatumStreamMetadata.java - 5/11/2020 4:03:50 pm
 * 
 * Copyright 2020 SolarNetwork.net Dev Team
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of 
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
 * 02111-1307 USA
 * ==================================================================
 */

package net.solarnetwork.domain.datum;

import java.util.Arrays;
import java.util.Objects;
import java.util.UUID;
import net.solarnetwork.domain.BasicLocation;
import net.solarnetwork.domain.Location;

/**
 * Basic implementation of {@link ObjectDatumStreamMetadata}.
 * 
 * @author matt
 * @version 1.0
 * @since 1.72
 */
public class BasicObjectDatumStreamMetadata extends BasicDatumStreamMetadata
		implements ObjectDatumStreamMetadata {

	private static final long serialVersionUID = -4093896601567626604L;

	/** The kind. */
	private final ObjectDatumKind kind;

	/** The object ID. */
	private final Long objectId;

	/** The source ID. */
	private final String sourceId;

	/** The location. */
	private final BasicLocation location;

	/** The JSON metadata. */
	private final String metaJson;

	/**
	 * Create a new metadata instance with no property names.
	 * 
	 * @param streamId
	 *        the stream ID
	 * @param timeZoneId
	 *        the time zone ID
	 * @param kind
	 *        the object kind
	 * @param objectId
	 *        the object ID
	 * @param sourceId
	 *        the source ID
	 * @return the new instance
	 */
	public static BasicObjectDatumStreamMetadata emptyMeta(UUID streamId, String timeZoneId,
			ObjectDatumKind kind, Long objectId, String sourceId) {
		return new BasicObjectDatumStreamMetadata(streamId, timeZoneId, kind, objectId, sourceId, null,
				null, null, null);
	}

	/**
	 * Constructor.
	 * 
	 * <p>
	 * All arguments except {@code streamId}, {@code objectId}, and
	 * {@code sourceId} are allowed to be {@literal null}. If any array is
	 * empty, it will be treated as if it were {@literal null}.
	 * </p>
	 * 
	 * @param streamId
	 *        the stream ID
	 * @param timeZoneId
	 *        the time zone ID
	 * @param kind
	 *        the object kind
	 * @param objectId
	 *        the object ID
	 * @param sourceId
	 *        the source ID
	 * @param instantaneousProperties
	 *        the instantaneous property names
	 * @param accumulatingProperties
	 *        the accumulating property names
	 * @param statusProperties
	 *        the status property names
	 * @throws IllegalArgumentException
	 *         if {@code streamId} or {@code objectId} or {@code sourceId} is
	 *         {@literal null}
	 */
	public BasicObjectDatumStreamMetadata(UUID streamId, String timeZoneId, ObjectDatumKind kind,
			Long objectId, String sourceId, String[] instantaneousProperties,
			String[] accumulatingProperties, String[] statusProperties) {
		this(streamId, timeZoneId, kind, objectId, sourceId, null, instantaneousProperties,
				accumulatingProperties, statusProperties, null);
	}

	/**
	 * Constructor.
	 * 
	 * <p>
	 * All arguments except {@code streamId}, {@code objectId}, and
	 * {@code sourceId} are allowed to be {@literal null}. If any array is
	 * empty, it will be treated as if it were {@literal null}.
	 * </p>
	 * 
	 * @param streamId
	 *        the stream ID
	 * @param timeZoneId
	 *        the time zone ID
	 * @param kind
	 *        the object kind
	 * @param objectId
	 *        the object ID
	 * @param sourceId
	 *        the source ID
	 * @param instantaneousProperties
	 *        the instantaneous property names
	 * @param accumulatingProperties
	 *        the accumulating property names
	 * @param statusProperties
	 *        the status property names
	 * @param metaJson
	 *        the JSON metadata
	 * @throws IllegalArgumentException
	 *         if {@code streamId} or {@code objectId} or {@code sourceId} is
	 *         {@literal null}
	 */
	public BasicObjectDatumStreamMetadata(UUID streamId, String timeZoneId, ObjectDatumKind kind,
			Long objectId, String sourceId, String[] instantaneousProperties,
			String[] accumulatingProperties, String[] statusProperties, String metaJson) {
		this(streamId, timeZoneId, kind, objectId, sourceId, null, instantaneousProperties,
				accumulatingProperties, statusProperties, metaJson);
	}

	/**
	 * Constructor.
	 * 
	 * <p>
	 * All arguments except {@code streamId}, {@code objectId}, and
	 * {@code sourceId} are allowed to be {@literal null}. If any array is
	 * empty, it will be treated as if it were {@literal null}.
	 * </p>
	 * 
	 * @param streamId
	 *        the stream ID
	 * @param timeZoneId
	 *        the time zone ID
	 * @param kind
	 *        the object kind
	 * @param objectId
	 *        the object ID
	 * @param sourceId
	 *        the source ID
	 * @param location
	 *        the location
	 * @param instantaneousProperties
	 *        the instantaneous property names
	 * @param accumulatingProperties
	 *        the accumulating property names
	 * @param statusProperties
	 *        the status property names
	 * @param metaJson
	 *        the JSON metadata
	 * @throws IllegalArgumentException
	 *         if {@code streamId} or {@code objectId} or {@code sourceId} is
	 *         {@literal null}
	 */
	public BasicObjectDatumStreamMetadata(UUID streamId, String timeZoneId, ObjectDatumKind kind,
			Long objectId, String sourceId, Location location, String[] instantaneousProperties,
			String[] accumulatingProperties, String[] statusProperties, String metaJson) {
		super(streamId, timeZoneId, instantaneousProperties, accumulatingProperties, statusProperties);
		if ( kind == null ) {
			throw new IllegalArgumentException("The kind argument must not be null.");
		}
		this.kind = kind;
		if ( objectId == null ) {
			throw new IllegalArgumentException("The objectId argument must not be null.");
		}
		this.objectId = objectId;
		if ( sourceId == null ) {
			throw new IllegalArgumentException("The sourceId argument must not be null.");
		}
		this.sourceId = sourceId;
		this.location = BasicLocation.locationValue(location);
		this.metaJson = metaJson;
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("BasicObjectDatumStreamMetadata{");
		builder.append("streamId=");
		builder.append(getStreamId());
		builder.append(", kind=");
		builder.append(kind);
		builder.append(", objectId=");
		builder.append(objectId);
		builder.append(", sourceId=");
		builder.append(sourceId);
		if ( getPropertyNames() != null ) {
			builder.append(", propertyNames=");
			builder.append(Arrays.toString(getPropertyNames()));
		}
		builder.append("}");
		return builder.toString();
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + Objects.hash(kind, location, metaJson, objectId, sourceId);
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if ( this == obj ) {
			return true;
		}
		if ( !super.equals(obj) ) {
			return false;
		}
		if ( !(obj instanceof BasicObjectDatumStreamMetadata) ) {
			return false;
		}
		BasicObjectDatumStreamMetadata other = (BasicObjectDatumStreamMetadata) obj;
		return kind == other.kind && Objects.equals(location, other.location)
				&& Objects.equals(metaJson, other.metaJson) && Objects.equals(objectId, other.objectId)
				&& Objects.equals(sourceId, other.sourceId);
	}

	@Override
	public Long getObjectId() {
		return objectId;
	}

	@Override
	public String getSourceId() {
		return sourceId;
	}

	@Override
	public String getMetaJson() {
		return metaJson;
	}

	@Override
	public ObjectDatumKind getKind() {
		return kind;
	}

	@Override
	public BasicLocation getLocation() {
		return location;
	}

}
