/* ==================================================================
 * AppEvent.java - 29/05/2020 3:35:12 pm
 * 
 * Copyright 2020 SolarNetwork.net Dev Team
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of 
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
 * 02111-1307 USA
 * ==================================================================
 */

package net.solarnetwork.event;

import java.time.Instant;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

/**
 * An application event.
 * 
 * <p>
 * Application events are generated by SolarNetwork applications themselves.
 * This has been modeled after <code>org.osgi.service.event.Event</code> to help
 * working in non-OSGi environments.
 * </p>
 * 
 * @author matt
 * @version 1.0
 * @since 2.0
 */
public interface AppEvent {

	/**
	 * Get the event topic.
	 * 
	 * <p>
	 * By convention topics are path-like structures like
	 * {@literal datum/agg/calculated}.
	 * </p>
	 * 
	 * @return the event topic, never {@literal null}
	 */
	String getTopic();

	/**
	 * Get the date the event was created.
	 * 
	 * @return the creation date, never {@literal null}
	 */
	Instant getCreated();

	/**
	 * Get the properties of the event.
	 * 
	 * <p>
	 * The available properties is topic-specific.
	 * </p>
	 * 
	 * @return the event properties, never {@literal null}
	 */
	Map<String, ?> getEventProperties();

	/**
	 * Test if a event property is available.
	 * 
	 * @param key
	 *        the event property name to test
	 * @return {@literal true} if a non-{@literal null} property exists for
	 *         {@code key}
	 */
	default boolean containsProperty(String key) {
		Map<String, ?> props = getEventProperties();
		return (props != null ? props.containsKey(key) : false);
	}

	/**
	 * Get all available property names as an array.
	 * 
	 * @return the available property names, never {@literal null}
	 */
	default String[] getPropertyNames() {
		Map<String, ?> props = getEventProperties();
		Set<String> keys = (props != null ? props.keySet() : Collections.emptySet());
		return keys.toArray(new String[keys.size()]);
	}

	/**
	 * Get an event property by key.
	 * 
	 * @param key
	 *        the event property name to get
	 * @return the associated property value, or {@literal null} if not
	 *         available
	 */
	default Object getProperty(String key) {
		Map<String, ?> props = getEventProperties();
		return (props != null ? props.get(key) : null);
	}

	/**
	 * Get an event property of a specific type.
	 * 
	 * @param <T>
	 *        the type of property value expected
	 * @param key
	 *        the property key to get
	 * @param type
	 *        the property type
	 * @return the associated property value, or {@literal null} if not
	 *         available or not of type {@code T}
	 */
	default <T> T getProperty(String key, Class<T> type) {
		Object o = getProperty(key);
		return (o != null && type.isAssignableFrom(o.getClass()) ? type.cast(o) : null);
	}

}
