/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the  "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.htmlunit.xpath.axes;

import net.sourceforge.htmlunit.xpath.XPathContext;
import net.sourceforge.htmlunit.xpath.objects.XObject;
import net.sourceforge.htmlunit.xpath.patterns.NodeTest;
import net.sourceforge.htmlunit.xpath.patterns.StepPattern;
import net.sourceforge.htmlunit.xpath.xml.dtm.DTM;
import net.sourceforge.htmlunit.xpath.xml.dtm.DTMAxisTraverser;
import net.sourceforge.htmlunit.xpath.xml.dtm.DTMIterator;

/**
 * This class treats a <a href="http://www.w3.org/TR/xpath#location-paths">LocationPath</a> as a
 * filtered iteration over the tree, evaluating each node in a super axis traversal against the
 * LocationPath interpreted as a match pattern. This class is useful to find nodes in document order
 * that are complex paths whose steps probably criss-cross each other.
 */
public class MatchPatternIterator extends LocPathIterator {

  /** This is the select pattern, translated into a match pattern. */
  protected StepPattern m_pattern;

  /** The traversal axis from where the nodes will be filtered. */
  protected int m_superAxis = -1;

  /** The DTM inner traversal class, that corresponds to the super axis. */
  protected DTMAxisTraverser m_traverser;

  /** DEBUG flag for diagnostic dumps. */
  private static final boolean DEBUG = false;

  // protected int m_nsElemBase = DTM.NULL;

  /** {@inheritDoc} */
  @Override
  public void setRoot(int context, Object environment) {
    super.setRoot(context, environment);
    m_traverser = m_cdtm.getAxisTraverser(m_superAxis);
  }

  /** {@inheritDoc} */
  @Override
  public void detach() {
    if (m_allowDetach) {
      m_traverser = null;

      // Always call the superclass detach last!
      super.detach();
    }
  }

  /**
   * Get the next node via getNextXXX. Bottlenecked for derived class override.
   *
   * @return The next node on the axis, or DTM.NULL.
   */
  protected int getNextNode() {
    m_lastFetched =
        (DTM.NULL == m_lastFetched)
            ? m_traverser.first(m_context)
            : m_traverser.next(m_context, m_lastFetched);
    return m_lastFetched;
  }

  /** {@inheritDoc} */
  @Override
  public int nextNode() {
    if (m_foundLast) return DTM.NULL;

    int next;

    try {
      if (DEBUG) System.out.println("m_pattern" + m_pattern.toString());

      do {
        next = getNextNode();

        if (DTM.NULL != next) {
          if (DTMIterator.FILTER_ACCEPT == acceptNode(next, m_execContext)) break;
          else continue;
        } else break;
      } while (next != DTM.NULL);

      if (DTM.NULL != next) {
        if (DEBUG) {
          System.out.println("next: " + next);
          System.out.println("name: " + m_cdtm.getNodeName(next));
        }
        incrementCurrentPos();

        return next;
      } else {
        m_foundLast = true;

        return DTM.NULL;
      }
    } finally {
    }
  }

  /**
   * Test whether a specified node is visible in the logical view of a TreeWalker or NodeIterator.
   * This function will be called by the implementation of TreeWalker and NodeIterator; it is not
   * intended to be called directly from user code.
   *
   * @param n The node to check to see if it passes the filter or not.
   * @return a constant to determine whether the node is accepted, rejected, or skipped, as defined
   *     above .
   */
  public short acceptNode(int n, XPathContext xctxt) {

    try {
      xctxt.pushCurrentNode(n);
      xctxt.pushIteratorRoot(m_context);
      XObject score = m_pattern.execute(xctxt);
      return (score == NodeTest.SCORE_NONE) ? DTMIterator.FILTER_SKIP : DTMIterator.FILTER_ACCEPT;
    } catch (javax.xml.transform.TransformerException se) {

      // TODO: Fix this.
      throw new RuntimeException(se.getMessage());
    } finally {
      xctxt.popCurrentNode();
      xctxt.popIteratorRoot();
    }
  }
}
