package it.openutils.testing.junit.dwr;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import junit.framework.TestCase;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import com.meterware.httpunit.GetMethodWebRequest;
import com.meterware.httpunit.WebResponse;
import com.meterware.servletunit.ServletRunner;
import com.meterware.servletunit.ServletUnitClient;


public class DwrTestCase extends TestCase
{

    private static Logger log = LoggerFactory.getLogger(DwrTestCase.class);

    public static final String SERVLET_UNIT_CLIENT = "SERVLET_UNIT_CLIENT";

    private Context context;

    private Scriptable scope;

    private ServletUnitClient servletUnitClient;

    private ServletRunner servletRunner;

    private String webXmlLocation;

    private String contextPath = "/webapp";

    private List<String> jsLibraries = new ArrayList<String>();

    protected void setAbsoluteWebXmlLocation(String absoluteWebXmlLocation)
    {
        this.webXmlLocation = absoluteWebXmlLocation;
    }

    protected void setContextPath(String contextPath)
    {
        this.contextPath = contextPath;
        if (this.contextPath != null)
        {
            this.contextPath = this.contextPath.replace('\\', '/');
            if (!this.contextPath.startsWith("/"))
            {
                this.contextPath = "/" + this.contextPath;
            }
        }
    }

    public void addJsLibrary(String library)
    {
        jsLibraries.add(library);
    }

    @Override
    protected void setUp() throws Exception
    {
        // run httpunit servlet engine
        servletRunner = new ServletRunner(new File(webXmlLocation), contextPath);
        servletUnitClient = servletRunner.newClient();

        // init scripting context
        context = Context.enter();
        context.setLanguageVersion(Context.VERSION_1_6);

        // store servletUnitClient in context
        context.putThreadLocal(DwrTestCase.SERVLET_UNIT_CLIENT, servletUnitClient);

        // load js support for window and document
        String jsWebEnvironment = "";
        InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream("webEnv.js");
        byte[] buffer = new byte[1024];
        int read = 0;
        int length = 1024;
        while ((read = is.read(buffer, 0, length)) > 0)
        {
            jsWebEnvironment += new String(buffer, 0, read);
        }
        is.close();

        //
        scope = context.initStandardObjects();
        ScriptableObject.defineClass(scope, XMLHttpRequest.class);
        // ScriptableObject.defineClass(scope, JsJUnitAssert.class);
        context.evaluateString(scope, jsWebEnvironment, "webEnv.js", -1, null);
        scope.put("Test", scope, this);
        scope.put("Log", scope, log);

        // add javascript
        for (String library : jsLibraries)
        {
            if (loadLibraryFromLocalServer(library))
            {
                log.info("Loaded library from local server: " + library);
            }
            else if (loadLibraryFromRemoteServer(library))
            {
                log.info("Loaded library from remote server: " + library);
            }
            else if (loadLibraryFromClasspath(library))
            {
                log.info("Loaded library from classpath: " + library);
            }
            else if (loadLibraryFromFileSystem(library))
            {
                log.info("Loaded library from filesystem: " + library);
            }
            else
            {
                log.error("Can't load library " + library);
                throw new FileNotFoundException(library);
            }
        }

        super.setUp();
    }

    private boolean loadLibraryFromLocalServer(String library) throws MalformedURLException, SAXException, IOException,
        URISyntaxException
    {
        if (!library.startsWith("http://localhost/"))
        {
            return false;
        }

        WebResponse response = servletUnitClient.getResponse(new GetMethodWebRequest(library));
        if (response == null)
        {
            return false;
        }
        URI uri = new URI(library);
        context.evaluateString(scope, response.getText(), uri.getPath(), -1, null);

        return true;
    }

    private boolean loadLibraryFromRemoteServer(String library) throws MalformedURLException, SAXException,
        IOException, URISyntaxException
    {
        if (!library.startsWith("http://"))
        {
            return false;
        }

        URL url = new URL(library);

        // connect to library
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setRequestMethod("GET");
        connection.setDoInput(true);
        connection.connect();

        // get response
        InputStream is = connection.getInputStream();
        StringBuffer sb = new StringBuffer();

        int i;
        while ((i = is.read()) != -1)
        {
            sb.append((char) i);
        }
        is.close();
        connection.disconnect();

        // put response in context
        context.evaluateString(scope, sb.toString(), url.getPath(), -1, null);

        return true;
    }

    private boolean loadLibraryFromClasspath(String library) throws IOException
    {
        String js = loadFileFromClassPath(library);

        if (js == null)
        {
            return false;
        }

        context.evaluateString(scope, js, library, -1, null);

        return true;
    }

    private String loadFileFromClassPath(String file) throws IOException
    {
        InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(file);
        if (is == null)
        {
            return null;
        }
        byte[] buffer = new byte[1024];
        int read = 0;
        String js = "";
        while ((read = is.read(buffer, 0, buffer.length)) > 0)
        {
            js += new String(buffer, 0, read);
        }
        is.close();

        return js;
    }

    private boolean loadLibraryFromFileSystem(String library) throws FileNotFoundException, IOException
    {
        String js = loadFileFromFileSystem(library);

        if (js == null)
        {
            return false;
        }

        context.evaluateString(scope, js, library, -1, null);

        return true;
    }

    private String loadFileFromFileSystem(String file) throws FileNotFoundException, IOException
    {
        File f = new File(file);
        if (!f.exists())
        {
            return null;
        }

        InputStream is = new FileInputStream(f);
        if (is == null)
        {
            return null;
        }
        byte[] buffer = new byte[1024];
        int read = 0;
        String js = "";
        while ((read = is.read(buffer, 0, buffer.length)) > 0)
        {
            js += new String(buffer, 0, read);
        }
        is.close();

        return js;
    }

    @Override
    protected void tearDown() throws Exception
    {
        scope = null;
        Context.exit();

        servletRunner.shutDown();

        super.tearDown();
    }

    public void runScript(String script)
    {
        context.evaluateString(scope, script, "inline", -1, null);
    }

    public void runScriptFile(String scriptFile) throws FileNotFoundException, IOException
    {
        String js = loadFileFromClassPath(scriptFile);
        if (js == null)
        {
            js = loadFileFromFileSystem(scriptFile);
            if (js == null)
            {
                throw new FileNotFoundException(scriptFile);
            }
        }

        context.evaluateString(scope, js, scriptFile, -1, null);
    }

}
