// THIS FILE HAS BEEN GENERATED BY A PREPROCESSOR.
/* +=======================================================================
 * |
 * |      PlantUML : a free UML diagram generator
 * |
 * +=======================================================================
 *
 * (C) Copyright 2009-2024, Arnaud Roques
 *
 * Project Info:  https://plantuml.com
 *
 * If you like this project or if you find it useful, you can support us at:
 *
 * https://plantuml.com/patreon (only 1$ per month!)
 * https://plantuml.com/liberapay (only 1€ per month!)
 * https://plantuml.com/paypal
 *
 *
 * PlantUML is free software; you can redistribute it and/or modify it
 * under the terms of the MIT License.
 *
 * See http://opensource.org/licenses/MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * PlantUML can occasionally display sponsored or advertising messages. Those
 * messages are usually generated on welcome or error images and never on
 * functional diagrams.
 * See https://plantuml.com/professional if you want to remove them
 *
 * Images (whatever their format : PNG, SVG, EPS...) generated by running PlantUML
 * are owned by the author of their corresponding sources code (that is, their
 * textual description in PlantUML language). Those images are not covered by
 * this MIT license.
 *
 * The generated images can then be used without any reference to the MIT license.
 * It is not even necessary to stipulate that they have been generated with PlantUML,
 * although this will be appreciated by the PlantUML team.
 *
 * There is an exception : if the textual description in PlantUML language is also covered
 * by any license, then the generated images are logically covered
 * by the very same license.
 *
 * This is the IGY distribution (Install GraphViz by Yourself).
 * You have to install GraphViz and to setup the GRAPHVIZ_DOT environment variable
 * (see https://plantuml.com/graphviz-dot )
 *
 * Icons provided by OpenIconic :  https://useiconic.com/open
 * Archimate sprites provided by Archi :  http://www.archimatetool.com
 * Stdlib AWS provided by https://github.com/milo-minderbinder/AWS-PlantUML
 * Stdlib Icons provided https://github.com/tupadr3/plantuml-icon-font-sprites
 * ASCIIMathML (c) Peter Jipsen http://www.chapman.edu/~jipsen
 * ASCIIMathML (c) David Lippman http://www.pierce.ctc.edu/dlippman
 * CafeUndZopfli ported by Eugene Klyuchnikov https://github.com/eustas/CafeUndZopfli
 * Brotli (c) by the Brotli Authors https://github.com/google/brotli
 * Themes (c) by Brett Schwarz https://github.com/bschwarz/puml-themes
 * Twemoji (c) by Twitter at https://twemoji.twitter.com/
 *
 */
package net.sourceforge.plantuml.quantization;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

/**
 * Implements median cut quantization.
 *
 * <p>
 * The algorithm works as follows:
 *
 * <ul>
 * <li>Begin with one cluster containing all the original colors.</li>
 * <li>Find the cluster containing the greatest spread along a single color
 * component (red, green or blue).</li>
 * <li>Find the median of that color component among colors in the cluster.</li>
 * <li>Split the cluster into two halves, using that median as a threshold.</li>
 * <li>Repeat this process until the desired number of clusters is reached.</li>
 * </ul>
 */
public final class MedianCutQuantizer implements ColorQuantizer {
	public static final MedianCutQuantizer INSTANCE = new MedianCutQuantizer();

	private MedianCutQuantizer() {
	}

	@Override
	public Set<QColor> quantize(Multiset<QColor> originalColors, int maxColorCount) {
		TreeSet<Cluster> clusters = new TreeSet<>(new ClusterSpreadComparator());
		clusters.add(new Cluster(originalColors));

		while (clusters.size() < maxColorCount) {
			Cluster clusterWithLargestSpread = clusters.pollFirst();
			clusters.addAll(clusterWithLargestSpread.split());
		}

		Set<QColor> clusterCentroids = new HashSet<>();
		for (Cluster cluster : clusters) {
			clusterCentroids.add(QColor.getCentroid(cluster.colors));
		}
		return clusterCentroids;
	}

	private static final class Cluster {
		final Multiset<QColor> colors;
		double largestSpread;
		int componentWithLargestSpread;

		Cluster(Multiset<QColor> colors) {
			this.colors = colors;
			this.largestSpread = -1;
			for (int component = 0; component < 3; ++component) {
				double componentSpread = getComponentSpread(component);
				if (componentSpread > largestSpread) {
					largestSpread = componentSpread;
					componentWithLargestSpread = component;
				}
			}
		}

		double getComponentSpread(int component) {
			double min = Double.POSITIVE_INFINITY;
			double max = Double.NEGATIVE_INFINITY;
			for (QColor color : colors) {
				min = Math.min(min, color.getComponent(component));
				max = Math.max(max, color.getComponent(component));
			}
			return max - min;
		}

		Collection<Cluster> split() {
			List<QColor> orderedColors = new ArrayList<>(colors);
			Collections.sort(orderedColors, new ColorComponentComparator(componentWithLargestSpread));
			int medianIndex = orderedColors.size() / 2;
			return Arrays.asList(new Cluster(new HashMultiset<>(orderedColors.subList(0, medianIndex))),
					new Cluster(new HashMultiset<>(orderedColors.subList(medianIndex, orderedColors.size()))));
		}
	}

	/**
	 * Orders clusters according to their maximum spread, in descending order.
	 */
	static final class ClusterSpreadComparator implements Comparator<Cluster> {
		@Override
		public int compare(Cluster a, Cluster b) {
			double spreadDifference = b.largestSpread - a.largestSpread;
			if (spreadDifference == 0) {
				return ArbitraryComparator.INSTANCE.compare(a, b);
			}
			return (int) Math.signum(spreadDifference);
		}
	}

	/**
	 * Orders colors according to the value of one particular component, in
	 * ascending order.
	 */
	static final class ColorComponentComparator implements Comparator<QColor> {
		final int component;

		ColorComponentComparator(int component) {
			this.component = component;
		}

		@Override
		public int compare(QColor a, QColor b) {
			double componentDifference = a.getComponent(component) - b.getComponent(component);
			if (componentDifference == 0) {
				return ArbitraryComparator.INSTANCE.compare(a, b);
			}
			return (int) Math.signum(componentDifference);
		}
	}
}
